document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const gridNameTitle = document.getElementById('grid-name-title');
    const projectNameSubtitle = document.getElementById('project-name-subtitle');
    const backToGridsBtn = document.getElementById('back-to-grids-btn');
    const generateGridBtn = document.getElementById('generate-grid-btn');
    const addManualPostBtn = document.getElementById('add-manual-post-btn');
    const gridPostsBody = document.getElementById('grid-posts-body');
    
    const postModalEl = document.getElementById('postModal');
    const postModal = new bootstrap.Modal(postModalEl);
    const postModalLabel = document.getElementById('postModalLabel');
    const postForm = document.getElementById('post-form');
    const savePostBtn = document.getElementById('save-post-btn');
    const socialNetworksCheckboxes = document.getElementById('social-networks-checkboxes');
    const generateCreativeBtn = document.getElementById('generate-creative-btn');
    const creativeAssetContainer = document.getElementById('creative-asset-container');

    // --- URLs Y ESTADO ---
    const urlParams = new URLSearchParams(window.location.search);
    const gridId = urlParams.get('grid_id');

    const PROJECTS_API_URL = 'api/projects.php';
    const GRIDS_API_URL = 'api/grids.php';
    const GRID_POSTS_API_URL = 'api/grid_posts.php';

    const appState = {
        project: null,
        grid: null,
        gridPosts: [],
        currentPostId: null,
        isEditMode: false
    };

    // --- FUNCIONES DE CARGA INICIAL ---

    async function initialLoad() {
        if (!gridId) {
            gridNameTitle.textContent = "Grilla no especificada";
            return;
        }

        try {
            const gridRes = await fetch(`${GRIDS_API_URL}?id=${gridId}`);
            if (!gridRes.ok) throw new Error('No se pudo cargar la grilla.');
            appState.grid = await gridRes.json();

            const [projectRes, postsRes] = await Promise.all([
                fetch(`${PROJECTS_API_URL}?id=${appState.grid.project_id}`),
                fetch(`${GRID_POSTS_API_URL}?grid_id=${gridId}`)
            ]);

            if (!projectRes.ok) throw new Error('No se pudo cargar el proyecto asociado.');
            if (!postsRes.ok) throw new Error('No se pudieron cargar los posteos de la grilla.');
            
            appState.project = await projectRes.json();
            appState.gridPosts = await postsRes.json();

            gridNameTitle.textContent = appState.grid.name;
            projectNameSubtitle.textContent = `Proyecto: ${appState.project.nombre_proyecto}`;
            document.title = `Editor: ${appState.grid.name} - Platflow`;
            backToGridsBtn.href = `grid-detail.html?project_id=${appState.project.id}`;

            populateSocialCheckboxes();
            renderGridPosts();

        } catch (error) {
            console.error("Error en la carga inicial:", error);
            gridNameTitle.textContent = "Error";
            gridPostsBody.innerHTML = `<tr><td colspan="7" class="text-center text-danger">${error.message}</td></tr>`;
        }
    }

    function populateSocialCheckboxes() {
        socialNetworksCheckboxes.innerHTML = (appState.grid.networks || []).map(net => `
            <div class="form-check form-check-inline">
                <input class="form-check-input" type="checkbox" id="net-post-${net}" value="${net}">
                <label class="form-check-label" for="net-post-${net}">${net}</label>
            </div>
        `).join('');
    }

    // --- FUNCIONES DE DATOS ---

    async function fetchGridPosts() {
        try {
            const response = await fetch(`${GRID_POSTS_API_URL}?grid_id=${gridId}`);
            if (!response.ok) throw new Error('Error al recargar los posteos.');
            appState.gridPosts = await response.json();
            renderGridPosts();
        } catch (error) {
            console.error("Error:", error);
            gridPostsBody.innerHTML = `<tr><td colspan="7" class="text-center text-danger">${error.message}</td></tr>`;
        }
    }

    async function savePost() {
        const selectedNetworks = Array.from(socialNetworksCheckboxes.querySelectorAll('input:checked')).map(cb => cb.value);

        const postData = {
            grid_id: gridId,
            fecha_posteo: document.getElementById('post-date').value,
            hora_posteo: document.getElementById('post-time').value,
            tipo_contenido: document.getElementById('post-type').value,
            redes: selectedNetworks,
            objetivo: document.getElementById('post-objective').value,
            hook: document.getElementById('post-hook').value,
            copy: document.getElementById('post-copy').value,
            hashtags: document.getElementById('post-hashtags').value,
            cta: document.getElementById('post-cta').value,
            resultado_esperado: document.getElementById('post-expected-results').value,
        };

        const url = appState.isEditMode ? `${GRID_POSTS_API_URL}?id=${appState.currentPostId}` : GRID_POSTS_API_URL;
        const method = appState.isEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, { method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(postData) });
            if (!response.ok) throw new Error(`Error al ${appState.isEditMode ? 'actualizar' : 'guardar'} el posteo.`);
            
            postModal.hide();
            fetchGridPosts();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    async function deletePost(postId) {
        if (!confirm('¿Estás seguro de que quieres eliminar este posteo?')) return;

        try {
            const response = await fetch(`${GRID_POSTS_API_URL}?id=${postId}`, { method: 'DELETE' });
            if (!response.ok) throw new Error('No se pudo eliminar el posteo.');
            fetchGridPosts();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    async function togglePostStatus(postId, currentStatus) {
        const newStatus = !currentStatus;
        try {
            const response = await fetch(`${GRID_POSTS_API_URL}?id=${postId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ realizado: newStatus })
            });
            if (!response.ok) throw new Error('No se pudo actualizar el estado.');
            fetchGridPosts();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    // --- FUNCIONES DE IA ---

    async function generatePostsWithAI() {
        if (!appState.project) return;

        generateGridBtn.disabled = true;
        generateGridBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Generando...';

        try {
            const response = await fetch('api/ai_handler.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    task_type: 'generate_grid_posts',
                    context: {
                        estrategico: appState.project.estrategia_estrategico,
                        tactico: appState.project.estrategia_tactico,
                        operativo: appState.project.estrategia_operativo
                    }
                })
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || 'La IA no pudo generar los posteos.');
            }
            const newPosts = await response.json();

            for (const post of newPosts) {
                post.grid_id = gridId;
                await fetch(GRID_POSTS_API_URL, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(post) });
            }

            await fetchGridPosts();

        } catch (error) {
            console.error("Error con la IA:", error);
            alert(error.message);
        } finally {
            generateGridBtn.disabled = false;
            generateGridBtn.innerHTML = '<i class="bi bi-robot"></i> Generar Posts con IA';
        }
    }

    async function generateCreativeAsset() {
        const postData = {
            tipo_contenido: document.getElementById('post-type').value,
            objetivo: document.getElementById('post-objective').value,
            hook: document.getElementById('post-hook').value,
            copy: document.getElementById('post-copy').value,
        };

        if (!postData.copy) {
            alert('Por favor, complete al menos el campo "Copy" para dar contexto a la IA.');
            return;
        }

        generateCreativeBtn.disabled = true;
        generateCreativeBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Generando...';
        creativeAssetContainer.innerHTML = '<p class="text-muted">La IA está trabajando en una idea creativa...</p>';

        try {
            const response = await fetch('api/ai_handler.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    task_type: 'generate_creative_asset',
                    context: postData
                })
            });
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || 'La IA no pudo generar el activo creativo.');
            }

            const result = await response.json();
            const creativeAsset = result.creative_asset || "La IA no devolvió un resultado válido.";
            creativeAssetContainer.innerHTML = `<pre class="mb-0" style="white-space: pre-wrap; font-family: inherit;">${creativeAsset}</pre>`;

        } catch (error) {
            console.error("Error con IA de activo creativo:", error);
            creativeAssetContainer.innerHTML = `<p class="text-danger">${error.message}</p>`;
        } finally {
            generateCreativeBtn.disabled = false;
            generateCreativeBtn.innerHTML = '<i class="bi bi-magic"></i> Generar Activo Creativo con IA';
        }
    }

    // --- FUNCIONES DE RENDERIZADO Y UI ---

    function renderGridPosts() {
        gridPostsBody.innerHTML = '';
        if (appState.gridPosts.length === 0) {
            gridPostsBody.innerHTML = `<tr><td colspan="7" class="text-center text-muted">No hay posteos en la grilla. Use el botón "Añadir Post Manual" o "Generar con IA" para empezar.</td></tr>`;
            return;
        }

        const formatDate = (dateString) => {
            if (!dateString) return 'N/A';
            const [year, month, day] = dateString.split('-');
            if (!year || !month || !day) return dateString;
            return `${day}/${month}/${year}`;
        };

        appState.gridPosts.sort((a, b) => new Date(a.fecha_posteo) - new Date(b.fecha_posteo));

        appState.gridPosts.forEach(item => {
            const row = document.createElement('tr');
            const isChecked = item.realizado ? 'checked' : '';
            
            row.innerHTML = `
                <td class="text-center">
                    <input class="form-check-input post-status-check" type="checkbox" data-id="${item.id}" ${isChecked} title="${item.fecha_realizado || ''}">
                </td>
                <td>${formatDate(item.fecha_posteo)} ${item.hora_posteo || ''}</td>
                <td>${Array.isArray(item.redes) ? item.redes.join(', ') : 'N/A'}</td>
                <td>${item.tipo_contenido}</td>
                <td>${item.objetivo}</td>
                <td class="text-truncate" style="max-width: 250px;">${item.copy}</td>
                <td>
                    <button class="btn btn-sm btn-outline-secondary edit-post-btn" data-id="${item.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                    <button class="btn btn-sm btn-outline-danger delete-post-btn" data-id="${item.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                </td>
            `;
            gridPostsBody.appendChild(row);
        });
    }

    function openPostModal(post = null) {
        postForm.reset();
        creativeAssetContainer.innerHTML = '<p class="text-muted">Aquí se mostrará el prompt para la imagen o el guion generado por la IA...</p>';
        
        if (post) { // Modo Edición
            appState.isEditMode = true;
            appState.currentPostId = post.id;
            postModalLabel.textContent = 'Editar Posteo';
            document.getElementById('post-date').value = post.fecha_posteo;
            document.getElementById('post-time').value = post.hora_posteo;
            document.getElementById('post-type').value = post.tipo_contenido;
            document.getElementById('post-objective').value = post.objetivo;
            document.getElementById('post-hook').value = post.hook;
            document.getElementById('post-copy').value = post.copy;
            document.getElementById('post-hashtags').value = post.hashtags;
            document.getElementById('post-cta').value = post.cta;
            document.getElementById('post-expected-results').value = post.resultado_esperado;
            
            socialNetworksCheckboxes.querySelectorAll('input').forEach(cb => {
                cb.checked = post.redes && post.redes.includes(cb.value);
            });

        } else { // Modo Creación
            appState.isEditMode = false;
            appState.currentPostId = null;
            postModalLabel.textContent = 'Añadir Posteo Manual';
        }
        postModal.show();
    }

    // --- EVENT LISTENERS ---

    generateGridBtn.addEventListener('click', generatePostsWithAI);
    addManualPostBtn.addEventListener('click', () => openPostModal());
    savePostBtn.addEventListener('click', savePost);
    generateCreativeBtn.addEventListener('click', generateCreativeAsset);

    gridPostsBody.addEventListener('click', (e) => {
        const editBtn = e.target.closest('.edit-post-btn');
        if (editBtn) {
            const postId = editBtn.dataset.id;
            const post = appState.gridPosts.find(p => p.id == postId);
            if (post) openPostModal(post);
        }

        const deleteBtn = e.target.closest('.delete-post-btn');
        if (deleteBtn) {
            deletePost(deleteBtn.dataset.id);
        }

        const statusCheck = e.target.closest('.post-status-check');
        if (statusCheck) {
            togglePostStatus(statusCheck.dataset.id, statusCheck.checked);
        }
    });

    // --- CARGA INICIAL ---
    initialLoad();
});