document.addEventListener('DOMContentLoaded', function() {
    let currentUserId = null; // Se obtendrá de la sesión
    let currentConversations = [];

    // Elementos del DOM
    const conversationsList = document.getElementById('conv-list-items');
    const chatHeader = document.getElementById('chat-header');
    const chatMessages = document.getElementById('chat-messages');
    const messageInput = document.getElementById('message-input');
    const sendButton = document.getElementById('send-button');
    const newChatButton = document.getElementById('new-chat-button');
    const newChatModalEl = document.getElementById('new-chat-modal');
    const newChatModal = new bootstrap.Modal(newChatModalEl);
    const closeModalButtons = newChatModalEl.querySelectorAll('.close-button');
    const searchProfessionalInput = document.getElementById('search-professional');
    // Elementos para creación de grupos
    const groupNameInput = document.getElementById('group-name');
    const groupPhotoInput = document.getElementById('group-photo');
    const groupPhotoPreview = document.getElementById('group-photo-preview');
    const searchGroupMembersInput = document.getElementById('search-group-members');
    const groupMembersList = document.getElementById('group-members-list');
    const selectedMembersContainer = document.getElementById('selected-members-container');
    const createGroupButton = document.getElementById('create-group-button');
    // Elementos de opciones del chat
    const chatOptionsDropdown = document.getElementById('chat-options-dropdown');
    const editGroupBtn = document.getElementById('edit-group-btn');
    const deleteGroupBtn = document.getElementById('delete-group-btn');

    let activeConversationId = null;
    let professionals = [];
    let selectedGroupMembers = new Map();

    // Estado de la aplicación, como el modo de edición
    const appState = {
        isEditMode: false
    };

    // --- Funciones Principales ---

    async function loadConversations() {
        try {
            const response = await fetch('api/chat.php?action=get_conversations');
            currentConversations = await response.json();
            const conversations = currentConversations;
            
            const conversationsList = document.getElementById('conv-list-items');
            conversationsList.innerHTML = '';

            if (conversations.length === 0) {
                conversationsList.innerHTML = '<p style="padding: 15px; text-align: center;">No tienes conversaciones. Inicia una nueva.</p>';
                return;
            }

            conversations.forEach(conv => {
                const convElement = document.createElement('div');
                convElement.className = 'conversation-item';
                convElement.dataset.convId = conv.id;
                convElement.innerHTML = `
                    <img src="${conv.photo || 'https://via.placeholder.com/50'}" alt="${conv.name}">
                    <div class="conversation-details">
                        <p class="name">${conv.name}</p>
                        <p class="last-message">${conv.last_message}</p>
                    </div>
                `;
                convElement.addEventListener('click', () => loadMessages(conv.id));
                conversationsList.appendChild(convElement);
            });

        } catch (error) {
            console.error('Error cargando conversaciones:', error);
        }
    }

    async function loadMessages(conversationId) {
        if (!conversationId) return;
        activeConversationId = conversationId;

        // Comprobar si el usuario está cerca del final antes de cargar mensajes
        const nearBottom = chatMessages.scrollTop + chatMessages.clientHeight >= chatMessages.scrollHeight - 50;

        try {
            const response = await fetch(`api/chat.php?action=get_messages&conversation_id=${conversationId}`);
            const data = await response.json();

            // No recargar todo si los mensajes no han cambiado
            // (Esta es una optimización simple, se podría mejorar con checksums o timestamps)
            const newMessagesContent = JSON.stringify(data.messages);
            if (chatMessages.dataset.lastContent === newMessagesContent) {
                return; // No hacer nada si no hay mensajes nuevos
            }
            chatMessages.dataset.lastContent = newMessagesContent;


            // Actualizar cabecera del chat
            document.getElementById('chat-header-info').innerHTML = `
                <img src="${data.conversation.photo || 'https://via.placeholder.com/40'}" alt="${data.conversation.name}">
                <strong>${data.conversation.name}</strong>
            `;
            chatOptionsDropdown.style.display = data.conversation.is_group_chat ? 'block' : 'none';

            // Actualizar mensajes
            chatMessages.innerHTML = '';
            data.messages.forEach(msg => {
                const messageElement = document.createElement('div');
                const messageClass = parseInt(msg.sender_id) === currentUserId ? 'sent' : 'received';
                messageElement.className = `message ${messageClass}`;
                messageElement.dataset.messageId = msg.id;

                let actions = '';
                if (parseInt(msg.sender_id) === currentUserId) {
                    actions = `
                        <div class="message-actions">
                            <button class="edit-message-btn" title="Editar"><i class="bi bi-pencil"></i></button>
                            <button class="delete-message-btn" title="Eliminar"><i class="bi bi-trash"></i></button>
                        </div>
                    `;
                }

                messageElement.innerHTML = `<div class="bubble">${msg.message_content}</div>${actions}`;
                chatMessages.appendChild(messageElement);
            });

            // Solo hacer scroll automático si el usuario ya estaba al final
            if (nearBottom) {
                chatMessages.scrollTop = chatMessages.scrollHeight;
            }

            // Marcar conversación como activa
            document.querySelectorAll('.conversation-item').forEach(el => {
                el.classList.remove('active');
                if (parseInt(el.dataset.convId) === conversationId) {
                    el.classList.add('active');
                }
            });

        } catch (error) {
            console.error(`Error cargando mensajes para la conversación ${conversationId}:`, error);
        }
    }

    async function sendMessage() {
        const content = messageInput.value.trim();
        if (content === '' || !activeConversationId) return;

        const tempMessage = {
            message_content: content,
            sender_id: currentUserId
        };
        appendMessageToUI(tempMessage);
        messageInput.value = '';

        try {
            const response = await fetch('api/chat.php?action=send_message', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    conversation_id: activeConversationId,
                    message_content: content
                })
            });

            const savedMessage = await response.json();
            if (savedMessage.error) {
                console.error('Error al enviar el mensaje:', savedMessage.error);
                // TODO: Añadir lógica para mostrar error en la UI
            } else {
                // Actualizar la lista de conversaciones con el nuevo último mensaje
                updateConversationList(activeConversationId, content);
            }
        } catch (error) {
            console.error('Error de red al enviar mensaje:', error);
        }
    }

    function appendMessageToUI(message) {
        const messageElement = document.createElement('div');
        const messageClass = parseInt(message.sender_id) === currentUserId ? 'sent' : 'received';
        messageElement.className = `message ${messageClass}`;
        
        // No se puede editar/eliminar un mensaje temporal
        let actions = '';
        if (message.id) { // Solo si el mensaje ya está guardado y tiene ID
             actions = `
                <div class="message-actions">
                    <button class="edit-message-btn" title="Editar"><i class="bi bi-pencil"></i></button>
                    <button class="delete-message-btn" title="Eliminar"><i class="bi bi-trash"></i></button>
                </div>
            `;
        }

        messageElement.innerHTML = `<div class="bubble">${message.message_content}</div>${actions}`;
        chatMessages.appendChild(messageElement);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    function updateConversationList(conversationId, lastMessage) {
        const convElement = document.querySelector(`.conversation-item[data-conv-id='${conversationId}']`);
        if (convElement) {
            convElement.querySelector('.last-message').textContent = lastMessage;
            // Mover al principio de la lista
            convElement.parentElement.prepend(convElement);
        }
    }

    function renderProfessionalsList(profsToRender) {
        const professionalsList = document.getElementById('professionals-list');
        professionalsList.innerHTML = '';

        if (profsToRender.length === 0) {
            professionalsList.innerHTML = '<li class="list-group-item text-muted">No se encontraron profesionales.</li>';
            return;
        }

        profsToRender.forEach(prof => {
            if (prof.id === currentUserId) return; // No mostrar al usuario actual en la lista

            const li = document.createElement('li');
            // Usamos clases de Bootstrap para un mejor estilo
            li.className = 'list-group-item list-group-item-action d-flex align-items-center conversation-item';
            li.dataset.userId = prof.id;
            li.style.cursor = 'pointer';
            li.innerHTML = `
                <img src="${prof.photo || 'https://via.placeholder.com/50'}" alt="${prof.name}" class="rounded-circle me-3" width="50" height="50">
                <div class="conversation-details">
                    <p class="name mb-0 fw-bold">${prof.name}</p>
                </div>
            `;
            li.addEventListener('click', () => startIndividualChat(prof.id));
            professionalsList.appendChild(li);
        });
    }

    async function loadProfessionals() {
        if (professionals.length > 0) return; // No volver a cargar si ya los tenemos

        try {
            const response = await fetch('api/chat.php?action=get_professionals');
            professionals = await response.json();
            // Renderizar la lista completa la primera vez
            renderProfessionalsList(professionals);
        } catch (error) {
            console.error('Error cargando profesionales:', error);
        }
    }

    async function startIndividualChat(otherUserId) {
        try {
            const response = await fetch(`api/chat.php?action=get_or_create_individual_conversation&user_id=${otherUserId}`);
            const data = await response.json();
            if (data.conversation_id) {
                closeNewChatModal();
                await loadConversations(); // Recargar la lista de conversaciones
                loadMessages(data.conversation_id); // Cargar el nuevo chat
            } else {
                console.error('No se pudo obtener el ID de la conversación.', data.error);
            }
        } catch (error) {
            console.error('Error al iniciar el chat individual:', error);
        }
    }

    function handleProfessionalSearch() {
        const searchTerm = searchProfessionalInput.value.toLowerCase();
        const filteredProfessionals = professionals.filter(prof =>
            prof.name.toLowerCase().includes(searchTerm)
        );
        renderProfessionalsList(filteredProfessionals);
    }

    function handleGroupMemberSearch() {
        const searchTerm = searchGroupMembersInput.value.toLowerCase();
        groupMembersList.innerHTML = '';
        if (searchTerm.length === 0) return;

        const filtered = professionals.filter(prof =>
            prof.name.toLowerCase().includes(searchTerm) &&
            prof.id !== currentUserId && // No puedes añadirte a ti mismo
            !selectedGroupMembers.has(prof.id) // No mostrar si ya está seleccionado
        );

        filtered.forEach(prof => {
            const li = document.createElement('li');
            li.className = 'list-group-item list-group-item-action';
            li.style.cursor = 'pointer';
            li.textContent = prof.name;
            li.addEventListener('click', () => selectMemberForGroup(prof));
            groupMembersList.appendChild(li);
        });
    }

    // --- Lógica del Modal ---

    function setupModalTabs() {
        const tabs = document.querySelectorAll('#modal-tabs .tab-link');
        const tabContents = document.querySelectorAll('#new-chat-modal .tab-content');

        tabs.forEach(tab => {
            tab.addEventListener('click', () => {
                // Desactivar todos
                tabs.forEach(t => t.classList.remove('active'));
                tabContents.forEach(c => c.classList.remove('active'));

                // Activar el seleccionado
                tab.classList.add('active');
                const targetContentId = tab.getAttribute('data-tab');
                document.getElementById(targetContentId).classList.add('active');

                // Mostrar/ocultar el botón de crear grupo
                if (targetContentId === 'group') {
                    createGroupButton.style.display = 'block';
                    createGroupButton.textContent = 'Crear Grupo';
                    // Si estamos editando, el título del modal cambia
                    if (!appState.isEditMode) {
                        newChatModalEl.querySelector('.modal-title').textContent = 'Nueva Conversación';
                    }
                } else {
                    createGroupButton.style.display = 'none';
                }
            });
        });
    }

    function selectMemberForGroup(prof) {
        if (selectedGroupMembers.has(prof.id) || prof.id === currentUserId) return;

        selectedGroupMembers.set(prof.id, prof);
        searchGroupMembersInput.value = '';
        groupMembersList.innerHTML = '';
        renderSelectedMembers();
    }

    function removeMemberFromGroup(userId) {
        selectedGroupMembers.delete(userId);
        renderSelectedMembers();
    }

    function renderSelectedMembers() {
        selectedMembersContainer.innerHTML = '';
        selectedGroupMembers.forEach(prof => {
            const pill = document.createElement('span');
            pill.className = 'selected-member-pill';
            pill.innerHTML = `
                ${prof.name}
                <span class="remove-member-btn" data-user-id="${prof.id}">&times;</span>
            `;
            selectedMembersContainer.appendChild(pill);
        });
    }

    async function createGroup() {
        const groupName = groupNameInput.value.trim();
        const memberIds = Array.from(selectedGroupMembers.keys());
        const groupPhotoFile = groupPhotoInput.files[0];
        const isEditing = appState.isEditMode;

        if (isEditing && !activeConversationId) {
            alert('Error: No se ha identificado el grupo a editar.');
        }

        if (!groupName) {
            alert('Por favor, dale un nombre al grupo.');
            return;
        }
        if (memberIds.length === 0) {
            alert('Debes añadir al menos un miembro al grupo.');
            return;
        }

        const formData = new FormData();
        formData.append('name', groupName);
        formData.append('members', JSON.stringify(memberIds)); // Enviamos los IDs como un string JSON
        
        if (groupPhotoFile) {
            formData.append('group_photo', groupPhotoFile);
        }

        const url = isEditing 
            ? `api/chat.php?action=edit_group&id=${activeConversationId}`
            : 'api/chat.php?action=create_group';
        
        try {
            
            // Al usar FormData, no se especifica el Content-Type, el navegador lo hace automáticamente
            const response = await fetch(url, {
                method: 'POST',
                body: formData,
            });
            const data = await response.json();
            if (data.conversation_id) {
                closeNewChatModal();
                await loadConversations();
                loadMessages(data.conversation_id);
            } else {
                throw new Error(data.error || `Error desconocido al ${isEditing ? 'editar' : 'crear'} el grupo.`);
            }
        } catch (error) {
            console.error(`Error al ${isEditing ? 'editar' : 'crear'} el grupo:`, error);
            alert(`No se pudo ${isEditing ? 'editar' : 'crear'} el grupo. Inténtalo de nuevo.`);
        }
    }

    function openNewChatModal() {
        newChatModal.show();
        searchProfessionalInput.value = ''; // Limpiar búsqueda anterior
        loadProfessionals();
    }

    function closeNewChatModal() {
        // Resetear formulario de grupo
        groupNameInput.value = '';
        groupPhotoInput.value = ''; // Limpiar el input de archivo
        groupPhotoPreview.style.display = 'none'; // Ocultar preview
        searchGroupMembersInput.value = '';
        groupMembersList.innerHTML = '';
        selectedGroupMembers.clear();
        renderSelectedMembers();
        appState.isEditMode = false; // Asegurarse de resetear el modo edición
        newChatModal.hide();
    }

    // --- Event Listeners ---

    newChatButton.addEventListener('click', openNewChatModal);
    closeModalButtons.forEach(btn => {
        btn.addEventListener('click', () => {
            closeNewChatModal();
        });
    });

    sendButton.addEventListener('click', sendMessage);
    messageInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            sendMessage();
        }
    });

    searchProfessionalInput.addEventListener('input', handleProfessionalSearch);
    searchGroupMembersInput.addEventListener('input', handleGroupMemberSearch);
    groupPhotoInput.addEventListener('change', () => {
        const file = groupPhotoInput.files[0];
        if (file) {
            groupPhotoPreview.src = URL.createObjectURL(file);
            groupPhotoPreview.style.display = 'block';
        }
    });
    createGroupButton.addEventListener('click', createGroup);

    selectedMembersContainer.addEventListener('click', (e) => {
        if (e.target.classList.contains('remove-member-btn')) {
            removeMemberFromGroup(parseInt(e.target.dataset.userId));
        }
    });

    // --- Listeners para acciones de mensajes y grupos ---

    chatMessages.addEventListener('click', e => {
        const messageElement = e.target.closest('.message');
        if (!messageElement) return;
        const messageId = messageElement.dataset.messageId;

        if (e.target.closest('.delete-message-btn')) {
            deleteMessage(messageId, messageElement);
        }
        if (e.target.closest('.edit-message-btn')) {
            editMessage(messageId, messageElement);
        }
    });

    editGroupBtn.addEventListener('click', () => {
        const conversation = currentConversations.find(c => c.id === activeConversationId);
        if (!conversation || !conversation.is_group_chat) return;

        appState.isEditMode = true;
        newChatModalEl.querySelector('.modal-title').textContent = 'Editar Grupo';
        
        // Forzar la vista de grupo
        document.querySelector('.tab-link[data-tab="group"]').click();
        createGroupButton.textContent = 'Guardar Cambios';

        groupNameInput.value = conversation.name;
        // Lógica para cargar miembros existentes (requiere más datos de la API)
        // Por ahora, se deja vacío para añadir/quitar.
        selectedGroupMembers.clear();
        // Aquí necesitaríamos una llamada a la API para obtener los miembros actuales del grupo
        // y poblarlos en `selectedGroupMembers` y `renderSelectedMembers()`.
        // Como no lo tenemos, el modal permitirá cambiar nombre y añadir nuevos miembros.

        newChatModal.show();
    });

    deleteGroupBtn.addEventListener('click', async () => {
        if (!activeConversationId) return;
        if (!confirm('¿Estás seguro de que quieres eliminar este grupo? Esta acción no se puede deshacer.')) return;

        try {
            const response = await fetch(`api/chat.php?action=delete_group&id=${activeConversationId}`, { method: 'POST' });
            const result = await response.json();
            if (result.success) {
                activeConversationId = null;
                document.getElementById('chat-header-info').innerHTML = '';
                chatMessages.innerHTML = '';
                chatOptionsDropdown.style.display = 'none';
                await loadConversations();
            } else {
                throw new Error(result.error || 'No se pudo eliminar el grupo.');
            }
        } catch (error) {
            alert(error.message);
        }
    });

    async function deleteMessage(messageId, element) {
        if (!confirm('¿Eliminar este mensaje?')) return;
        try {
            const response = await fetch(`api/chat.php?action=delete_message&id=${messageId}`, { method: 'POST' });
            const result = await response.json();
            if (result.success) {
                element.remove();
            } else {
                throw new Error(result.error || 'No se pudo eliminar el mensaje.');
            }
        } catch (error) {
            alert(error.message);
        }
    }

    function editMessage(messageId, element) {
        const bubble = element.querySelector('.bubble');
        const originalContent = bubble.innerHTML;
        bubble.innerHTML = `<input type="text" class="form-control form-control-sm" value="${originalContent}">`;
        const input = bubble.querySelector('input');
        input.focus();

        input.addEventListener('blur', async () => { // Guardar al perder el foco
            const newContent = input.value.trim();
            if (newContent && newContent !== originalContent) {
                try {
                    const response = await fetch(`api/chat.php?action=edit_message`, {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ message_id: messageId, content: newContent })
                    });
                    const result = await response.json();
                    if (result.success) bubble.innerHTML = newContent; else throw new Error();
                } catch (e) { bubble.innerHTML = originalContent; alert('No se pudo editar.'); }
            } else { bubble.innerHTML = originalContent; }
        });
        input.addEventListener('keydown', e => { if (e.key === 'Enter') input.blur(); });
    }

    async function initializeChat() {
        try {
            const response = await fetch('api/chat.php?action=get_session_user');
            const data = await response.json();
            if (data.user_id) {
                currentUserId = data.user_id;
                loadConversations();
            } else {
                console.error('No se pudo obtener el ID del usuario. Es posible que no haya iniciado sesión.');
                document.getElementById('chat-container').innerHTML = '<p style="text-align: center; padding: 20px;">Error de autenticación. Por favor, <a href="login.html">inicie sesión</a> de nuevo.</p>';
            }
        } catch (error) {
            console.error('Error al inicializar el chat:', error);
            document.getElementById('chat-container').innerHTML = '<p style="text-align: center; padding: 20px;">Error al cargar el chat. Inténtelo de nuevo más tarde.</p>';
        }
    }

    // --- Inicialización ---

    initializeChat();
    setupModalTabs();

    // Polling para nuevos mensajes (se activará cuando una conversación esté abierta)
    setInterval(() => {
        if (activeConversationId) {
            loadMessages(activeConversationId);
        }
    }, 5000); // Cada 5 segundos

});
