<?php
// Backend para gestionar los proyectos (CRUD)

header("Content-Type: application/json");

$database_path = '../config/database.json';

// Función para leer la base de datos
function get_database() {
    global $database_path;
    if (!file_exists($database_path)) {
        $base_structure = ["professionals" => [], "clients" => [], "projects" => [], "contacts" => []];
        file_put_contents($database_path, json_encode($base_structure, JSON_PRETTY_PRINT));
        return $base_structure;
    }
    $json_data = file_get_contents($database_path);
    return json_decode($json_data, true);
}

// Función para guardar la base de datos
function save_database($data) {
    global $database_path;
    $json_data = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    file_put_contents($database_path, $json_data);
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

if ($method === 'GET') {
    $db = get_database();
    if ($id !== null) {
        $project = array_values(array_filter($db['projects'], function($proj) use ($id) {
            return $proj['id'] === $id;
        }));
        if (count($project) > 0) {
            echo json_encode($project[0]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Proyecto no encontrado']);
        }
    } else {
        // Asegurarnos que 'projects' exista
        echo json_encode($db['projects'] ?? []);
    }
} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (empty($input['nombre_proyecto']) || empty($input['cliente_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El nombre del proyecto y el cliente son requeridos.']);
        exit;
    }

    $db = get_database();
    
    $new_id = 1;
    if (!empty($db['projects'])) {
        $ids = array_column($db['projects'], 'id');
        $new_id = max($ids) + 1;
    }
    
    $new_project = [
        'id' => $new_id,
        'nombre_proyecto' => $input['nombre_proyecto'] ?? '',
        'cliente_id' => $input['cliente_id'] ?? null,
        'descripcion' => $input['descripcion'] ?? '',
        'fecha_inicio' => $input['fecha_inicio'] ?? null,
        'fecha_fin_estimada' => $input['fecha_fin_estimada'] ?? null,
        'tipo_marketing' => $input['tipo_marketing'] ?? 'Mixto',
        'estado' => $input['estado'] ?? 'Planificación',
        'profesionales_asignados' => $input['profesionales_asignados'] ?? [],
        'estrategia_estrategico' => $input['estrategia_estrategico'] ?? '',
        'estrategia_tactico' => $input['estrategia_tactico'] ?? '',
        'estrategia_operativo' => $input['estrategia_operativo'] ?? '',
        'budget_total' => 0,
        'ingresos' => [],
        'egresos' => []
    ];

    $db['projects'][] = $new_project;
    save_database($db);

    http_response_code(201);
    echo json_encode($new_project);

} elseif ($method === 'PUT' && $id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);
    $db = get_database();
    
    $updated_project = null;
    $found = false;

    foreach ($db['projects'] as $key => $proj) {
        if ($proj['id'] === $id) {
            // Lógica para añadir ingresos/egresos sin sobreescribir
            if (isset($input['new_ingreso'])) {
                if (!isset($proj['ingresos'])) $proj['ingresos'] = [];
                $proj['ingresos'][] = $input['new_ingreso'];
                unset($input['new_ingreso']);
            }
            if (isset($input['new_egreso'])) {
                if (!isset($proj['egresos'])) $proj['egresos'] = [];
                $proj['egresos'][] = $input['new_egreso'];
                unset($input['new_egreso']);
            }

            $db['projects'][$key] = array_merge($proj, $input);
            $db['projects'][$key]['id'] = $id; // Asegurar que el ID no cambie
            $updated_project = $db['projects'][$key];
            $found = true;
            break;
        }
    }

    if ($found) {
        save_database($db);
        echo json_encode($updated_project);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Proyecto no encontrado para actualizar']);
    }

} elseif ($method === 'DELETE' && $id !== null) {
    $db = get_database();
    
    $original_count = count($db['projects']);
    $db['projects'] = array_values(array_filter($db['projects'], function($proj) use ($id) {
        return $proj['id'] !== $id;
    }));
    
    if (count($db['projects']) < $original_count) {
        save_database($db);
        http_response_code(204);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Proyecto no encontrado']);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID no proporcionado']);
}
?>