<?php
header("Content-Type: application/json");

$database_path = '../config/database.json';

function get_database() {
    global $database_path;
    if (!file_exists($database_path)) {
        return ["professionals" => [], "clients" => [], "projects" => [], "tasks" => [], "grid_posts" => [], "content_grids" => []];
    }
    $json_data = file_get_contents($database_path);
    $data = json_decode($json_data, true);
    if (!isset($data['content_grids'])) $data['content_grids'] = [];
    return $data;
}

function save_database($data) {
    global $database_path;
    file_put_contents($database_path, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

$method = $_SERVER['REQUEST_METHOD'];
$project_id = isset($_GET['project_id']) ? (int)$_GET['project_id'] : null;
$grid_id = isset($_GET['id']) ? (int)$_GET['id'] : null;

if ($method === 'GET') {
    $db = get_database();
    if ($grid_id !== null) {
        // Obtener una grilla específica por su ID
        $grid = array_values(array_filter($db['content_grids'], function($g) use ($grid_id) {
            return $g['id'] === $grid_id;
        }));
        if (count($grid) > 0) {
            echo json_encode($grid[0]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Grilla no encontrada']);
        }
    } elseif ($project_id !== null) {
        // Obtener todas las grillas de un proyecto
        $grids = array_values(array_filter($db['content_grids'], function($g) use ($project_id) {
            return isset($g['project_id']) && $g['project_id'] == $project_id;
        }));
        echo json_encode($grids);
    } else {
        http_response_code(400);
        echo json_encode(['error' => 'Se requiere un project_id o un id de grilla.']);
    }

} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    if (empty($input['project_id']) || empty($input['name']) || empty($input['start_date']) || empty($input['end_date'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Nombre, fechas y ID de proyecto son obligatorios.']);
        exit;
    }

    $db = get_database();
    
    $new_id = empty($db['content_grids']) ? 1 : max(array_column($db['content_grids'], 'id')) + 1;
    
    $new_grid = [
        'id' => $new_id,
        'project_id' => (int)$input['project_id'],
        'name' => $input['name'],
        'start_date' => $input['start_date'],
        'end_date' => $input['end_date'],
        'networks' => $input['networks'] ?? [],
        'responsables' => $input['responsables'] ?? []
    ];

    $db['content_grids'][] = $new_grid;
    save_database($db);

    http_response_code(201);
    echo json_encode($new_grid);

} elseif ($method === 'PUT' && $grid_id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);
    $db = get_database();
    
    $updated_grid = null;
    $found = false;

    foreach ($db['content_grids'] as $key => $grid) {
        if ($grid['id'] === $grid_id) {
            $db['content_grids'][$key] = array_merge($grid, $input);
            $db['content_grids'][$key]['id'] = $grid_id; // Asegurar que el ID no cambie
            $updated_grid = $db['content_grids'][$key];
            $found = true;
            break;
        }
    }

    if ($found) {
        save_database($db);
        echo json_encode($updated_grid);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Grilla no encontrada para actualizar']);
    }
} elseif ($method === 'DELETE' && $grid_id !== null) {
    $db = get_database();
    $original_count = count($db['content_grids']);
    
    // Eliminar la grilla
    $db['content_grids'] = array_values(array_filter($db['content_grids'], function($g) use ($grid_id) {
        return $g['id'] !== $grid_id;
    }));
    // Eliminar los posteos asociados a esa grilla
    $db['grid_posts'] = array_values(array_filter($db['grid_posts'], function($p) use ($grid_id) {
        return $p['grid_id'] !== $grid_id;
    }));
    
    if (count($db['content_grids']) < $original_count) {
        save_database($db);
        http_response_code(204);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Grilla no encontrada']);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
}
?>