<?php
// Backend para gestionar los contactos (CRUD)

header("Content-Type: application/json");

$database_path = '../config/database.json';

// Función para leer la base de datos
function get_database() {
    global $database_path;
    if (!file_exists($database_path)) {
        $base_structure = ["professionals" => [], "clients" => [], "projects" => [], "contacts" => []];
        file_put_contents($database_path, json_encode($base_structure, JSON_PRETTY_PRINT));
        return $base_structure;
    }
    $json_data = file_get_contents($database_path);
    return json_decode($json_data, true);
}

// Función para guardar la base de datos
function save_database($data) {
    global $database_path;
    $json_data = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    file_put_contents($database_path, $json_data);
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

if ($method === 'GET') {
    $db = get_database();
    if ($id !== null) {
        $contact = array_values(array_filter($db['contacts'], function($cont) use ($id) {
            return $cont['id'] === $id;
        }));
        if (count($contact) > 0) {
            echo json_encode($contact[0]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Contacto no encontrado']);
        }
    } else {
        echo json_encode($db['contacts']);
    }
} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    // Validación básica
    if (empty($input['empresa_id']) || empty($input['nombre_completo'])) {
        http_response_code(400); // Bad Request
        echo json_encode(['error' => 'El nombre y la empresa son campos requeridos.']);
        exit;
    }
    $db = get_database();
    
    $new_id = 1;
    if (!empty($db['contacts'])) {
        $ids = array_column($db['contacts'], 'id');
        $new_id = max($ids) + 1;
    }
    
    $new_contact = [
        'id' => $new_id,
        'empresa_id' => $input['empresa_id'] ?? null,
        'nombre_completo' => $input['nombre_completo'] ?? '',
        'cargo' => $input['cargo'] ?? '',
        'correo' => $input['correo'] ?? '',
        'telefono' => $input['telefono'] ?? ''
    ];

    $db['contacts'][] = $new_contact;
    save_database($db);

    http_response_code(201);
    echo json_encode($new_contact);

} elseif ($method === 'PUT' && $id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);
    $db = get_database();
    
    $updated_contact = null;
    $found = false;

    foreach ($db['contacts'] as $key => $cont) {
        if ($cont['id'] === $id) {
            $db['contacts'][$key] = array_merge($cont, $input);
            $db['contacts'][$key]['id'] = $id;
            $updated_contact = $db['contacts'][$key];
            $found = true;
            break;
        }
    }

    if ($found) {
        save_database($db);
        echo json_encode($updated_contact);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Contacto no encontrado para actualizar']);
    }

} elseif ($method === 'DELETE' && $id !== null) {
    $db = get_database();
    
    $original_count = count($db['contacts']);
    $db['contacts'] = array_filter($db['contacts'], function($cont) use ($id) {
        return $cont['id'] !== $id;
    });
    
    if (count($db['contacts']) < $original_count) {
        $db['contacts'] = array_values($db['contacts']);
        save_database($db);
        http_response_code(204);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Contacto no encontrado']);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID no proporcionado']);
}

?>
