<?php
session_start();
header('Content-Type: application/json');

// --- Funciones de Ayuda para la Base de Datos JSON ---
function readDatabase() {
    $db_file = __DIR__ . '/../config/database.json';
    if (!file_exists($db_file)) {
        // Si no existe, creamos la estructura base con arrays vacíos para el chat
        $base_structure = [
            'professionals' => [],
            'clients' => [],
            'projects' => [],
            'tasks' => [],
            'chat_conversations' => [],
            'chat_messages' => [],
            'chat_conversation_members' => []
        ];
        writeDatabase($base_structure);
        return $base_structure;
    }
    $json_data = file_get_contents($db_file);
    $data = json_decode($json_data, true);

    // Asegurarse de que las claves de chat existan
    if (!isset($data['chat_conversations'])) {
        $data['chat_conversations'] = [];
    }
    if (!isset($data['chat_messages'])) {
        $data['chat_messages'] = [];
    }
    return $data;
}

function writeDatabase($data) {
    $db_file = __DIR__ . '/../config/database.json';
    // Usar JSON_UNESCAPED_UNICODE para correcta visualización de caracteres como ñ o tildes
    $json_data = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    file_put_contents($db_file, $json_data, LOCK_EX);
}

/**
 * Maneja la subida de un archivo de imagen para el chat.
 * @param array $file - El array del archivo de $_FILES.
 * @return string|null - La URL del archivo guardado o null si hay error.
 */
function handleChatPhotoUpload($file) {
    if (isset($file) && $file['error'] === UPLOAD_ERR_OK) {
        $upload_dir = __DIR__ . '/../uploads/chat_photos/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $file_name = 'group_' . uniqid() . '.' . $file_extension;
        $target_path = $upload_dir . $file_name;

        if (move_uploaded_file($file['tmp_name'], $target_path)) {
            // Devuelve la URL relativa accesible desde el cliente
            return 'uploads/chat_photos/' . $file_name;
        }
    }
    return null;
}

// --- Lógica Principal ---

$action = $_GET['action'] ?? '';

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'No autenticado. Por favor, inicie sesión.']);
    exit();
}
$currentUser = $_SESSION['user_id'];

switch ($action) {
    case 'get_session_user':
        echo json_encode(['user_id' => $currentUser]);
        break;

    case 'get_professionals':
        $db = readDatabase();
        $professionals = $db['professionals'] ?? [];
        $users = array_map(function($prof) {
            return [
                'id' => $prof['id'],
                'name' => $prof['nombre_completo'],
                'photo' => $prof['foto_url']
            ];
        }, $professionals);
        echo json_encode($users);
        break;

    case 'get_conversations':
        $db = readDatabase();
        $my_id = $currentUser;

        $my_memberships = array_filter($db['chat_conversation_members'] ?? [], function($m) use ($my_id) {
            return $m['user_id'] == $my_id;
        });

        $my_conv_ids = array_column($my_memberships, 'conversation_id');

        $conversations = array_filter($db['chat_conversations'] ?? [], function($c) use ($my_conv_ids) {
            return in_array($c['id'], $my_conv_ids);
        });

        $result = [];
        foreach ($conversations as $conv) {
            $conv_id = $conv['id'];
            $last_message = array_reduce($db['chat_messages'] ?? [], function ($latest, $msg) use ($conv_id) {
                if ($msg['conversation_id'] == $conv_id) {
                    if ($latest == null || $msg['timestamp'] > $latest['timestamp']) {
                        return $msg;
                    }
                }
                return $latest;
            });

            $conv_data = [
                'id' => $conv_id,
                'is_group_chat' => $conv['is_group_chat'],
                'last_message' => $last_message ? $last_message['message_content'] : 'No hay mensajes aún.',
                'last_message_time' => $last_message ? $last_message['timestamp'] : $conv['created_at']
            ];

            if ($conv['is_group_chat']) {
                $conv_data['name'] = $conv['name'];
                $conv_data['photo'] = $conv['group_photo_url'];
            } else {
                $members = array_filter($db['chat_conversation_members'], function($m) use ($conv_id) {
                    return $m['conversation_id'] == $conv_id;
                });
                $other_user_id = null;
                foreach ($members as $member) {
                    if ($member['user_id'] != $my_id) {
                        $other_user_id = $member['user_id'];
                        break;
                    }
                }

                if ($other_user_id) {
                    $other_user_details = array_values(array_filter($db['professionals'], function($p) use ($other_user_id) {
                        return $p['id'] == $other_user_id;
                    }));
                    if (!empty($other_user_details)) {
                        $conv_data['name'] = $other_user_details[0]['nombre_completo'];
                        $conv_data['photo'] = $other_user_details[0]['foto_url'];
                    }
                }
            }
            $result[] = $conv_data;
        }
        
        // Ordenar por el último mensaje
        usort($result, function($a, $b) {
            return strtotime($b['last_message_time']) - strtotime($a['last_message_time']);
        });

        echo json_encode($result);
        break;

    case 'check_new_messages':
        $last_check_timestamp = $_GET['since'] ?? '1970-01-01 00:00:00';
        $db = readDatabase();
        $my_id = $currentUser;

        // Obtener IDs de mis conversaciones
        $my_memberships = array_filter($db['chat_conversation_members'] ?? [], fn($m) => $m['user_id'] == $my_id);
        $my_conv_ids = array_column($my_memberships, 'conversation_id');

        if (empty($my_conv_ids)) {
            echo json_encode(['has_new_messages' => false]);
            break;
        }

        // Buscar mensajes nuevos en mis conversaciones que no sean míos
        $has_new = !empty(array_filter($db['chat_messages'] ?? [], function($msg) use ($my_conv_ids, $my_id, $last_check_timestamp) {
            return in_array($msg['conversation_id'], $my_conv_ids) &&
                   $msg['sender_id'] != $my_id &&
                   strtotime($msg['timestamp']) > strtotime($last_check_timestamp);
        }));

        echo json_encode([
            'has_new_messages' => $has_new
        ]);
        break;

    case 'send_message':
        $input = json_decode(file_get_contents('php://input'), true);
        $conversationId = $input['conversation_id'] ?? 0;
        $messageContent = trim($input['message_content'] ?? '');

        if ($conversationId === 0 || empty($messageContent)) {
            http_response_code(400);
            echo json_encode(['error' => 'Datos incompletos.']);
            break;
        }

        $db = readDatabase();
        $my_id = $currentUser;

        // Validar que el usuario pertenezca a la conversación
        $is_member = !empty(array_filter($db['chat_conversation_members'] ?? [], function($m) use ($my_id, $conversationId) {
            return $m['conversation_id'] == $conversationId && $m['user_id'] == $my_id;
        }));

        if (!$is_member) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes permiso para enviar mensajes a esta conversación.']);
            break;
        }

        // Crear nuevo mensaje
        $new_message_id = empty($db['chat_messages']) ? 1 : max(array_column($db['chat_messages'], 'id')) + 1;
        $new_message = [
            'id' => $new_message_id,
            'conversation_id' => $conversationId,
            'sender_id' => $my_id,
            'message_content' => $messageContent,
            'timestamp' => date('Y-m-d H:i:s')
        ];

        $db['chat_messages'][] = $new_message;
        writeDatabase($db);

        echo json_encode($new_message);
        break;

    case 'get_messages':
        $conversationId = (int)($_GET['conversation_id'] ?? 0);
        if ($conversationId === 0) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de conversación no válido.']);
            break;
        }

        $db = readDatabase();
        $my_id = $currentUser;

        // Validar que el usuario pertenezca a la conversación
        $is_member = !empty(array_filter($db['chat_conversation_members'] ?? [], function($m) use ($my_id, $conversationId) {
            return $m['conversation_id'] == $conversationId && $m['user_id'] == $my_id;
        }));

        if (!$is_member) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes acceso a esta conversación.']);
            break;
        }

        // Obtener mensajes
        $messages = array_filter($db['chat_messages'] ?? [], function($msg) use ($conversationId) {
            return $msg['conversation_id'] == $conversationId;
        });

        // Obtener detalles de la conversación para la cabecera
        $conversation = array_values(array_filter($db['chat_conversations'] ?? [], function($c) use ($conversationId) {
            return $c['id'] == $conversationId;
        }))[0];

        $conv_details = [
            'is_group_chat' => $conversation['is_group_chat']
        ];
        if ($conversation['is_group_chat']) {
            $conv_details['name'] = $conversation['name'];
            $conv_details['photo'] = $conversation['group_photo_url'];
        } else {
            $members = array_filter($db['chat_conversation_members'], function($m) use ($conversationId, $my_id) {
                return $m['conversation_id'] == $conversationId && $m['user_id'] != $my_id;
            });
            $other_user_id = array_values($members)[0]['user_id'];
            
            $other_user_details = array_values(array_filter($db['professionals'], function($p) use ($other_user_id) {
                return $p['id'] == $other_user_id;
            }))[0];

            $conv_details['name'] = $other_user_details['nombre_completo'];
            $conv_details['photo'] = $other_user_details['foto_url'];
        }

        echo json_encode([
            'conversation' => $conv_details,
            'messages' => array_values($messages)
        ]);
        break;

    case 'get_or_create_individual_conversation':
        $otherUserId = (int)($_GET['user_id'] ?? 0);
        if ($otherUserId === 0) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de usuario no válido.']);
            break;
        }

        $db = readDatabase();
        $conversations = $db['chat_conversations'] ?? [];
        $members = $db['chat_conversation_members'] ?? [];

        // Buscar si ya existe una conversación individual
        $found_conversation_id = null;
        $user_conversations = array_filter($members, function($m) use ($currentUser) {
            return $m['user_id'] == $currentUser;
        });

        foreach ($user_conversations as $user_conv) {
            $conv_id = $user_conv['conversation_id'];
            $conv_details = array_values(array_filter($conversations, function($c) use ($conv_id) {
                return $c['id'] == $conv_id && !$c['is_group_chat'];
            }));

            if (!empty($conv_details)) {
                $members_of_conv = array_filter($members, function($m) use ($conv_id) {
                    return $m['conversation_id'] == $conv_id;
                });
                $is_target_conv = false;
                foreach ($members_of_conv as $member) {
                    if ($member['user_id'] == $otherUserId) {
                        $is_target_conv = true;
                        break;
                    }
                }
                if (count($members_of_conv) == 2 && $is_target_conv) {
                    $found_conversation_id = $conv_id;
                    break;
                }
            }
        }

        if ($found_conversation_id) {
            echo json_encode(['conversation_id' => $found_conversation_id]);
        } else {
            // Crear nueva conversación
            $new_conv_id = empty($conversations) ? 1 : max(array_column($conversations, 'id')) + 1;
            $new_conversation = [
                'id' => $new_conv_id,
                'name' => null,
                'group_photo_url' => null,
                'is_group_chat' => false,
                'created_by' => $currentUser,
                'created_at' => date('Y-m-d H:i:s')
            ];
            $db['chat_conversations'][] = $new_conversation;

            // Añadir miembros
            $db['chat_conversation_members'][] = ['conversation_id' => $new_conv_id, 'user_id' => $currentUser];
            $db['chat_conversation_members'][] = ['conversation_id' => $new_conv_id, 'user_id' => $otherUserId];

            writeDatabase($db);
            echo json_encode(['conversation_id' => $new_conv_id]);
        }
        break;

    case 'create_group':
        // Ahora leemos desde $_POST y $_FILES porque usamos FormData
        $groupName = trim($_POST['name'] ?? '');
        // Los IDs de miembros vienen como un string JSON, hay que decodificarlo
        $memberIds = json_decode($_POST['members'] ?? '[]', true);
        $groupPhotoFile = $_FILES['group_photo'] ?? null;

        if (empty($groupName) || empty($memberIds)) {
            http_response_code(400);
            echo json_encode(['error' => 'El nombre del grupo y los miembros son requeridos.']);
            break;
        }

        $db = readDatabase();
        
        // Manejar la subida de la foto
        $photo_url = handleChatPhotoUpload($groupPhotoFile);

        // Crear la nueva conversación de grupo
        $new_conv_id = empty($db['chat_conversations']) ? 1 : max(array_column($db['chat_conversations'], 'id')) + 1;
        $new_conversation = [
            'id' => $new_conv_id,
            'name' => $groupName,
            'group_photo_url' => $photo_url,
            'is_group_chat' => true,
            'created_by' => $currentUser,
            'created_at' => date('Y-m-d H:i:s')
        ];
        $db['chat_conversations'][] = $new_conversation;

        // Añadir al creador y a los miembros a la conversación
        $all_member_ids = array_unique(array_merge([$currentUser], $memberIds));

        foreach ($all_member_ids as $user_id) {
            $db['chat_conversation_members'][] = ['conversation_id' => $new_conv_id, 'user_id' => (int)$user_id];
        }

        writeDatabase($db);

        echo json_encode(['conversation_id' => $new_conv_id]);
        break;

    case 'edit_group':
        $groupId = (int)($_GET['id'] ?? 0);
        if ($groupId === 0) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de grupo no válido.']);
            break;
        }

        $db = readDatabase();

        // Validar que el usuario sea el creador del grupo para poder editarlo
        $group_key = array_search($groupId, array_column($db['chat_conversations'], 'id'));
        if ($group_key === false || $db['chat_conversations'][$group_key]['created_by'] != $currentUser) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes permiso para editar este grupo.']);
            break;
        }

        $groupName = trim($_POST['name'] ?? '');
        $memberIds = json_decode($_POST['members'] ?? '[]', true);
        $groupPhotoFile = $_FILES['group_photo'] ?? null;

        if (empty($groupName)) {
            http_response_code(400);
            echo json_encode(['error' => 'El nombre del grupo es requerido.']);
            break;
        }

        // Actualizar nombre
        $db['chat_conversations'][$group_key]['name'] = $groupName;

        // Actualizar foto si se subió una nueva
        if ($groupPhotoFile) {
            $photo_url = handleChatPhotoUpload($groupPhotoFile);
            if ($photo_url) {
                // Opcional: borrar foto antigua
                $old_photo = $db['chat_conversations'][$group_key]['group_photo_url'];
                if ($old_photo && file_exists(__DIR__ . '/../' . $old_photo)) {
                    unlink(__DIR__ . '/../' . $old_photo);
                }
                $db['chat_conversations'][$group_key]['group_photo_url'] = $photo_url;
            }
        }

        // Actualizar miembros (una forma simple es borrar los existentes y añadir los nuevos)
        $db['chat_conversation_members'] = array_filter($db['chat_conversation_members'], function($m) use ($groupId) {
            return $m['conversation_id'] != $groupId;
        });
        $all_member_ids = array_unique(array_merge([$currentUser], $memberIds));
        foreach ($all_member_ids as $user_id) {
            $db['chat_conversation_members'][] = ['conversation_id' => $groupId, 'user_id' => (int)$user_id];
        }

        writeDatabase($db);
        echo json_encode(['conversation_id' => $groupId]);
        break;

    case 'delete_group':
        $groupId = (int)($_GET['id'] ?? 0);
        $db = readDatabase();
        $group_key = array_search($groupId, array_column($db['chat_conversations'], 'id'));

        if ($group_key === false || $db['chat_conversations'][$group_key]['created_by'] != $currentUser) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes permiso para eliminar este grupo.']);
            break;
        }

        // Eliminar conversación, miembros y mensajes asociados
        $db['chat_conversations'] = array_filter($db['chat_conversations'], fn($c) => $c['id'] != $groupId);
        $db['chat_conversation_members'] = array_filter($db['chat_conversation_members'], fn($m) => $m['conversation_id'] != $groupId);
        $db['chat_messages'] = array_filter($db['chat_messages'], fn($msg) => $msg['conversation_id'] != $groupId);

        writeDatabase($db);
        echo json_encode(['success' => true]);
        break;

    case 'edit_message':
        $input = json_decode(file_get_contents('php://input'), true);
        $messageId = $input['message_id'] ?? 0;
        $content = trim($input['content'] ?? '');
        $db = readDatabase();
        $msg_key = array_search($messageId, array_column($db['chat_messages'], 'id'));

        if ($msg_key !== false && $db['chat_messages'][$msg_key]['sender_id'] == $currentUser && !empty($content)) {
            $db['chat_messages'][$msg_key]['message_content'] = $content;
            writeDatabase($db);
            echo json_encode(['success' => true]);
        } else { echo json_encode(['success' => false, 'error' => 'No se pudo editar.']); }
        break;

    case 'delete_message':
        $messageId = (int)($_GET['id'] ?? 0);
        $db = readDatabase();
        $msg_key = array_search($messageId, array_column($db['chat_messages'], 'id'));
        if ($msg_key !== false && $db['chat_messages'][$msg_key]['sender_id'] == $currentUser) {
            array_splice($db['chat_messages'], $msg_key, 1);
            writeDatabase($db);
            echo json_encode(['success' => true]);
        } else { echo json_encode(['success' => false, 'error' => 'No se pudo eliminar.']); }
        break;

    default:
        http_response_code(400);
        echo json_encode(['error' => 'Acción no válida o no especificada.']);
        break;
}
?>