<?php
header('Content-Type: application/json');

// Incluir la configuración de la base de datos para obtener los ajustes
require_once '../config/database.php';

// --- Función para obtener la configuración de la IA ---
function get_ai_settings($pdo) {
    try {
        $stmt = $pdo->query("SELECT value FROM settings WHERE `key` = 'ai_settings'");
        $settings_json = $stmt->fetchColumn();
        if ($settings_json) {
            return json_decode($settings_json, true);
        }
        return null;
    } catch (PDOException $e) {
        return null;
    }
}

// --- Función para llamar a la API de Gemini ---
function call_gemini_api($api_key, $model, $parts) {
    $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$api_key}";

    $data = [
        'contents' => [
            [
                'parts' => $parts
            ]
        ],
        'generationConfig' => [
            'temperature' => 0.5,
            'topK' => 1,
            'topP' => 1,
            'maxOutputTokens' => 8192,
            'stopSequences' => []
        ],
        'safetySettings' => [
            ['category' => 'HARM_CATEGORY_HARASSMENT', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'],
            ['category' => 'HARM_CATEGORY_HATE_SPEECH', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'],
            ['category' => 'HARM_CATEGORY_SEXUALLY_EXPLICIT', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'],
            ['category' => 'HARM_CATEGORY_DANGEROUS_CONTENT', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE']
        ]
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Desactivar en desarrollo si hay problemas de SSL

    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
        return ['success' => false, 'message' => 'Error de cURL: ' . $error];
    }

    $result = json_decode($response, true);

    if (isset($result['candidates'][0]['content']['parts'][0]['text'])) {
        return ['success' => true, 'html' => $result['candidates'][0]['content']['parts'][0]['text']];
    } else {
        return ['success' => false, 'message' => 'No se pudo generar el contenido. Respuesta de la API: ' . json_encode($result)];
    }
}


// --- INICIO DEL SCRIPT PRINCIPAL ---

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Método no permitido.']);
    exit;
}

// Obtener configuración de la IA
$pdo = get_db_connection();
$ai_settings = get_ai_settings($pdo);

if (!$ai_settings || empty($ai_settings['api_key']) || empty($ai_settings['ai_model'])) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'La configuración de la IA (API Key y Modelo) no está completa. Por favor, configúrela en la sección de Ajustes.']);
    exit;
}

// Recopilar datos de la solicitud
$user_prompt = isset($_POST['prompt']) ? trim($_POST['prompt']) : '';
$files = isset($_FILES['data_files']) ? $_FILES['data_files'] : [];

if (empty($user_prompt) && empty($files['name'][0])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Debe proporcionar un prompt o al menos un archivo.']);
    exit;
}

// Construir las partes de la solicitud para la API de Gemini
$request_parts = [];

// 1. El prompt del sistema
$system_prompt = '''
Eres un analista de datos experto trabajando para una agencia. Tu tarea es analizar los datos y el prompt proporcionado por el usuario para generar un reporte.
El reporte final debe ser un **código HTML interactivo y visualmente atractivo**, utilizando exclusivamente **Bootstrap 5** para los estilos y componentes (como tablas, tarjetas, gráficos, etc.).
NO incluyas las etiquetas `<html>`, `<head>`, o `<body>`. El resultado debe ser solo el contenido que se pueda insertar directamente dentro de un `div` en una página existente.
Si el usuario pide gráficos, usa una librería de gráficos de JavaScript como Chart.js, pero asegúrate de incluir el tag `<script>` necesario y el código de inicialización del gráfico dentro del HTML que generes.
Analiza la solicitud del usuario y los datos de los archivos adjuntos para crear el reporte más útil y claro posible.
''';

// 2. El prompt del usuario
$full_user_prompt = "Solicitud del usuario: \n" . $user_prompt;

// 3. Procesar archivos adjuntos
$file_contents_prompt = "\n\nDatos de los archivos adjuntos:\n";
$has_files = false;

if (!empty($files['name'][0])) {
    $has_files = true;
    for ($i = 0; $i < count($files['name']); $i++) {
        $tmp_name = $files['tmp_name'][$i];
        $mime_type = $files['type'][$i];
        $file_name = $files['name'][$i];

        // Solo procesar archivos que se subieron correctamente
        if (file_exists($tmp_name) && filesize($tmp_name) > 0) {
            $file_data = base64_encode(file_get_contents($tmp_name));
            $request_parts[] = [
                'inline_data' => [
                    'mimeType' => $mime_type,
                    'data' => $file_data
                ]
            ];
            $file_contents_prompt .= "- Archivo '{$file_name}' (tipo: {$mime_type})\n";
        }
    }
}

// Combinar los prompts
$final_prompt = $system_prompt . $full_user_prompt . ($has_files ? $file_contents_prompt : "");
$request_parts[] = ['text' => $final_prompt];


// Llamar a la API de Gemini
$api_result = call_gemini_api($ai_settings['api_key'], $ai_settings['ai_model'], $request_parts);

if ($api_result['success']) {
    echo json_encode([
        'success' => true,
        'message' => 'Análisis completado con éxito.',
        'data' => [
            'report_html' => $api_result['html']
        ]
    ]);
} else {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $api_result['message']
    ]);
}

?>