<?php
header('Content-Type: application/json');

function slugify($text) {
    // Reemplaza los caracteres no alfanuméricos con guiones
    $text = preg_replace('~[^\pL\d]+~u', '-', $text);
    // Translitera
    $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
    // Elimina caracteres no deseados
    $text = preg_replace('~[^-\w]+~', '', $text);
    // Elimina guiones al principio y al final
    $text = trim($text, '-');
    // Elimina guiones duplicados
    $text = preg_replace('~-+~', '-', $text);
    // Pasa a minúsculas
    $text = strtolower($text);

    if (empty($text)) {
        return 'n-a';
    }

    return $text;
}

function send_json_response($success, $message, $data = []) {
    $response = ['success' => $success, 'message' => $message];
    if (!empty($data)) {
        $response = array_merge($response, $data);
    }
    echo json_encode($response);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    send_json_response(false, 'Método no permitido.');
}

$title = isset($_POST['title']) ? trim($_POST['title']) : '';
$postId = isset($_POST['postId']) ? trim($_POST['postId']) : null;
if (empty($title)) {
    send_json_response(false, 'El título es obligatorio.');
}

// --- Directorios de subida ---
$upload_dir_images = 'uploads/images/';
$upload_dir_attachments = 'uploads/attachments/';

// Crear directorios si no existen
if (!is_dir($upload_dir_images)) mkdir($upload_dir_images, 0777, true);
if (!is_dir($upload_dir_attachments)) mkdir($upload_dir_attachments, 0777, true);

$image_path = null;
$attachment_path = null;

// --- Procesar imagen principal ---
if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
    $image_name = uniqid('img_', true) . '_' . basename($_FILES['image']['name']);
    $target_file = $upload_dir_images . $image_name;
    if (move_uploaded_file($_FILES['image']['tmp_name'], $target_file)) {
        $image_path = $target_file;
    } else {
        send_json_response(false, 'Error al mover la imagen subida.');
    }
}

// --- Procesar archivo adjunto ---
if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
    $attachment_name = uniqid('att_', true) . '_' . basename($_FILES['attachment']['name']);
    $target_file = $upload_dir_attachments . $attachment_name;
    if (move_uploaded_file($_FILES['attachment']['tmp_name'], $target_file)) {
        $attachment_path = $target_file;
    } else {
        send_json_response(false, 'Error al mover el archivo adjunto.');
    }
}


// --- Guardar en el archivo JSON ---
$posts_file = 'blog_posts.json';
$posts = [];

if (file_exists($posts_file)) {
    $json_content = file_get_contents($posts_file);
    // Si el archivo está vacío o corrupto, inicializamos como array vacío
    $decoded_posts = json_decode($json_content, true);
    if (is_array($decoded_posts)) {
        $posts = $decoded_posts;
    }
}

if ($postId) {
    // --- Actualizar un post existente ---
    $post_found = false;
    foreach ($posts as $index => &$post) {
        if ($post['id'] === $postId) {
            $post['title'] = $title;
            $post['subtitle'] = isset($_POST['subtitle']) ? $_POST['subtitle'] : '';
            $post['content'] = isset($_POST['content']) ? $_POST['content'] : '';

            // Manejar la eliminación de la imagen
            if (isset($_POST['delete_image']) && $_POST['delete_image'] === 'on') {
                if (!empty($post['image']) && file_exists($post['image'])) {
                    unlink($post['image']);
                }
                $post['image'] = null; // Eliminar la referencia
            }

            // Manejar la eliminación del adjunto
            if (isset($_POST['delete_attachment']) && $_POST['delete_attachment'] === 'on') {
                if (!empty($post['attachment']) && file_exists($post['attachment'])) {
                    unlink($post['attachment']);
                }
                $post['attachment'] = null; // Eliminar la referencia
            }

            // Solo actualiza la imagen/adjunto si se sube uno nuevo
            if ($image_path) $post['image'] = $image_path;
            if ($attachment_path) $post['attachment'] = $attachment_path;
            
            $post_found = true;
            $saved_post = $post;
            break;
        }
    }

    if ($post_found) {
        if (file_put_contents($posts_file, json_encode($posts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
            send_json_response(true, 'Artículo actualizado correctamente.', ['post' => $saved_post]);
        } else {
            send_json_response(false, 'Error al escribir en el archivo JSON.');
        }
    } else {
        send_json_response(false, 'No se encontró el artículo para actualizar.');
    }
} else {
    // --- Crear un nuevo post ---
    $new_post = [
        'id' => uniqid('post_', true),
        'slug' => slugify($title) . '-' . substr(uniqid(), -5), // Slug único
        'title' => $title,
        'subtitle' => isset($_POST['subtitle']) ? $_POST['subtitle'] : '',
        'content' => isset($_POST['content']) ? $_POST['content'] : '',
        'image' => $image_path,
        'attachment' => $attachment_path,
        'createdAt' => date('Y-m-d H:i:s')
    ];
    array_unshift($posts, $new_post);
    if (file_put_contents($posts_file, json_encode($posts, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
        send_json_response(true, 'Artículo guardado correctamente.', ['post' => $new_post]);
    } else {
        send_json_response(false, 'Error al escribir en el archivo JSON.');
    }
}
?>

?>