<?php
// Iniciar la sesin
session_start();

// Verificar si el usuario est autenticado
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    // Si no est autenticado, redirigir al login
    header("Location: login.html");
    exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panel de Administración - Contactos DIFSA</title>
    <!-- Icono de la página (Favicon) -->
    <link rel="icon" type="image/png" href="images/iso.png">
    <!-- Carga de Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Carga de Chart.js para gráficos -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <!-- Carga de Font Awesome para iconos -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" />
    <style>
        :root {
            --color-primary: #05331d;
            --color-secondary: #b07825;
            --color-light-green: #21472e;
        }
        body {
            font-family: 'Inter', sans-serif;
        }
        .text-primary { color: var(--color-primary); }
        .bg-primary { background-color: var(--color-primary); }
        .text-secondary { color: var(--color-secondary); }

        /* Estilo para enlaces del menú lateral */
        .sidebar-link {
            display: flex;
            align-items: center;
            padding: 0.75rem 1.5rem;
            color: #d1d5db; /* gray-300 */
            transition: background-color 0.2s, color 0.2s;
            border-left: 4px solid transparent;
        }
        .sidebar-link:hover {
            background-color: var(--color-light-green);
            color: white;
        }
        .sidebar-link.active {
            background-color: var(--color-light-green);
            color: white;
            font-weight: 600;
            border-left-color: var(--color-secondary);
        }
    </style>
</head>
<body class="bg-gray-100">

    <div class="flex h-screen">
        <!-- Sidebar -->
        <aside class="w-64 bg-primary text-white flex flex-col">
            <div class="h-20 flex items-center justify-center border-b border-light-green">
                <img src="images/difsacompleto.png" alt="Logo DIFSA" class="h-10">
            </div>
            <nav class="flex-1 mt-4">
                <a href="#dashboard" class="sidebar-link active" onclick="showPage('dashboard')"><i class="fas fa-tachometer-alt w-6 mr-2"></i> Dashboard</a>
                <a href="#contactos" class="sidebar-link" onclick="showPage('contactos')"><i class="fas fa-envelope-open-text w-6 mr-2"></i> Contactos</a>
                <a href="#blog" class="sidebar-link" onclick="showPage('blog')"><i class="fas fa-newspaper w-6 mr-2"></i> Blog / Recursos</a>
                <a href="#descargas" class="sidebar-link" onclick="showPage('descargas')"><i class="fas fa-download w-6 mr-2"></i> Descargas</a>
                <a href="#configuracion" class="sidebar-link" onclick="showPage('configuracion')"><i class="fas fa-cogs w-6 mr-2"></i> Configuración</a>
            </nav>
            <div class="p-4 border-t border-light-green space-y-2">
                <a href="logout.php" class="block text-sm text-gray-300 hover:text-white"><i class="fas fa-sign-out-alt mr-2"></i>Cerrar Sesin</a>
                <a href="index.html" class="text-sm text-gray-300 hover:text-white">Volver al Sitio Principal &rarr;</a>
            </div>
        </aside>

        <!-- Main Content -->
        <div class="flex-1 flex flex-col overflow-hidden">
            <header class="bg-white shadow-sm p-4">
                <h1 id="page-title" class="text-2xl font-bold text-primary">Dashboard</h1>
            </header>

            <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-100 p-4 sm:p-6 lg:p-8">
                
                <!-- === PÁGINA: DASHBOARD === -->
                <section id="page-dashboard" class="page-content space-y-8">
                    <!-- Tarjetas de Resumen -->
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
                        <div class="bg-white p-6 rounded-lg shadow-md flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-500">Total de Artículos</p>
                                <p id="total-articles" class="text-3xl font-bold text-primary">0</p>
                            </div>
                            <i class="fas fa-newspaper text-4xl text-gray-300"></i>
                        </div>
                        <div class="bg-white p-6 rounded-lg shadow-md flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-500">Total de Contactos</p>
                                <p id="total-contacts" class="text-3xl font-bold text-primary">0</p>
                            </div>
                            <i class="fas fa-envelope-open-text text-4xl text-gray-300"></i>
                        </div>
                        <div class="bg-white p-6 rounded-lg shadow-md flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-500">Total de Descargas</p>
                                <p id="total-downloads" class="text-3xl font-bold text-primary">0</p>
                            </div>
                            <i class="fas fa-download text-4xl text-gray-300"></i>
                        </div>
                        <div class="bg-white p-6 rounded-lg shadow-md flex items-center justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-500">Visitas Hoy</p>
                                <p id="today-visits" class="text-3xl font-bold text-primary">0</p>
                            </div>
                            <i class="fas fa-chart-bar text-4xl text-gray-300"></i>
                        </div>
                    </div>

                    <!-- Gráfico de Visitas -->
                    <div class="bg-white p-6 rounded-lg shadow-md">
                        <div class="flex flex-wrap justify-between items-center mb-4 gap-4">
                            <h3 class="text-lg font-semibold text-gray-700">Visitas del Sitio Web</h3>
                            <div class="flex items-center gap-2 flex-wrap">
                                <button onclick="renderVisitsChart('day')" class="chart-filter-btn px-3 py-1 text-sm bg-gray-200 rounded-md hover:bg-secondary hover:text-white">Hoy</button>
                                <button onclick="renderVisitsChart('week')" class="chart-filter-btn px-3 py-1 text-sm bg-secondary text-white rounded-md hover:bg-accent">Semana</button>
                                <button onclick="renderVisitsChart('month')" class="chart-filter-btn px-3 py-1 text-sm bg-gray-200 rounded-md hover:bg-secondary hover:text-white">Mes</button>
                                <button onclick="renderVisitsChart('year')" class="chart-filter-btn px-3 py-1 text-sm bg-gray-200 rounded-md hover:bg-secondary hover:text-white">Año</button>
                            </div>
                        </div>
                        <canvas id="visits-chart" class="mb-4"></canvas>
                        <form id="custom-range-form" class="flex items-center gap-2 text-sm">
                            <label for="start-date">Desde:</label>
                            <input type="date" id="start-date" class="p-1 border rounded-md">
                            <label for="end-date">Hasta:</label>
                            <input type="date" id="end-date" class="p-1 border rounded-md">
                            <button type="submit" class="px-3 py-1 bg-primary text-white rounded-md hover:bg-light-green">Aplicar</button>
                        </form>
                    </div>

                    <!-- Tabla de Actividad Reciente -->
                    <div id="recent-activity-container" class="bg-white shadow-md rounded-lg p-6">
                        <!-- La tabla se generará aquí -->
                    </div>
                </section>

                <!-- === PÁGINA: CONTACTOS === -->
                <section id="page-contactos" class="page-content hidden">
                    <!-- El contenido se cargará aquí dinámicamente -->
                </section>

                <!-- === PÁGINA: BLOG / RECURSOS === -->
                <section id="page-blog" class="page-content hidden">
                    <div class="bg-white shadow-md rounded-lg p-6">
                        <h2 class="text-xl font-semibold text-gray-800 mb-4">Crear Nuevo Artículo</h2>
                        <form id="blog-post-form" class="space-y-4" enctype="multipart/form-data">
                            <div>
                                <label for="blog-title" class="block text-sm font-medium text-gray-700">Título del Artículo</label>
                                <input type="text" id="blog-title" name="title" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm p-2 border" required>
                            </div>
                            <div>
                                <label for="blog-subtitle" class="block text-sm font-medium text-gray-700">Subtítulo</label>
                                <input type="text" id="blog-subtitle" name="subtitle" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm p-2 border">
                            </div>
                            <div>
                                <label for="blog-image" class="block text-sm font-medium text-gray-700">Imagen Principal</label>
                                <div id="current-image-container" class="hidden mt-2">
                                    <img id="current-image-preview" src="" alt="Imagen actual" class="h-20 w-auto rounded-md mb-2">
                                    <div class="flex items-center">
                                        <input type="checkbox" id="delete-image" name="delete_image" class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                        <label for="delete-image" class="ml-2 font-medium text-red-600">Eliminar imagen actual al guardar</label>
                                    </div>
                                </div>
                                <input type="file" id="blog-image" name="image" accept="image/*" class="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-secondary file:text-white hover:file:bg-accent">
                            </div>
                            <div>
                                <label for="blog-content" class="block text-sm font-medium text-gray-700">Contenido del Artículo</label>
                                <textarea id="blog-content" name="content" rows="15" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm p-2 border" placeholder="Escribe el contenido de tu artículo aquí. Puedes usar etiquetas HTML como <b>, <ul>, <li>, <h1>, <h2>, <h3>, etc."></textarea>
                                <p class="mt-1 text-xs text-gray-500">Soporta etiquetas HTML básicas para formato de texto.</p>
                            </div>
                            <div>
                                <label for="blog-attachment" class="block text-sm font-medium text-gray-700">Adjuntar Documento (Opcional)</label>
                                <input type="file" id="blog-attachment" name="attachment" class="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-secondary file:text-white hover:file:bg-accent">
                                <p class="mt-1 text-xs text-gray-500">Puedes adjuntar un archivo PDF, DOCX, etc., para descarga.</p>
                            </div>
                            <div id="current-attachment-container" class="hidden mt-2 text-sm">
                                <p class="text-gray-600">Adjunto actual: <a id="current-attachment-link" href="#" target="_blank" class="text-blue-600 hover:underline"></a></p>
                                <div class="flex items-center mt-1">
                                    <input type="checkbox" id="delete-attachment" name="delete_attachment" class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                    <label for="delete-attachment" class="ml-2 font-medium text-red-600">Eliminar adjunto actual al guardar</label>
                                </div>
                            </div>
                            <!-- Campo oculto para el ID del post a editar -->
                            <input type="hidden" id="blog-post-id" name="postId">
                            
                            <div>
                                <button type="submit" id="blog-submit-button" class="px-4 py-2 bg-primary text-white rounded-md hover:bg-secondary">Guardar Artículo</button>
                                <button type="button" id="blog-cancel-edit" class="hidden px-4 py-2 bg-gray-300 text-gray-800 rounded-md hover:bg-gray-400">Cancelar Edición</button>
                            </div>
                        </form>
                    </div>

                    <!-- Lista de Artículos Existentes -->
                    <div class="mt-8 bg-white shadow-md rounded-lg p-6">
                        <h2 class="text-xl font-semibold text-gray-800 mb-4">Artículos Existentes</h2>
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Título</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Fecha de Creación</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody id="blog-posts-list" class="bg-white divide-y divide-gray-200">
                                    <!-- Las filas de los artículos se cargarán aquí -->
                                </tbody>
                            </table>
                        </form>
                    </div>
                </section>

                <!-- === PÁGINA: DESCARGAS === -->
                <section id="page-descargas" class="page-content hidden">
                     <div class="bg-white shadow-md rounded-lg overflow-hidden">
                        <div class="px-6 py-4 border-b">
                            <h2 class="text-xl font-semibold text-gray-800">Registro de Descargas</h2>
                            <p class="text-gray-600 text-sm mt-1">Usuarios que han descargado recursos desde el blog.</p>
                        </div>
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Fecha</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nombre</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Recurso Descargado</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white divide-y divide-gray-200">
                                    <!-- Filas de ejemplo -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </section>

                <!-- === PÁGINA: USUARIOS === -->
                <section id="page-usuarios" class="page-content hidden"></section>

                <!-- === PÁGINA: CONFIGURACIÓN === -->
                <section id="page-configuracion" class="page-content hidden">
                    <div class="bg-white shadow-md rounded-lg p-6">
                        <h2 class="text-xl font-semibold text-gray-800 mb-4">Parametrización del Sitio</h2>
                        <form id="config-form" class="space-y-6">
                            <div>
                                <label for="google_analytics_key" class="block text-sm font-medium text-gray-700">Google Analytics Key</label>
                                <input type="text" id="google_analytics_key" name="google_analytics_key" class="mt-1 block w-full md:w-1/2 rounded-md border-gray-300 shadow-sm p-2 border" placeholder="G-XXXXXXXXXX o UA-XXXXX-Y">
                            </div>
                            <div>
                                <label for="recaptcha_site_key" class="block text-sm font-medium text-gray-700">Google reCAPTCHA v3 Site Key</label>
                                <input type="text" id="recaptcha_site_key" name="recaptcha_site_key" class="mt-1 block w-full md:w-1/2 rounded-md border-gray-300 shadow-sm p-2 border">
                            </div>
                            <button type="submit" class="px-4 py-2 bg-primary text-white rounded-md hover:bg-secondary">Guardar Cambios</button>
                        </form>
                    </div>
                </section>

            </main>
        </div>
    </div>

    <script>
        const pageTitle = document.getElementById('page-title');

        function showPage(pageId) {
            // Ocultar todas las páginas
            document.querySelectorAll('.page-content').forEach(page => {
                page.classList.add('hidden');
            });

            // Mostrar la página seleccionada
            const activePage = document.getElementById('page-' + pageId);
            if (activePage) {
                activePage.classList.remove('hidden');
            }

            // Actualizar el título principal
            const activeLink = document.querySelector(`.sidebar-link[href="#${pageId}"]`);
            if (activeLink) {
                pageTitle.textContent = activeLink.textContent;
            }

            // Actualizar la clase 'active' en el menú
            document.querySelectorAll('.sidebar-link').forEach(link => {
                link.classList.remove('active');
            });
            if(activeLink) {
                activeLink.classList.add('active');
            }

            // Cargar contenido dinámico si es necesario
            if (pageId === 'dashboard') {
                document.getElementById('page-dashboard').classList.remove('hidden');
                loadDashboardData();
            }

            if (pageId === 'contactos') {
                loadContacts();
            }

            // Asegurarse de que la página del blog se muestre
            if (pageId === 'blog') {
                document.getElementById('page-blog').classList.remove('hidden');
                loadBlogPosts();
            }

            if (pageId === 'descargas') {
                document.getElementById('page-descargas').classList.remove('hidden');
                loadDownloads();
            }

            if (pageId === 'configuracion') {
                document.getElementById('page-configuracion').classList.remove('hidden');
                loadConfig();
            }

            if (pageId === 'contactos') {
                document.getElementById('page-contactos').classList.remove('hidden');
            }
        }

        let currentContacts = []; // Para almacenar los contactos cargados y usarlos en las funciones de acción.

        async function loadContacts() {
            const contactsSection = document.getElementById('page-contactos');
            contactsSection.innerHTML = '<p class="text-center">Cargando contactos...</p>'; // Mensaje de carga

            try {
                const response = await fetch('contacts.json?v=' + new Date().getTime()); // Cache-busting
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                currentContacts = await response.json(); // Almacenar los contactos

                if (!currentContacts || currentContacts.length === 0) {
                    contactsSection.innerHTML = '<p class="text-center text-gray-500">No hay consultas de contacto para mostrar.</p>';
                    return;
                }

                // Construir la tabla
                const tableHtml = `
                    <div class="bg-white shadow-md rounded-lg overflow-hidden">
                        <div class="px-6 py-4 border-b">
                            <h2 class="text-xl font-semibold text-gray-800">Consultas Recibidas</h2>
                            <p class="text-gray-600 text-sm mt-1">Mensajes enviados desde el formulario de contacto.</p>
                        </div>
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Fecha</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nombre</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Servicios de Interés</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Mensaje</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white divide-y divide-gray-200">
                                    ${currentContacts.map((contact, index) => `
                                        <tr>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${contact.date || ''}</td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">${contact.name || ''}</td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${contact.email || ''}</td>
                                            <td class="px-6 py-4 text-sm text-gray-500">
                                                ${(contact.services && contact.services.length > 0)
                                                    ? contact.services.map(service => `<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800 mr-1">${service}</span>`).join('')
                                                    : 'Ninguno'
                                                }
                                            </td>
                                            <td class="px-6 py-4 text-sm text-gray-500 max-w-sm truncate" title="${contact.message || ''}">${contact.message || ''}</td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                <button onclick="viewContact(${index})" class="text-indigo-600 hover:text-indigo-900 mr-3">Ver</button>
                                                <button onclick="deleteContact(${index})" class="text-red-600 hover:text-red-900">Eliminar</button>
                                            </td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    </div>
                `;
                contactsSection.innerHTML = tableHtml;

            } catch (error) {
                console.error('Error al cargar los contactos:', error);
                contactsSection.innerHTML = '<p class="text-center text-red-500">Error al cargar los contactos. Revisa la consola para más detalles.</p>';
            }
        }

        function viewContact(index) {
            const contact = currentContacts[index];
            if (contact) {
                let servicesList = (contact.services && contact.services.length > 0) ? contact.services.join(', ') : 'Ninguno';
                alert(
                    `Detalles del Contacto:\n\n` +
                    `Fecha: ${contact.date || 'N/A'}\n` +
                    `Nombre: ${contact.name || 'N/A'}\n` +
                    `Email: ${contact.email || 'N/A'}\n` +
                    `Servicios de Interés: ${servicesList}\n` +
                    `Mensaje: ${contact.message || 'N/A'}`
                );
            }
        }

        async function deleteContact(index) {
            if (!confirm('¿Estás seguro de que quieres eliminar este contacto?')) {
                return;
            }

            try {
                const response = await fetch('delete_contact.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `index=${index}`
                });

                const result = await response.json();

                if (result.success) {
                    alert('Contacto eliminado con éxito.');
                    loadContacts(); // Recargar la tabla
                } else {
                    alert('Error al eliminar el contacto: ' + (result.message || 'Error desconocido.'));
                }
            } catch (error) {
                console.error('Error al eliminar el contacto:', error);
                alert('Error de conexión al intentar eliminar el contacto.');
            }
        }

        async function loadDashboardData() {
            // Función para obtener el conteo de un archivo JSON
            const getCount = async (file) => {
                try {
                    const response = await fetch(`${file}?v=${new Date().getTime()}`);
                    if (!response.ok) return 0;
                    const data = await response.json();
                    return Array.isArray(data) ? data.length : 0;
                } catch {
                    return 0;
                }
            };

            // Actualizar tarjetas
            document.getElementById('total-articles').textContent = await getCount('blog_posts.json');
            document.getElementById('total-contacts').textContent = await getCount('contacts.json');
            document.getElementById('total-downloads').textContent = await getCount('downloads.json');

            // Procesar visitas
            try {
                const response = await fetch(`visits.json?v=${new Date().getTime()}`);
                allVisits = await response.json();
                
                // Renderizar gráfico por defecto (semana)
                renderVisitsChart('week');

            } catch (error) {
                console.error("Error al procesar datos de visitas:", error);
                document.getElementById('today-visits').textContent = 'Error';
            }

            // Cargar actividad reciente
            const contacts = await (getCount('contacts.json') > 0 ? fetch('contacts.json').then(res => res.json()) : Promise.resolve([]));
            const downloads = await (getCount('downloads.json') > 0 ? fetch('downloads.json').then(res => res.json()) : Promise.resolve([]));

            const recentActivity = [
                ...contacts.map(c => ({ ...c, type: 'Contacto' })),
                ...downloads.map(d => ({ ...d, type: 'Descarga' }))
            ].sort((a, b) => new Date(b.date) - new Date(a.date)).slice(0, 5);

            const activityContainer = document.getElementById('recent-activity-container');
            if (recentActivity.length > 0) {
                activityContainer.innerHTML = `
                    <h3 class="text-lg font-semibold text-gray-700 mb-4">Actividad Reciente</h3>
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Tipo</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Nombre / Email</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Detalle</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Fecha</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                ${recentActivity.map(item => `
                                    <tr>
                                        <td class="px-6 py-4"><span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${item.type === 'Contacto' ? 'bg-blue-100 text-blue-800' : 'bg-green-100 text-green-800'}">${item.type}</span></td>
                                        <td class="px-6 py-4 text-sm text-gray-900">${item.name}<br><span class="text-xs text-gray-500">${item.email}</span></td>
                                        <td class="px-6 py-4 text-sm text-gray-500">${item.type === 'Contacto' ? (item.services.join(', ') || 'Consulta general') : item.resource}</td>
                                        <td class="px-6 py-4 text-sm text-gray-500">${new Date(item.date).toLocaleString('es-ES')}</td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                `;
            } else {
                activityContainer.innerHTML = '<p class="text-center text-gray-500">No hay actividad reciente para mostrar.</p>';
            }
        }

        // === Lógica para el Dashboard ===
        let visitsChartInstance = null;
        let allVisits = [];

        function renderVisitsChart(period, startDateStr = null, endDateStr = null) {
            const today = new Date();
            const todayStr = today.toISOString().slice(0, 10);
            let labels = [];
            let data = [];
            let filteredVisits = [];

            document.getElementById('today-visits').textContent = allVisits.filter(v => v.timestamp.startsWith(todayStr)).length;

            // Actualizar estilo de botones de filtro
            document.querySelectorAll('.chart-filter-btn').forEach(btn => {
                if (btn.textContent.toLowerCase() === period) {
                    btn.classList.add('bg-secondary', 'text-white');
                    btn.classList.remove('bg-gray-200');
                } else {
                    btn.classList.remove('bg-secondary', 'text-white');
                    btn.classList.add('bg-gray-200');
                }
            });

            switch (period) {
                case 'day':
                    filteredVisits = allVisits.filter(v => v.timestamp.startsWith(todayStr));
                    const visitsByHour = new Array(24).fill(0);
                    filteredVisits.forEach(v => {
                        const hour = new Date(v.timestamp).getHours();
                        visitsByHour[hour]++;
                    });
                    labels = Array.from({ length: 24 }, (_, i) => `${i.toString().padStart(2, '0')}:00`);
                    data = visitsByHour;
                    break;

                case 'month':
                    const startOfMonth = new Date(today.getFullYear(), today.getMonth(), 1);
                    filteredVisits = allVisits.filter(v => new Date(v.timestamp) >= startOfMonth);
                    const daysInMonth = today.getDate();
                    const visitsByDayOfMonth = new Array(daysInMonth).fill(0);
                    filteredVisits.forEach(v => {
                        const day = new Date(v.timestamp).getDate() - 1;
                        if (day < daysInMonth) visitsByDayOfMonth[day]++;
                    });
                    labels = Array.from({ length: daysInMonth }, (_, i) => i + 1);
                    data = visitsByDayOfMonth;
                    break;

                case 'year':
                    const startOfYear = new Date(today.getFullYear(), 0, 1);
                    filteredVisits = allVisits.filter(v => new Date(v.timestamp) >= startOfYear);
                    const visitsByMonth = new Array(12).fill(0);
                    filteredVisits.forEach(v => {
                        const month = new Date(v.timestamp).getMonth();
                        visitsByMonth[month]++;
                    });
                    labels = ['Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun', 'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'];
                    data = visitsByMonth;
                    break;
                
                case 'custom':
                    const startDate = new Date(startDateStr + 'T00:00:00');
                    const endDate = new Date(endDateStr + 'T23:59:59');
                    filteredVisits = allVisits.filter(v => {
                        const visitDate = new Date(v.timestamp);
                        return visitDate >= startDate && visitDate <= endDate;
                    });
                    const diffTime = Math.abs(endDate - startDate);
                    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

                    const visitsByDay = {};
                    for (let i = 0; i < diffDays; i++) {
                        const d = new Date(startDate);
                        d.setDate(d.getDate() + i);
                        visitsByDay[d.toISOString().slice(0, 10)] = 0;
                    }
                    filteredVisits.forEach(v => {
                        const day = v.timestamp.slice(0, 10);
                        if (visitsByDay.hasOwnProperty(day)) visitsByDay[day]++;
                    });
                    labels = Object.keys(visitsByDay).map(d => new Date(d).toLocaleDateString('es-ES', { day: 'numeric', month: 'short' }));
                    data = Object.values(visitsByDay);
                    break;

                case 'week':
                default:
                    const last7Days = {};
                    for (let i = 6; i >= 0; i--) {
                        const d = new Date();
                        d.setDate(d.getDate() - i);
                        last7Days[d.toISOString().slice(0, 10)] = 0;
                    }
                    allVisits.forEach(v => {
                        const visitDay = v.timestamp.slice(0, 10);
                        if (last7Days.hasOwnProperty(visitDay)) last7Days[visitDay]++;
                    });
                    labels = Object.keys(last7Days).map(day => new Date(day).toLocaleDateString('es-ES', { day: 'numeric', month: 'short' }));
                    data = Object.values(last7Days);
                    break;
            }

            const ctx = document.getElementById('visits-chart').getContext('2d');
            if (visitsChartInstance) {
                visitsChartInstance.destroy();
            }
            visitsChartInstance = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Visitas',
                        data: data,
                        borderColor: 'var(--color-secondary)',
                        backgroundColor: 'rgba(176, 120, 37, 0.1)',
                        fill: true,
                        tension: 0.3
                    }]
                },
                options: { responsive: true, scales: { y: { beginAtZero: true, ticks: { stepSize: 1 } } } }
            });
        }

        document.getElementById('custom-range-form').addEventListener('submit', (e) => {
            e.preventDefault();
            const startDate = document.getElementById('start-date').value;
            const endDate = document.getElementById('end-date').value;
            if (startDate && endDate) {
                renderVisitsChart('custom', startDate, endDate);
            }
        });

        async function loadConfig() {
            try {
                const response = await fetch('config.json?v=' + new Date().getTime());
                if (response.ok) {
                    const config = await response.json();
                    document.getElementById('google_analytics_key').value = config.google_analytics_key || '';
                    document.getElementById('recaptcha_site_key').value = config.recaptcha_site_key || '';
                    document.getElementById('recaptcha_secret_key').value = config.recaptcha_secret_key || '';
                }
            } catch (error) {
                console.error('No se pudo cargar el archivo de configuración, se usarán valores por defecto.', error);
            }
        }

        const configForm = document.getElementById('config-form');
        if (configForm) {
            configForm.addEventListener('submit', async (e) => {
                e.preventDefault();
                const formData = new FormData(configForm);

                try {
                    const response = await fetch('save_config.php', {
                        method: 'POST',
                        body: formData
                    });

                    const result = await response.json();

                    if (result.success) {
                        alert('Configuración guardada con éxito.');
                    } else {
                        alert('Error al guardar la configuración: ' + (result.message || 'Error desconocido.'));
                    }
                } catch (error) {
                    console.error('Error de conexión al guardar la configuración:', error);
                    alert('Error de conexión al intentar guardar la configuración.');
                }
            });
        }

        document.addEventListener('DOMContentLoaded', () => {
            // Mostrar la página inicial (dashboard)
            const initialPage = window.location.hash.substring(1) || 'dashboard';
            showPage(initialPage);

            // Manejar la navegación con hash
            window.onhashchange = () => {
                const hash = window.location.hash.substring(1);
                if (hash) {
                    showPage(hash);
                }
            };

            // === Lógica para el formulario del Blog ===
            const blogPostForm = document.getElementById('blog-post-form');
            if (blogPostForm) {
                blogPostForm.addEventListener('submit', async (event) => {
                    event.preventDefault();
                    const formData = new FormData(blogPostForm);
                    try {
                        const response = await fetch('save_blog_post.php', { method: 'POST', body: formData });
                        const result = await response.json();
                        if (result.success) {
                            alert(result.message);
                            document.getElementById('blog-cancel-edit').click();
                            loadBlogPosts();
                        } else {
                            alert('Error: ' + (result.message || 'Error desconocido.'));
                        }
                    } catch (error) {
                        console.error('Error al enviar el formulario del blog:', error);
                        alert('Error de conexión al intentar guardar el artículo.');
                    }
                });
            }
        });

        async function loadDownloads() {
            const downloadsBody = document.querySelector('#page-descargas tbody');
            if (!downloadsBody) return;
            downloadsBody.innerHTML = '<tr><td colspan="4" class="text-center p-4">Cargando registros...</td></tr>';

            try {
                const response = await fetch('downloads.json?v=' + new Date().getTime());
                if (!response.ok) {
                    if (response.status === 404) {
                         downloadsBody.innerHTML = '<tr><td colspan="4" class="text-center p-4 text-gray-500">Aún no hay registros de descargas.</td></tr>';
                         return;
                    }
                    throw new Error('Error al cargar el archivo.');
                }
                const currentDownloads = await response.json();

                if (!currentDownloads || currentDownloads.length === 0) {
                    downloadsBody.innerHTML = '<tr><td colspan="4" class="text-center p-4 text-gray-500">No hay registros de descargas para mostrar.</td></tr>';
                    return;
                }

                downloadsBody.innerHTML = currentDownloads.map((d, index) => `
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${new Date(d.date).toLocaleString('es-ES', { dateStyle: 'short', timeStyle: 'short' })}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">${d.name}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${d.email}</td>
                        <td class="px-6 py-4 text-sm text-gray-500">${d.resource}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <button onclick="deleteDownload(${index})" class="text-red-600 hover:text-red-900">Eliminar</button>
                        </td>
                    </tr>
                `).join('');

            } catch (error) {
                console.error('Error al cargar las descargas:', error);
                downloadsBody.innerHTML = '<tr><td colspan="4" class="text-center p-4 text-red-500">Error al cargar los registros.</td></tr>';
            }
        }

        async function deleteDownload(index) {
            if (!confirm('¿Estás seguro de que quieres eliminar este registro de descarga?')) {
                return;
            }

            try {
                const response = await fetch('delete_download.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `index=${index}`
                });
                const result = await response.json();
                if (result.success) {
                    alert('Registro de descarga eliminado con éxito.');
                    loadDownloads();
                } else {
                    alert('Error al eliminar el registro: ' + (result.message || 'Error desconocido.'));
                }
            } catch (error) {
                console.error('Error al eliminar el registro:', error);
                alert('Error de conexión al intentar eliminar el registro.');
            }
        }



        // === Lógica para la sección de Blog ===
        let allPosts = [];

        async function loadBlogPosts() {
            const postsList = document.getElementById('blog-posts-list');
            if (!postsList) return;
            postsList.innerHTML = '<tr><td colspan="3" class="text-center p-4">Cargando artículos...</td></tr>';

            try {
                const response = await fetch('blog_posts.json?v=' + new Date().getTime());
                allPosts = await response.json();

                if (!allPosts || allPosts.length === 0) {
                    postsList.innerHTML = '<tr><td colspan="3" class="text-center p-4 text-gray-500">No hay artículos para mostrar.</td></tr>';
                    return;
                }

                postsList.innerHTML = allPosts.map(post => `
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">${post.title}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${new Date(post.createdAt).toLocaleDateString('es-ES')}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <a href="recurso-detalle.html?slug=${post.slug}" target="_blank" class="text-blue-600 hover:text-blue-900 mr-3">Ver</a>
                            <button onclick="editBlogPost('${post.id}')" class="text-indigo-600 hover:text-indigo-900 mr-3">Editar</button>
                            <button onclick="deleteBlogPost('${post.id}')" class="text-red-600 hover:text-red-900">Eliminar</button>
                        </td>
                    </tr>
                `).join('');

            } catch (error) {
                console.error('Error al cargar los artículos:', error);
                postsList.innerHTML = '<tr><td colspan="3" class="text-center p-4 text-red-500">Error al cargar los artículos.</td></tr>';
            }
        }

        function editBlogPost(postId) {
            const post = allPosts.find(p => p.id === postId);
            if (!post) return;

            document.getElementById('blog-post-id').value = post.id;
            document.getElementById('blog-title').value = post.title;
            document.getElementById('blog-subtitle').value = post.subtitle;
            document.getElementById('blog-content').value = post.content;

            // Manejar la imagen actual
            const imageContainer = document.getElementById('current-image-container');
            const imagePreview = document.getElementById('current-image-preview');
            const deleteImageCheckbox = document.getElementById('delete-image');

            if (post.image) {
                imagePreview.src = post.image;
                imageContainer.classList.remove('hidden');
            } else {
                imageContainer.classList.add('hidden');
            }
            deleteImageCheckbox.checked = false;

            const attachmentContainer = document.getElementById('current-attachment-container');
            const attachmentLink = document.getElementById('current-attachment-link');
            const deleteCheckbox = document.getElementById('delete-attachment');

            if (post.attachment) {
                attachmentLink.href = post.attachment;
                attachmentLink.textContent = post.attachment.split('/').pop();
                attachmentContainer.classList.remove('hidden');
            } else {
                attachmentContainer.classList.add('hidden');
            }
            deleteCheckbox.checked = false;

            document.getElementById('blog-submit-button').textContent = 'Actualizar Artículo';
            document.getElementById('blog-cancel-edit').classList.remove('hidden');
            
            // Scroll hasta el formulario
            document.getElementById('blog-post-form').scrollIntoView({ behavior: 'smooth' });
        }

        document.getElementById('blog-cancel-edit')?.addEventListener('click', () => {
            const form = document.getElementById('blog-post-form');
            form.reset();
            document.getElementById('blog-post-id').value = '';
            document.getElementById('blog-submit-button').textContent = 'Guardar Artículo';
            document.getElementById('blog-cancel-edit').classList.add('hidden');
            document.getElementById('current-image-container').classList.add('hidden');
            document.getElementById('current-attachment-container').classList.add('hidden');
        });

        async function deleteBlogPost(postId) {
            if (!confirm('¿Estás seguro de que quieres eliminar este artículo? Esta acción no se puede deshacer.')) {
                return;
            }

            try {
                const response = await fetch('delete_blog_post.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `postId=${postId}`
                });

                const result = await response.json();

                if (result.success) {
                    alert('Artículo eliminado con éxito.');
                    loadBlogPosts(); // Recargar la lista
                } else {
                    alert('Error al eliminar el artículo: ' + (result.message || 'Error desconocido.'));
                }
            } catch (error) {
                console.error('Error al eliminar el artículo:', error);
                alert('Error de conexión al intentar eliminar el artículo.');
            }
        }

    </script>
</body>
</html>
