<?php
header('Content-Type: application/json');

$jsonDbFile = 'usuarios.json';

// --- GET: Listar todos los usuarios (sin contraseñas) ---
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $users = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];
    
    // Nunca devolver las contraseñas, ni siquiera hasheadas.
    $safeUsers = array_map(function($user) {
        unset($user['password_hash']);
        return $user;
    }, $users);

    echo json_encode($safeUsers);

// --- POST: Crear un nuevo usuario ---
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (empty($data['name']) || empty($data['username']) || empty($data['password'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Nombre, usuario y contraseña son obligatorios.']);
        exit;
    }

    $users = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];

    // Verificar si el usuario ya existe
    foreach ($users as $user) {
        if ($user['username'] === $data['username']) {
            http_response_code(409); // Conflict
            echo json_encode(['status' => 'error', 'message' => 'El nombre de usuario ya está en uso.']);
            exit;
        }
    }

    $newUser = [
        'id' => uniqid('user_'),
        'name' => htmlspecialchars($data['name']),
        'username' => htmlspecialchars($data['username']),
        'password_hash' => password_hash($data['password'], PASSWORD_DEFAULT) // Hashear la contraseña
    ];

    array_push($users, $newUser);
    file_put_contents($jsonDbFile, json_encode($users, JSON_PRETTY_PRINT));

    echo json_encode(['status' => 'success', 'message' => 'Usuario creado con éxito.']);

// --- DELETE: Eliminar un usuario ---
} elseif ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    $userIdToDelete = $data['id'] ?? null;

    if (!$userIdToDelete) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'No se proporcionó el ID del usuario.']);
        exit;
    }

    $users = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];
    $updatedUsers = array_values(array_filter($users, fn($user) => $user['id'] !== $userIdToDelete));
    file_put_contents($jsonDbFile, json_encode($updatedUsers, JSON_PRETTY_PRINT));

    echo json_encode(['status' => 'success', 'message' => 'Usuario eliminado.']);

// --- PUT: Actualizar un usuario existente ---
} elseif ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    $userIdToUpdate = $data['id'] ?? null;

    if (!$userIdToUpdate || empty($data['name'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'ID y Nombre son obligatorios.']);
        exit;
    }

    $users = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];
    $userFound = false;

    foreach ($users as $i => $user) {
        if ($user['id'] === $userIdToUpdate) {
            $users[$i]['name'] = htmlspecialchars($data['name']);
            // Actualizar la contraseña solo si se proporciona una nueva
            if (!empty($data['password'])) {
                $users[$i]['password_hash'] = password_hash($data['password'], PASSWORD_DEFAULT);
            }
            $userFound = true;
            break;
        }
    }

    file_put_contents($jsonDbFile, json_encode($users, JSON_PRETTY_PRINT));

    echo json_encode(['status' => 'success', 'message' => 'Usuario actualizado con éxito.']);
} else {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Método no permitido.']);
}
?>