<?php
// Establece la cabecera para devolver una respuesta en formato JSON.
header('Content-Type: application/json');

// Define la ruta al archivo que funcionará como base de datos.
$jsonDbFile = 'contactos.json';

// Verifica que la solicitud sea de tipo POST.
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Obtiene el cuerpo de la solicitud (que debería ser un JSON).
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    // --- Verificación de reCAPTCHA v3 ---
    $config = json_decode(file_get_contents('config.json'), true);
    $recaptchaSecret = $config['recaptcha_secret_key'] ?? '';

    if (!empty($recaptchaSecret)) {
        if (empty($data['g-recaptcha-response'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Verificación anti-spam fallida.']);
            exit;
        }

        $recaptcha_url = 'https://www.google.com/recaptcha/api/siteverify';
        $recaptcha_response = $data['g-recaptcha-response'];

        $recaptcha = file_get_contents($recaptcha_url . '?secret=' . $recaptchaSecret . '&response=' . $recaptcha_response);
        $recaptcha_data = json_decode($recaptcha);

        // Si la verificación falla o la puntuación es muy baja, rechaza la solicitud.
        if (!$recaptcha_data->success || $recaptcha_data->score < 0.5) {
            http_response_code(403); // Forbidden
            echo json_encode(['status' => 'error', 'message' => 'La solicitud parece ser de un bot. Inténtalo de nuevo.']);
            exit;
        }
    }
    // --- Fin de la verificación ---

    // Valida que los datos necesarios no estén vacíos.
    if (empty($data['name']) || empty($data['email']) || empty($data['message'])) {
        // Si faltan datos, devuelve un error.
        http_response_code(400); // Bad Request
        echo json_encode(['status' => 'error', 'message' => 'Todos los campos son obligatorios.']);
        exit;
    }

    // Lee el contenido actual de la base de datos JSON.
    // Si el archivo no existe o está vacío, inicializa con un array vacío.
    $contacts = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];

    // Crea un nuevo registro de contacto con un ID único y una marca de tiempo.
    $newContact = [
        'id' => uniqid('contact_'),
        'name' => htmlspecialchars($data['name']), // Sanitiza los datos para evitar XSS
        'email' => htmlspecialchars($data['email']),
        'company' => isset($data['company']) ? htmlspecialchars($data['company']) : '',
        'help' => isset($data['help']) ? htmlspecialchars($data['help']) : '',
        'message' => htmlspecialchars($data['message']),
        'timestamp' => date('Y-m-d H:i:s')
    ];

    // Agrega el nuevo contacto al principio del array.
    array_unshift($contacts, $newContact);

    // Guarda el array actualizado de nuevo en el archivo JSON.
    file_put_contents($jsonDbFile, json_encode($contacts, JSON_PRETTY_PRINT));

    // Devuelve una respuesta de éxito.
    echo json_encode(['status' => 'success', 'message' => 'Mensaje recibido correctamente. ¡Gracias por contactarnos!']);

} elseif ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    // Obtiene el cuerpo de la solicitud (que debería ser un JSON con el ID).
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    $contactIdToDelete = $data['id'] ?? null;

    if (!$contactIdToDelete) {
        http_response_code(400); // Bad Request
        echo json_encode(['status' => 'error', 'message' => 'No se proporcionó el ID del contacto a eliminar.']);
        exit;
    }

    // Lee los contactos existentes.
    $contacts = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];

    // Filtra el array, manteniendo todos los contactos excepto el que se va a eliminar.
    $updatedContacts = array_values(array_filter($contacts, function($contact) use ($contactIdToDelete) {
        return $contact['id'] !== $contactIdToDelete;
    }));

    // Guarda el array actualizado.
    file_put_contents($jsonDbFile, json_encode($updatedContacts, JSON_PRETTY_PRINT));

    echo json_encode(['status' => 'success', 'message' => 'Contacto eliminado correctamente.']);
} else {
    // Si el método no es POST, devuelve un error.
    http_response_code(405); // Method Not Allowed
    echo json_encode(['status' => 'error', 'message' => 'Método no permitido.']);
}
?>