<?php
// lead_handler.php

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Método no permitido.']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);

$name = trim($input['name'] ?? '');
$email = trim($input['email'] ?? '');
$filePath = $input['filePath'] ?? '';

if (empty($name) || empty($email) || empty($filePath)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Nombre, email y archivo son requeridos.']);
    exit;
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Email no válido.']);
    exit;
}

// --- Seguridad: Validar la ruta del archivo ---
// Asegurarse de que el archivo está en el directorio de adjuntos esperado
$baseDir = __DIR__ . '/uploads/blog/attachments/';
$realBasePath = realpath($baseDir);
$realFilePath = realpath(__DIR__ . '/' . $filePath);

if ($realFilePath === false || strpos($realFilePath, $realBasePath) !== 0) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Acceso denegado al archivo.']);
    exit;
}

/* --- Enviar notificación por correo (Desactivado para evitar errores en el servidor) ---
 $to = "oscarmorinigo@dunamis.com.py, contacto@dunamis.com.py, didierluckmann@dunamis.com.py";
 $email_subject = "Nuevo Lead por Descarga desde el Blog";
 $email_body = "Un usuario ha descargado un archivo desde el blog.\n\n" .
               "Detalles del Lead:\n" .
               "--------------------------------------------------\n" .
               "Nombre: $name\n" .
               "Email: $email\n" .
               "Archivo Descargado: " . basename($filePath) . "\n" .
               "--------------------------------------------------\n";
 $headers = "From: no-reply@dunamis.com.py\r\n" .
            "Reply-To: $email";
 
 @mail($to, $email_subject, $email_body, $headers);
*/

// --- Guardar el Lead ---
$db_file = 'db.json';
$db_handle = @fopen($db_file, 'c+'); // Abrir para leer/escribir, crear si no existe.

if ($db_handle === false) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error del servidor: no se puede acceder a la base de datos.']);
    exit;
}

// Bloquear el archivo para evitar condiciones de carrera (race conditions)
if (flock($db_handle, LOCK_EX)) {
    $json_data = stream_get_contents($db_handle);
    $db_template = ['users' => [], 'contacts' => [], 'blogPosts' => [], 'leads' => [], 'crmLeads' => [], 'nextcloudLeads' => []];
    $db = $db_template;

    if (!empty(trim($json_data))) {
        $decoded_db = json_decode($json_data, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            flock($db_handle, LOCK_UN); // Liberar bloqueo
            fclose($db_handle);
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Error del servidor: la base de datos parece estar corrupta.']);
            exit;
        }

        if (is_array($decoded_db)) {
            $db = array_merge($db_template, $decoded_db);
        }
    }

    // Crear el nuevo lead
    $new_lead = [
        'id' => 'lead_' . uniqid(),
        'date' => date('Y-m-d H:i:s'),
        'name' => $name,
        'email' => $email,
        'downloaded_file' => basename($filePath)
    ];

    // Añadir el nuevo lead al principio del array
    // Asegurarse de que leads es un array antes de añadir el nuevo lead
    if (!isset($db['leads']) || !is_array($db['leads'])) {
        $db['leads'] = [];
    }
    array_unshift($db['leads'], $new_lead);

    // Guardar en la base de datos
    ftruncate($db_handle, 0); // Vaciar el archivo
    rewind($db_handle); // Mover el puntero al inicio
    fwrite($db_handle, json_encode($db, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
    
    flock($db_handle, LOCK_UN); // Liberar el bloqueo
} else {
    // No se pudo obtener el bloqueo
    http_response_code(503); // Service Unavailable
    echo json_encode(['success' => false, 'message' => 'El servidor está ocupado. Por favor, inténtelo de nuevo más tarde.']);
    exit;
}

fclose($db_handle);

// --- INICIO: CÓDIGO MODIFICADO PARA FORZAR LA DESCARGA ---
// En lugar de devolver una URL, forzamos la descarga del archivo desde PHP.
if (file_exists($realFilePath)) {
    // Limpiar cualquier salida anterior
    if (ob_get_level()) {
        ob_end_clean();
    }

    header('Content-Description: File Transfer');
    header('Content-Type: application/octet-stream'); // Tipo de contenido genérico para forzar descarga
    header('Content-Disposition: attachment; filename="' . basename($realFilePath) . '"');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . filesize($realFilePath));
    readfile($realFilePath); // Lee el archivo y lo envía al navegador
    exit; // Termina el script después de enviar el archivo
} else {
    http_response_code(404);
    // Devolvemos un JSON en caso de error para que el frontend pueda manejarlo.
    echo json_encode(['success' => false, 'message' => 'El archivo solicitado no fue encontrado en el servidor.']);
}
// --- FIN: CÓDIGO MODIFICADO ---