<?php
// crm_lead_handler.php

// Establecer la cabecera para devolver JSON
header('Content-Type: application/json');

// Solo permitir peticiones POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['success' => false, 'message' => 'Método no permitido.']);
    exit;
}

// Obtener los datos enviados en el cuerpo de la petición
$input = json_decode(file_get_contents('php://input'), true);

// Validar datos
$name = trim($input['nombre'] ?? '');
$email = trim($input['email'] ?? '');
$company = trim($input['empresa'] ?? '');
$sector = trim($input['rubro'] ?? '');
$message = trim($input['mensaje'] ?? '');

if (empty($name) || empty($email) || empty($company) || empty($sector)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Por favor, complete todos los campos requeridos.']);
    exit;
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Por favor, ingrese una dirección de correo electrónico válida.']);
    exit;
}

// Cargar la base de datos
$db_file = 'db.json';
$db_handle = @fopen($db_file, 'c+'); // Abrir para leer/escribir, crear si no existe.

if ($db_handle === false) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error del servidor: no se puede acceder a la base de datos.']);
    exit;
}

// Bloquear el archivo para evitar condiciones de carrera (race conditions)
if (flock($db_handle, LOCK_EX)) {
    $json_data = stream_get_contents($db_handle);
    $db_template = ['users' => [], 'contacts' => [], 'blogPosts' => [], 'leads' => [], 'crmLeads' => [], 'nextcloudLeads' => []];
    $db = $db_template;

    if (!empty(trim($json_data))) {
        $decoded_db = json_decode($json_data, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            flock($db_handle, LOCK_UN); // Liberar bloqueo
            fclose($db_handle);
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Error del servidor: la base de datos parece estar corrupta.']);
            exit;
        }

        if (is_array($decoded_db)) {
            $db = array_merge($db_template, $decoded_db);
        }
    }

    // Crear el nuevo lead
    $new_lead = [
        'id' => 'crmlead_' . uniqid(),
        'date' => date('Y-m-d H:i:s'),
        'name' => $name,
        'email' => $email,
        'company' => $company,
        'sector' => $sector,
        'message' => $message,
        'read' => false
    ];

    // Añadir el nuevo lead al principio del array
    // Asegurarse de que crmLeads es un array antes de añadir el nuevo lead
    if (!isset($db['crmLeads']) || !is_array($db['crmLeads'])) {
        $db['crmLeads'] = [];
    }
    array_unshift($db['crmLeads'], $new_lead);

    // Guardar en la base de datos
    ftruncate($db_handle, 0); // Vaciar el archivo
    rewind($db_handle); // Mover el puntero al inicio
    fwrite($db_handle, json_encode($db, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
    
    flock($db_handle, LOCK_UN); // Liberar el bloqueo
} else {
    // No se pudo obtener el bloqueo
    http_response_code(503); // Service Unavailable
    echo json_encode(['success' => false, 'message' => 'El servidor está ocupado. Por favor, inténtelo de nuevo más tarde.']);
    exit;
}

fclose($db_handle);

echo json_encode(['success' => true, 'message' => '¡Solicitud enviada con éxito! Nos pondremos en contacto con usted pronto.']);