<?php
/**
 * LexiPro - Gestión de Usuarios
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            background: white;
            padding: 20px 30px;
            border-radius: 12px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            font-size: 28px;
            color: #1e3c72;
        }
        
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(30, 60, 114, 0.3);
        }
        
        .filters {
            background: white;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }
        
        .filter-group {
            flex: 1;
            min-width: 200px;
        }
        
        .filter-group label {
            display: block;
            margin-bottom: 5px;
            font-size: 13px;
            font-weight: 600;
            color: #555;
        }
        
        .filter-group input,
        .filter-group select {
            width: 100%;
            padding: 10px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .table-container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            overflow: hidden;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        thead {
            background: #f9fafb;
        }
        
        th {
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #374151;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        td {
            padding: 15px;
            border-top: 1px solid #f3f4f6;
        }
        
        tbody tr:hover {
            background: #f9fafb;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .user-name {
            font-weight: 600;
            color: #1f2937;
        }
        
        .user-email {
            font-size: 13px;
            color: #6b7280;
        }
        
        .badge {
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .badge-success {
            background: #d1fae5;
            color: #065f46;
        }
        
        .badge-danger {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .badge-info {
            background: #dbeafe;
            color: #1e40af;
        }
        
        .actions {
            display: flex;
            gap: 8px;
        }
        
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }
        
        .btn-edit {
            background: #3b82f6;
            color: white;
        }
        
        .btn-delete {
            background: #ef4444;
            color: white;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }
        
        .modal.active {
            display: flex;
        }
        
        .modal-content {
            background: white;
            border-radius: 16px;
            max-width: 600px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }
        
        .modal-header {
            padding: 20px 30px;
            border-bottom: 1px solid #e5e7eb;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h2 {
            font-size: 20px;
            color: #1f2937;
        }
        
        .modal-close {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #6b7280;
        }
        
        .modal-body {
            padding: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #374151;
        }
        
        .form-group input,
        .form-group select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            color: #6b7280;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>👥 Gestión de Usuarios</h1>
                <p style="color: #6b7280; margin-top: 5px;">Administra los usuarios del sistema</p>
            </div>
            <?php if ($auth->hasPermission('usuarios', 'crear')): ?>
                <button class="btn btn-primary" onclick="openModal()">+ Nuevo Usuario</button>
            <?php endif; ?>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>🔍 Buscar</label>
                <input type="text" id="searchInput" placeholder="Nombre, usuario, email...">
            </div>
            <div class="filter-group">
                <label>Rol</label>
                <select id="rolFilter">
                    <option value="">Todos los roles</option>
                    <?php foreach ($roles as $rol): ?>
                        <option value="<?= $rol['id'] ?>"><?= e($rol['nombre']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label>Estado</label>
                <select id="activoFilter">
                    <option value="">Todos</option>
                    <option value="1">Activos</option>
                    <option value="0">Inactivos</option>
                </select>
            </div>
        </div>
        
        <div class="table-container">
            <table id="usuariosTable">
                <thead>
                    <tr>
                        <th>Usuario</th>
                        <th>Rol</th>
                        <th>Tipo de Pago</th>
                        <th>Estado</th>
                        <th>Último Acceso</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="usuariosBody">
                    <tr>
                        <td colspan="6" class="loading">Cargando usuarios...</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Modal de Usuario -->
    <div class="modal" id="userModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitle">Nuevo Usuario</h2>
                <button class="modal-close" onclick="closeModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="userForm">
                    <input type="hidden" id="userId" name="id">
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Nombre *</label>
                            <input type="text" name="nombre" required>
                        </div>
                        <div class="form-group">
                            <label>Apellido *</label>
                            <input type="text" name="apellido" required>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Usuario *</label>
                            <input type="text" name="usuario" required>
                        </div>
                        <div class="form-group">
                            <label>Email *</label>
                            <input type="email" name="email" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Contraseña *</label>
                        <input type="password" name="password" id="passwordField">
                        <small style="color: #6b7280;">Dejar en blanco para mantener la contraseña actual</small>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Teléfono</label>
                            <input type="text" name="telefono">
                        </div>
                        <div class="form-group">
                            <label>Fecha de Nacimiento</label>
                            <input type="date" name="fecha_nacimiento">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Cargo</label>
                            <input type="text" name="cargo">
                        </div>
                        <div class="form-group">
                            <label>Rol *</label>
                            <select name="rol_id" required>
                                <?php foreach ($roles as $rol): ?>
                                    <option value="<?= $rol['id'] ?>"><?= e($rol['nombre']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Tipo de Pago</label>
                            <select name="tipo_pago" id="tipoPago" onchange="togglePagoFields()">
                                <option value="mensual">Mensual</option>
                                <option value="hora">Por Hora</option>
                            </select>
                        </div>
                        <div class="form-group" id="tarifaHoraGroup" style="display: none;">
                            <label>Tarifa por Hora (Gs.)</label>
                            <input type="number" name="tarifa_hora" step="0.01">
                        </div>
                        <div class="form-group" id="salarioMensualGroup">
                            <label>Salario Mensual (Gs.)</label>
                            <input type="number" name="salario_mensual" step="0.01">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>
                                <input type="checkbox" name="es_proveedor_externo" value="1">
                                Proveedor Externo
                            </label>
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" name="activo" value="1" checked>
                                Activo
                            </label>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Foto de Perfil</label>
                        <input type="file" name="foto" accept="image/*">
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                        Guardar Usuario
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let usuarios = [];
        
        // Cargar usuarios
        async function loadUsuarios() {
            try {
                const search = document.getElementById('searchInput').value;
                const rol = document.getElementById('rolFilter').value;
                const activo = document.getElementById('activoFilter').value;
                
                const params = new URLSearchParams();
                if (search) params.append('search', search);
                if (rol) params.append('rol', rol);
                if (activo !== '') params.append('activo', activo);
                
                const response = await fetch(`api.php?${params}`);
                const data = await response.json();
                
                if (data.success) {
                    usuarios = data.usuarios;
                    renderUsuarios();
                }
            } catch (error) {
                console.error('Error al cargar usuarios:', error);
            }
        }
        
        // Renderizar tabla
        function renderUsuarios() {
            const tbody = document.getElementById('usuariosBody');
            
            if (usuarios.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" class="loading">No se encontraron usuarios</td></tr>';
                return;
            }
            
            tbody.innerHTML = usuarios.map(user => `
                <tr>
                    <td>
                        <div class="user-info">
                            <img src="${user.foto || '../../assets/img/default-avatar.png'}" alt="${user.nombre}" class="user-avatar">
                            <div>
                                <div class="user-name">${user.nombre} ${user.apellido}</div>
                                <div class="user-email">${user.email}</div>
                            </div>
                        </div>
                    </td>
                    <td><span class="badge badge-info">${user.rol_nombre}</span></td>
                    <td>${user.tipo_pago === 'hora' ? 'Por Hora' : 'Mensual'}</td>
                    <td>
                        <span class="badge ${user.activo == 1 ? 'badge-success' : 'badge-danger'}">
                            ${user.activo == 1 ? 'Activo' : 'Inactivo'}
                        </span>
                    </td>
                    <td>${user.ultimo_acceso ? new Date(user.ultimo_acceso).toLocaleString('es-PY') : 'Nunca'}</td>
                    <td>
                        <div class="actions">
                            <?php if ($auth->hasPermission('usuarios', 'editar')): ?>
                                <button class="btn btn-sm btn-edit" onclick="editUser(${user.id})">Editar</button>
                            <?php endif; ?>
                            <?php if ($auth->hasPermission('usuarios', 'eliminar')): ?>
                                <button class="btn btn-sm btn-delete" onclick="deleteUser(${user.id})">Eliminar</button>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
            `).join('');
        }
        
        // Abrir modal
        function openModal(userId = null) {
            const modal = document.getElementById('userModal');
            const form = document.getElementById('userForm');
            const title = document.getElementById('modalTitle');
            
            form.reset();
            
            if (userId) {
                title.textContent = 'Editar Usuario';
                const user = usuarios.find(u => u.id == userId);
                if (user) {
                    Object.keys(user).forEach(key => {
                        const field = form.elements[key];
                        if (field) {
                            if (field.type === 'checkbox') {
                                field.checked = user[key] == 1;
                            } else {
                                field.value = user[key] || '';
                            }
                        }
                    });
                    document.getElementById('passwordField').removeAttribute('required');
                }
            } else {
                title.textContent = 'Nuevo Usuario';
                document.getElementById('passwordField').setAttribute('required', 'required');
            }
            
            togglePagoFields();
            modal.classList.add('active');
        }
        
        // Cerrar modal
        function closeModal() {
            document.getElementById('userModal').classList.remove('active');
        }
        
        // Toggle campos de pago
        function togglePagoFields() {
            const tipoPago = document.getElementById('tipoPago').value;
            const tarifaHoraGroup = document.getElementById('tarifaHoraGroup');
            const salarioMensualGroup = document.getElementById('salarioMensualGroup');
            
            if (tipoPago === 'hora') {
                tarifaHoraGroup.style.display = 'block';
                salarioMensualGroup.style.display = 'none';
            } else {
                tarifaHoraGroup.style.display = 'none';
                salarioMensualGroup.style.display = 'block';
            }
        }
        
        // Guardar usuario
        document.getElementById('userForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = Object.fromEntries(formData);
            
            try {
                const method = data.id ? 'PUT' : 'POST';
                const response = await fetch('api.php', {
                    method,
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    closeModal();
                    loadUsuarios();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar usuario');
                console.error(error);
            }
        });
        
        // Editar usuario
        function editUser(id) {
            openModal(id);
        }
        
        // Eliminar usuario
        async function deleteUser(id) {
            if (!confirm('¿Estás seguro de eliminar este usuario?')) return;
            
            try {
                const response = await fetch('api.php', {
                    method: 'DELETE',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({id})
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    loadUsuarios();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al eliminar usuario');
                console.error(error);
            }
        }
        
        // Event listeners
        document.getElementById('searchInput').addEventListener('input', loadUsuarios);
        document.getElementById('rolFilter').addEventListener('change', loadUsuarios);
        document.getElementById('activoFilter').addEventListener('change', loadUsuarios);
        
        // Cargar al inicio
        loadUsuarios();
    </script>
</body>
</html>
