<?php
/**
 * LexiPro - API de Usuarios
 */

session_start();
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$method = $_SERVER['REQUEST_METHOD'];
$userId = $_SESSION['user_id'];

try {
    // GET - Listar usuarios
    if ($method === 'GET') {
        $auth->requirePermission('usuarios', 'ver');
        
        if (isset($_GET['id'])) {
            // Obtener un usuario específico
            $user = $db->fetchOne(
                "SELECT u.*, r.nombre as rol_nombre 
                 FROM usuarios u 
                 INNER JOIN roles r ON u.rol_id = r.id 
                 WHERE u.id = ?",
                [$_GET['id']]
            );
            
            if ($user) {
                unset($user['password']);
                jsonSuccess(['usuario' => $user]);
            } else {
                jsonError('Usuario no encontrado', 404);
            }
        } else {
            // Listar todos los usuarios
            $search = $_GET['search'] ?? '';
            $rol = $_GET['rol'] ?? '';
            $activo = $_GET['activo'] ?? '';
            
            $sql = "SELECT u.*, r.nombre as rol_nombre 
                    FROM usuarios u 
                    INNER JOIN roles r ON u.rol_id = r.id 
                    WHERE 1=1";
            $params = [];
            
            if ($search) {
                $sql .= " AND (u.nombre LIKE ? OR u.apellido LIKE ? OR u.usuario LIKE ? OR u.email LIKE ?)";
                $searchParam = "%{$search}%";
                $params = array_merge($params, [$searchParam, $searchParam, $searchParam, $searchParam]);
            }
            
            if ($rol) {
                $sql .= " AND u.rol_id = ?";
                $params[] = $rol;
            }
            
            if ($activo !== '') {
                $sql .= " AND u.activo = ?";
                $params[] = $activo;
            }
            
            $sql .= " ORDER BY u.created_at DESC";
            
            $usuarios = $db->fetchAll($sql, $params);
            
            // Remover contraseñas
            foreach ($usuarios as &$user) {
                unset($user['password']);
            }
            
            jsonSuccess(['usuarios' => $usuarios]);
        }
    }
    
    // POST - Crear usuario
    elseif ($method === 'POST') {
        $auth->requirePermission('usuarios', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validaciones
        if (empty($data['nombre']) || empty($data['apellido']) || empty($data['usuario']) || empty($data['email']) || empty($data['password'])) {
            jsonError('Todos los campos obligatorios deben ser completados');
        }
        
        // Verificar si el usuario ya existe
        $existingUser = $db->fetchOne(
            "SELECT id FROM usuarios WHERE usuario = ? OR email = ?",
            [$data['usuario'], $data['email']]
        );
        
        if ($existingUser) {
            jsonError('El usuario o email ya existe');
        }
        
        // Manejar foto de perfil
        $foto = null;
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
            $uploadResult = uploadFile($_FILES['foto'], 'uploads/usuarios');
            $foto = $uploadResult['ruta'];
        }
        
        // Crear usuario
        $newUserId = $db->insert('usuarios', [
            'nombre' => sanitize($data['nombre']),
            'apellido' => sanitize($data['apellido']),
            'usuario' => sanitize($data['usuario']),
            'password' => password_hash($data['password'], PASSWORD_DEFAULT),
            'email' => sanitize($data['email']),
            'telefono' => sanitize($data['telefono'] ?? null),
            'fecha_nacimiento' => $data['fecha_nacimiento'] ?? null,
            'foto' => $foto,
            'cargo' => sanitize($data['cargo'] ?? null),
            'rol_id' => (int)$data['rol_id'],
            'tipo_pago' => $data['tipo_pago'] ?? 'mensual',
            'tarifa_hora' => $data['tarifa_hora'] ?? null,
            'salario_mensual' => $data['salario_mensual'] ?? null,
            'es_proveedor_externo' => (int)($data['es_proveedor_externo'] ?? 0),
            'activo' => (int)($data['activo'] ?? 1)
        ]);
        
        // Log de actividad
        logActivity($userId, 'crear', 'usuarios', $newUserId, "Usuario {$data['usuario']} creado");
        
        // Notificar al nuevo usuario
        createNotification(
            $newUserId,
            'sistema',
            'Bienvenido a LexiPro',
            'Tu cuenta ha sido creada exitosamente. Ya puedes acceder al sistema.',
            '/lexipro.php'
        );
        
        jsonSuccess(['id' => $newUserId], 'Usuario creado exitosamente');
    }
    
    // PUT - Actualizar usuario
    elseif ($method === 'PUT') {
        $auth->requirePermission('usuarios', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de usuario requerido');
        }
        
        $updateData = [
            'nombre' => sanitize($data['nombre']),
            'apellido' => sanitize($data['apellido']),
            'email' => sanitize($data['email']),
            'telefono' => sanitize($data['telefono'] ?? null),
            'fecha_nacimiento' => $data['fecha_nacimiento'] ?? null,
            'cargo' => sanitize($data['cargo'] ?? null),
            'rol_id' => (int)$data['rol_id'],
            'tipo_pago' => $data['tipo_pago'] ?? 'mensual',
            'tarifa_hora' => $data['tarifa_hora'] ?? null,
            'salario_mensual' => $data['salario_mensual'] ?? null,
            'es_proveedor_externo' => (int)($data['es_proveedor_externo'] ?? 0),
            'activo' => (int)($data['activo'] ?? 1)
        ];
        
        // Actualizar contraseña solo si se proporciona
        if (!empty($data['password'])) {
            $updateData['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
        }
        
        // Manejar foto de perfil
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
            // Eliminar foto anterior
            $oldUser = $db->fetchOne("SELECT foto FROM usuarios WHERE id = ?", [$data['id']]);
            if ($oldUser && $oldUser['foto']) {
                deleteFile($oldUser['foto']);
            }
            
            $uploadResult = uploadFile($_FILES['foto'], 'uploads/usuarios');
            $updateData['foto'] = $uploadResult['ruta'];
        }
        
        $db->update('usuarios', $updateData, 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'actualizar', 'usuarios', $data['id'], "Usuario actualizado");
        
        jsonSuccess([], 'Usuario actualizado exitosamente');
    }
    
    // DELETE - Eliminar usuario
    elseif ($method === 'DELETE') {
        $auth->requirePermission('usuarios', 'eliminar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de usuario requerido');
        }
        
        // No permitir eliminar al usuario actual
        if ($data['id'] == $userId) {
            jsonError('No puedes eliminar tu propio usuario');
        }
        
        // Obtener foto para eliminar
        $user = $db->fetchOne("SELECT foto FROM usuarios WHERE id = ?", [$data['id']]);
        
        // Eliminar usuario
        $db->delete('usuarios', 'id = ?', [$data['id']]);
        
        // Eliminar foto
        if ($user && $user['foto']) {
            deleteFile($user['foto']);
        }
        
        // Log de actividad
        logActivity($userId, 'eliminar', 'usuarios', $data['id'], "Usuario eliminado");
        
        jsonSuccess([], 'Usuario eliminado exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de usuarios: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
