<?php
/**
 * LexiPro - API de Tareas
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Listar tareas
    if ($method === 'GET') {
        $auth->requirePermission('tareas', 'ver');
        
        if (isset($_GET['caso_id'])) {
            // Tareas de un caso específico
            $tareas = $db->fetchAll(
                "SELECT t.*, u.nombre as responsable_nombre, u.apellido as responsable_apellido
                 FROM tareas t
                 INNER JOIN usuarios u ON t.responsable_id = u.id
                 WHERE t.caso_id = ?
                 ORDER BY t.es_plazo_juridico DESC, t.fecha_limite ASC",
                [$_GET['caso_id']]
            );
            
            jsonSuccess(['tareas' => $tareas]);
        } else {
            // Todas las tareas del usuario
            $tareas = $db->fetchAll(
                "SELECT t.*, c.nombre as caso_nombre, c.numero_caso
                 FROM tareas t
                 INNER JOIN casos c ON t.caso_id = c.id
                 WHERE t.responsable_id = ? AND t.estado != 'completada'
                 ORDER BY t.es_plazo_juridico DESC, t.fecha_limite ASC
                 LIMIT 50",
                [$userId]
            );
            
            jsonSuccess(['tareas' => $tareas]);
        }
    }
    
    // POST - Crear tarea
    elseif ($method === 'POST') {
        $auth->requirePermission('tareas', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['caso_id']) || empty($data['titulo']) || empty($data['responsable_id'])) {
            jsonError('Caso, título y responsable son obligatorios');
        }
        
        $tareaId = $db->insert('tareas', [
            'caso_id' => (int)$data['caso_id'],
            'titulo' => sanitize($data['titulo']),
            'descripcion' => sanitize($data['descripcion'] ?? null),
            'responsable_id' => (int)$data['responsable_id'],
            'fecha_inicio' => $data['fecha_inicio'] ?? null,
            'fecha_limite' => $data['fecha_limite'] ?? null,
            'horas_estimadas' => $data['horas_estimadas'] ?? null,
            'estado' => $data['estado'] ?? 'pendiente',
            'prioridad' => $data['prioridad'] ?? 'media',
            'es_plazo_juridico' => (int)($data['es_plazo_juridico'] ?? 0),
            'notas' => sanitize($data['notas'] ?? null)
        ]);
        
        // Log de actividad
        logActivity($userId, 'crear', 'tareas', $tareaId, "Tarea creada: {$data['titulo']}");
        
        // Notificar al responsable si es diferente
        if ($data['responsable_id'] != $userId) {
            createNotification(
                $data['responsable_id'],
                'tarea',
                'Nueva tarea asignada',
                "Se te ha asignado la tarea: {$data['titulo']}",
                "/modules/casos/view.php?id={$data['caso_id']}"
            );
        }
        
        // Si es plazo jurídico, notificar a todos los usuarios del caso
        if (!empty($data['es_plazo_juridico'])) {
            $usuariosCaso = $db->fetchAll(
                "SELECT DISTINCT usuario_id FROM casos_usuarios WHERE caso_id = ?",
                [$data['caso_id']]
            );
            
            foreach ($usuariosCaso as $usuario) {
                if ($usuario['usuario_id'] != $userId) {
                    createNotification(
                        $usuario['usuario_id'],
                        'plazo',
                        '⚠️ Nuevo plazo jurídico',
                        "Plazo: {$data['titulo']} - Vence: " . formatDate($data['fecha_limite']),
                        "/modules/casos/view.php?id={$data['caso_id']}"
                    );
                }
            }
        }
        
        jsonSuccess(['id' => $tareaId], 'Tarea creada exitosamente');
    }
    
    // PUT - Actualizar tarea
    elseif ($method === 'PUT') {
        $auth->requirePermission('tareas', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de tarea requerido');
        }
        
        $updateData = [
            'titulo' => sanitize($data['titulo']),
            'descripcion' => sanitize($data['descripcion'] ?? null),
            'responsable_id' => (int)$data['responsable_id'],
            'fecha_inicio' => $data['fecha_inicio'] ?? null,
            'fecha_limite' => $data['fecha_limite'] ?? null,
            'horas_estimadas' => $data['horas_estimadas'] ?? null,
            'horas_trabajadas' => $data['horas_trabajadas'] ?? 0,
            'estado' => $data['estado'],
            'prioridad' => $data['prioridad'],
            'es_plazo_juridico' => (int)($data['es_plazo_juridico'] ?? 0),
            'notas' => sanitize($data['notas'] ?? null)
        ];
        
        // Si se marca como completada, registrar fecha
        if ($data['estado'] === 'completada') {
            $updateData['fecha_completada'] = date('Y-m-d H:i:s');
        }
        
        $db->update('tareas', $updateData, 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'actualizar', 'tareas', $data['id'], "Tarea actualizada");
        
        jsonSuccess([], 'Tarea actualizada exitosamente');
    }
    
    // DELETE - Eliminar tarea
    elseif ($method === 'DELETE') {
        $auth->requirePermission('tareas', 'eliminar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de tarea requerido');
        }
        
        $db->delete('tareas', 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'eliminar', 'tareas', $data['id'], "Tarea eliminada");
        
        jsonSuccess([], 'Tarea eliminada exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de tareas: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
