<?php
/**
 * LexiPro - Gestión de Recursos Humanos
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Breadcrumbs
$breadcrumbs = [
    ['label' => 'RRHH']
];
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>RRHH - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .rrhh-header {
            background: linear-gradient(135deg, #ec4899 0%, #db2777 100%);
            color: white;
            padding: 30px;
            border-radius: 12px 12px 0 0;
            margin-bottom: 0;
        }
        
        .filter-bar {
            background: white;
            padding: 20px;
            border-radius: 0 0 12px 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            margin-bottom: 20px;
            display: flex;
            gap: 20px;
            align-items: flex-end;
        }
        
        .user-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            transition: all 0.3s;
            border-left: 5px solid transparent;
        }
        
        .user-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .user-card.pendiente { border-left-color: #f59e0b; }
        .user-card.pagado { border-left-color: #10b981; }
        
        .grid-users {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .amount-display {
            font-size: 24px;
            font-weight: 700;
            color: #1f2937;
            margin: 10px 0;
        }
        
        .detail-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 5px;
            font-size: 14px;
            color: #6b7280;
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="rrhh-header">
            <h1>👥 Gestión de Salarios y Honorarios</h1>
            <p style="opacity: 0.9; margin-top: 5px;">Control de pagos al personal y colaboradores</p>
        </div>
        
        <div class="filter-bar">
            <div class="form-group" style="margin-bottom: 0;">
                <label>Mes de Pago</label>
                <input type="month" id="mesSelector" value="<?= date('Y-m') ?>" onchange="loadUsuarios()">
            </div>
            <div style="flex-grow: 1; text-align: right;">
                <span id="totalPagarDisplay" style="font-size: 18px; font-weight: 600; color: #4b5563;">Total a Pagar: Gs. 0</span>
            </div>
        </div>
        
        <div id="usersGrid" class="grid-users">
            <!-- Cards se cargarán aquí -->
            <div class="loading">Cargando información...</div>
        </div>
        
        <h2 style="margin-top: 40px; margin-bottom: 20px;">Historial de Pagos Recientes</h2>
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>Fecha</th>
                        <th>Beneficiario</th>
                        <th>Mes</th>
                        <th>Monto</th>
                        <th>Forma Pago</th>
                        <th>Estado</th>
                    </tr>
                </thead>
                <tbody id="historialBody">
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Modal de Pago -->
    <div class="modal" id="pagoModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Registrar Pago</h2>
                <button class="modal-close" onclick="closePagoModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="pagoForm">
                    <input type="hidden" name="usuario_id" id="pagoUsuarioId">
                    <input type="hidden" name="mes" id="pagoMes">
                    
                    <div class="form-group">
                        <label>Beneficiario</label>
                        <input type="text" id="pagoNombreUsuario" readonly style="background: #f3f4f6;">
                    </div>
                    
                    <div class="form-group">
                        <label>Monto a Pagar *</label>
                        <input type="number" name="monto" id="pagoMonto" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Forma de Pago</label>
                        <select name="forma_pago">
                            <option value="transferencia">Transferencia Bancaria</option>
                            <option value="efectivo">Efectivo</option>
                            <option value="cheque">Cheque</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Observaciones</label>
                        <textarea name="observaciones" rows="2" style="width: 100%; padding: 10px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="alert" style="background: #dbeafe; border-left: 4px solid #3b82f6; padding: 10px; border-radius: 6px; font-size: 13px;">
                        ℹ️ Se generará automáticamente un asiento contable de egreso.
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 15px;">Confirmar Pago</button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        // Cargar usuarios y pagos
        async function loadUsuarios() {
            const mes = document.getElementById('mesSelector').value;
            const grid = document.getElementById('usersGrid');
            
            grid.innerHTML = '<div class="loading">Calculando pagos...</div>';
            
            try {
                const response = await fetch(`api.php?action=usuarios_pagos&mes=${mes}`);
                const data = await response.json();
                
                if (data.success) {
                    renderUsers(data.usuarios);
                    loadHistorial();
                }
            } catch (error) {
                console.error(error);
                grid.innerHTML = '<div class="alert alert-danger">Error al cargar datos</div>';
            }
        }
        
        function renderUsers(usuarios) {
            const grid = document.getElementById('usersGrid');
            let totalPagar = 0;
            
            if (usuarios.length === 0) {
                grid.innerHTML = '<p>No hay usuarios activos.</p>';
                return;
            }
            
            grid.innerHTML = usuarios.map(u => {
                const esMensual = u.tipo_pago === 'mensual';
                const monto = parseFloat(u.monto_calculado);
                const pagado = u.estado_pago === 'pagado';
                
                if (!pagado) totalPagar += monto;
                
                return `
                    <div class="user-card ${u.estado_pago}">
                        <div style="display: flex; justify-content: space-between; align-items: start;">
                            <div>
                                <h3 style="font-size: 18px; margin-bottom: 2px;">${u.nombre} ${u.apellido}</h3>
                                <span class="badge badge-info">${u.cargo || 'Sin cargo'}</span>
                            </div>
                            ${pagado ? '<span class="badge badge-success">PAGADO</span>' : '<span class="badge badge-warning">PENDIENTE</span>'}
                        </div>
                        
                        <div style="margin: 15px 0; border-top: 1px solid #f3f4f6; border-bottom: 1px solid #f3f4f6; padding: 10px 0;">
                            <div class="detail-row">
                                <span>Tipo de Pago:</span>
                                <strong>${esMensual ? 'Mensual Fijo' : 'Por Hora'}</strong>
                            </div>
                            ${!esMensual ? `
                                <div class="detail-row">
                                    <span>Horas Registradas:</span>
                                    <strong>${u.horas_trabajadas} hrs</strong>
                                </div>
                                <div class="detail-row">
                                    <span>Costo por Hora:</span>
                                    <strong>${LexiPro.formatMoney(u.costo_hora)}</strong>
                                </div>
                            ` : `
                                <div class="detail-row">
                                    <span>Salario Base:</span>
                                    <strong>${LexiPro.formatMoney(u.salario_mensual)}</strong>
                                </div>
                            `}
                        </div>
                        
                        <div class="amount-display">${LexiPro.formatMoney(monto)}</div>
                        
                        ${!pagado ? `
                            <button class="btn btn-primary" style="width: 100%;" 
                                onclick="openPagoModal(${u.id}, '${u.nombre} ${u.apellido}', ${monto})">
                                Registrar Pago
                            </button>
                        ` : `
                            <div style="text-align: center; font-size: 13px; color: #10b981;">
                                Pagado el ${LexiPro.formatDate(u.pago_info.fecha_pago)}
                            </div>
                        `}
                    </div>
                `;
            }).join('');
            
            document.getElementById('totalPagarDisplay').textContent = `Total Pendiente: ${LexiPro.formatMoney(totalPagar)}`;
        }
        
        async function loadHistorial() {
            try {
                const response = await fetch('api.php?action=historial');
                const data = await response.json();
                
                if (data.success) {
                    const tbody = document.getElementById('historialBody');
                    tbody.innerHTML = data.pagos.map(p => `
                        <tr>
                            <td>${LexiPro.formatDate(p.fecha_pago)}</td>
                            <td>${p.nombre} ${p.apellido}</td>
                            <td>${p.mes_correspondiente}</td>
                            <td>${LexiPro.formatMoney(p.monto)}</td>
                            <td>${p.forma_pago}</td>
                            <td><span class="badge badge-success">Completado</span></td>
                        </tr>
                    `).join('');
                }
            } catch (error) {
                console.error(error);
            }
        }
        
        function openPagoModal(id, nombre, monto) {
            document.getElementById('pagoModal').classList.add('active');
            document.getElementById('pagoUsuarioId').value = id;
            document.getElementById('pagoNombreUsuario').value = nombre;
            document.getElementById('pagoMonto').value = monto;
            document.getElementById('pagoMes').value = document.getElementById('mesSelector').value;
        }
        
        function closePagoModal() {
            document.getElementById('pagoModal').classList.remove('active');
            document.getElementById('pagoForm').reset();
        }
        
        document.getElementById('pagoForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = Object.fromEntries(formData.entries());
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('Pago registrado correctamente');
                    closePagoModal();
                    loadUsuarios();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al registrar pago');
            }
        });
        
        // Iniciar
        loadUsuarios();
    </script>
</body>
</html>
