<?php
/**
 * LexiPro - API de Recursos Humanos (Salarios y Honorarios)
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../includes/contabilidad_functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Obtener datos
    if ($method === 'GET') {
        $auth->requirePermission('rrhh', 'ver');
        
        $action = $_GET['action'] ?? 'list';
        
        if ($action === 'usuarios_pagos') {
            // Listar usuarios con información de pagos pendientes/realizados
            $mes = $_GET['mes'] ?? date('Y-m');
            $anio = $_GET['anio'] ?? date('Y');
            
            // Obtener todos los usuarios activos
            $usuarios = $db->fetchAll("SELECT id, nombre, apellido, cargo, tipo_pago, costo_hora, salario_mensual FROM usuarios WHERE activo = 1 ORDER BY nombre");
            
            foreach ($usuarios as &$usuario) {
                // Calcular horas trabajadas en el mes
                $horas = $db->fetchOne(
                    "SELECT COALESCE(SUM(horas_trabajadas), 0) as total 
                     FROM tareas 
                     WHERE usuario_id = ? AND estado = 'completada' 
                     AND MONTH(fecha_fin) = ? AND YEAR(fecha_fin) = ?",
                    [$usuario['id'], substr($mes, 5, 2), substr($mes, 0, 4)]
                );
                
                $usuario['horas_trabajadas'] = (float)$horas['total'];
                
                // Calcular monto a pagar
                if ($usuario['tipo_pago'] === 'hora') {
                    $usuario['monto_calculado'] = $usuario['horas_trabajadas'] * $usuario['costo_hora'];
                } else {
                    $usuario['monto_calculado'] = $usuario['salario_mensual'];
                }
                
                // Verificar si ya se pagó este mes
                $pago = $db->fetchOne(
                    "SELECT * FROM pagos_rrhh 
                     WHERE usuario_destino_id = ? AND mes_correspondiente = ?",
                    [$usuario['id'], $mes]
                );
                
                $usuario['estado_pago'] = $pago ? 'pagado' : 'pendiente';
                $usuario['pago_info'] = $pago;
            }
            
            jsonSuccess(['usuarios' => $usuarios]);
        }
        elseif ($action === 'historial') {
            // Historial de pagos
            $sql = "SELECT p.*, u.nombre, u.apellido 
                    FROM pagos_rrhh p
                    INNER JOIN usuarios u ON p.usuario_destino_id = u.id
                    ORDER BY p.fecha_pago DESC LIMIT 50";
            
            $pagos = $db->fetchAll($sql);
            jsonSuccess(['pagos' => $pagos]);
        }
    }
    
    // POST - Registrar pago
    elseif ($method === 'POST') {
        $auth->requirePermission('rrhh', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['usuario_id']) || empty($data['monto']) || empty($data['mes'])) {
            jsonError('Datos incompletos');
        }
        
        // Verificar si ya existe pago para ese mes
        $existe = $db->fetchOne(
            "SELECT id FROM pagos_rrhh WHERE usuario_destino_id = ? AND mes_correspondiente = ?",
            [$data['usuario_id'], $data['mes']]
        );
        
        if ($existe) {
            jsonError('Ya existe un pago registrado para este usuario en el mes seleccionado');
        }
        
        $db->beginTransaction();
        
        try {
            // Registrar pago
            $pagoId = $db->insert('pagos_rrhh', [
                'usuario_destino_id' => $data['usuario_id'],
                'monto' => $data['monto'],
                'fecha_pago' => date('Y-m-d'),
                'mes_correspondiente' => $data['mes'],
                'forma_pago' => $data['forma_pago'] ?? 'transferencia',
                'observaciones' => sanitize($data['observaciones'] ?? ''),
                'registrado_por' => $userId
            ]);
            
            // Generar asiento contable automático
            // DEBE: Sueldos y Salarios (5.3) o Honorarios Profesionales (5.1)
            // HABER: Caja/Bancos (1.1.1 / 1.1.2)
            
            $usuario = $db->fetchOne("SELECT tipo_pago FROM usuarios WHERE id = ?", [$data['usuario_id']]);
            $cuentaGastoCodigo = ($usuario['tipo_pago'] === 'mensual') ? '5.3' : '5.1'; // 5.3 Sueldos, 5.1 Gastos Operativos (Honorarios)
            
            $cuentaGasto = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = ?", [$cuentaGastoCodigo]);
            $cuentaCaja = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.1'"); // Asumimos Caja por defecto, podría ser Bancos
            
            if ($cuentaGasto && $cuentaCaja) {
                $asientoId = $db->insert('asientos_contables', [
                    'fecha' => date('Y-m-d'),
                    'descripcion' => "Pago de " . ($usuario['tipo_pago'] === 'mensual' ? 'Salario' : 'Honorarios') . " - Mes " . $data['mes'],
                    'tipo' => 'automatico',
                    'referencia_tipo' => 'pago_rrhh',
                    'referencia_id' => $pagoId,
                    'usuario_id' => $userId
                ]);
                
                // DEBE Gasto
                $db->insert('asientos_detalle', [
                    'asiento_id' => $asientoId,
                    'cuenta_id' => $cuentaGasto['id'],
                    'debe' => $data['monto'],
                    'haber' => 0,
                    'descripcion' => 'Pago de haberes'
                ]);
                
                // HABER Caja
                $db->insert('asientos_detalle', [
                    'asiento_id' => $asientoId,
                    'cuenta_id' => $cuentaCaja['id'],
                    'debe' => 0,
                    'haber' => $data['monto'],
                    'descripcion' => 'Salida de fondos'
                ]);
            }
            
            $db->commit();
            logActivity($userId, 'pago', 'rrhh', $pagoId, "Pago registrado: " . formatMoney($data['monto']));
            
            jsonSuccess(['id' => $pagoId], 'Pago registrado exitosamente');
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
} catch (Exception $e) {
    error_log("Error en API de RRHH: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
