<?php
/**
 * LexiPro - Reportes y Analytics
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reportes - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .kpi-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .kpi-card {
            background: var(--bg-card);
            padding: 24px;
            border-radius: 16px;
            box-shadow: var(--shadow);
            border: 1px solid var(--border);
            display: flex;
            align-items: center;
            gap: 20px;
        }
        
        .kpi-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
        }
        
        .kpi-info h3 {
            font-size: 24px;
            font-weight: 700;
            margin: 0;
            color: var(--text-main);
        }
        
        .kpi-info p {
            color: var(--text-muted);
            font-size: 13px;
            margin: 0;
        }
        
        .charts-grid {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 24px;
            margin-bottom: 24px;
        }
        
        .chart-container {
            background: var(--bg-card);
            padding: 24px;
            border-radius: 16px;
            box-shadow: var(--shadow);
            border: 1px solid var(--border);
            height: 400px;
        }
        
        @media (max-width: 1024px) {
            .charts-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="header-flex">
            <div>
                <h1>📊 Reportes y Analytics</h1>
                <p>Visión general del rendimiento del estudio</p>
            </div>
            <div style="display: flex; gap: 10px;">
                <select id="yearSelect" class="form-control" style="width: auto;" onchange="loadDashboard()">
                    <?php 
                    $currentYear = date('Y');
                    for($i = $currentYear; $i >= $currentYear - 4; $i--) {
                        echo "<option value='$i'>$i</option>";
                    }
                    ?>
                </select>
                <button class="btn btn-secondary" onclick="window.print()">🖨️ Imprimir</button>
            </div>
        </div>
        
        <!-- KPIs -->
        <div class="kpi-grid">
            <div class="kpi-card">
                <div class="kpi-icon" style="background: rgba(16, 185, 129, 0.1); color: var(--success);">💰</div>
                <div class="kpi-info">
                    <h3 id="kpiIngresos">0</h3>
                    <p>Ingresos Totales</p>
                </div>
            </div>
            <div class="kpi-card">
                <div class="kpi-icon" style="background: rgba(239, 68, 68, 0.1); color: var(--danger);">💸</div>
                <div class="kpi-info">
                    <h3 id="kpiGastos">0</h3>
                    <p>Gastos Totales</p>
                </div>
            </div>
            <div class="kpi-card">
                <div class="kpi-icon" style="background: rgba(59, 130, 246, 0.1); color: var(--info);">📈</div>
                <div class="kpi-info">
                    <h3 id="kpiBeneficio">0</h3>
                    <p>Beneficio Neto</p>
                </div>
            </div>
            <div class="kpi-card">
                <div class="kpi-icon" style="background: rgba(245, 158, 11, 0.1); color: var(--warning);">⚖️</div>
                <div class="kpi-info">
                    <h3 id="kpiCasos">0</h3>
                    <p>Casos Nuevos</p>
                </div>
            </div>
        </div>
        
        <!-- Gráficos Fila 1 -->
        <div class="charts-grid">
            <div class="chart-container">
                <h3>Ingresos vs Gastos</h3>
                <canvas id="financeChart"></canvas>
            </div>
            <div class="chart-container">
                <h3>Estado de Casos</h3>
                <canvas id="casosChart"></canvas>
            </div>
        </div>
        
        <!-- Gráficos Fila 2 -->
        <div class="charts-grid">
            <div class="chart-container">
                <h3>Top Clientes</h3>
                <canvas id="clientesChart"></canvas>
            </div>
            <div class="chart-container" style="display: flex; align-items: center; justify-content: center; flex-direction: column; text-align: center;">
                <div style="font-size: 48px; margin-bottom: 20px;">🤖</div>
                <h3>Análisis IA</h3>
                <p style="color: var(--text-muted); margin-bottom: 20px;">Genera un reporte detallado con Gemini</p>
                <button class="btn btn-primary" onclick="window.location.href='../ia/'">Ir al Asistente</button>
            </div>
        </div>
    </div>

    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let financeChart, casosChart, clientesChart;

        document.addEventListener('DOMContentLoaded', () => {
            loadDashboard();
        });

        async function loadDashboard() {
            const year = document.getElementById('yearSelect').value;
            const response = await fetch(`api.php?action=dashboard&year=${year}`);
            const result = await response.json();
            
            if (result.success) {
                updateKPIs(result.data.kpis);
                renderFinanceChart(result.data.chart_finance);
                renderCasosChart(result.data.chart_casos);
                renderClientesChart(result.data.top_clientes);
            }
        }

        function updateKPIs(kpis) {
            document.getElementById('kpiIngresos').textContent = LexiPro.formatMoney(kpis.ingresos);
            document.getElementById('kpiGastos').textContent = LexiPro.formatMoney(kpis.gastos);
            document.getElementById('kpiBeneficio').textContent = LexiPro.formatMoney(kpis.beneficio);
            document.getElementById('kpiCasos').textContent = kpis.casos_nuevos;
        }

        function renderFinanceChart(data) {
            const ctx = document.getElementById('financeChart').getContext('2d');
            
            if (financeChart) financeChart.destroy();
            
            financeChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: data.labels,
                    datasets: [
                        {
                            label: 'Ingresos',
                            data: data.ingresos,
                            borderColor: '#10b981',
                            backgroundColor: 'rgba(16, 185, 129, 0.1)',
                            fill: true,
                            tension: 0.4
                        },
                        {
                            label: 'Gastos',
                            data: data.gastos,
                            borderColor: '#ef4444',
                            backgroundColor: 'rgba(239, 68, 68, 0.1)',
                            fill: true,
                            tension: 0.4
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'top' }
                    },
                    scales: {
                        y: { beginAtZero: true }
                    }
                }
            });
        }

        function renderCasosChart(data) {
            const ctx = document.getElementById('casosChart').getContext('2d');
            
            if (casosChart) casosChart.destroy();
            
            const labels = data.map(item => item.estado.replace('_', ' ').toUpperCase());
            const values = data.map(item => item.total);
            const colors = ['#3b82f6', '#10b981', '#f59e0b', '#ef4444', '#6366f1'];
            
            casosChart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: labels,
                    datasets: [{
                        data: values,
                        backgroundColor: colors
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'bottom' }
                    }
                }
            });
        }

        function renderClientesChart(data) {
            const ctx = document.getElementById('clientesChart').getContext('2d');
            
            if (clientesChart) clientesChart.destroy();
            
            const labels = data.map(item => `${item.nombre} ${item.apellido}`);
            const values = data.map(item => item.total_facturado);
            
            clientesChart = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Facturación',
                        data: values,
                        backgroundColor: '#3b82f6',
                        borderRadius: 6
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false }
                    },
                    scales: {
                        y: { beginAtZero: true }
                    }
                }
            });
        }
    </script>
</body>
</html>
