<?php
/**
 * LexiPro - Gestión de Proveedores
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Breadcrumbs
$breadcrumbs = [
    ['label' => 'Proveedores']
];
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Proveedores - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="header">
            <div>
                <h1>🏪 Gestión de Proveedores</h1>
                <p style="color: #6b7280; margin-top: 5px;">Administra los proveedores del estudio</p>
            </div>
            <button class="btn btn-primary" onclick="openProveedorModal()">+ Nuevo Proveedor</button>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>🔍 Buscar</label>
                <input type="text" id="searchInput" placeholder="Nombre, RUC, email...">
            </div>
            <div class="filter-group">
                <label>Tipo</label>
                <select id="tipoFilter">
                    <option value="">Todos</option>
                    <option value="servicio">Servicio</option>
                    <option value="producto">Producto</option>
                    <option value="mixto">Mixto</option>
                </select>
            </div>
            <div class="filter-group">
                <label>Estado</label>
                <select id="activoFilter">
                    <option value="">Todos</option>
                    <option value="1" selected>Activos</option>
                    <option value="0">Inactivos</option>
                </select>
            </div>
        </div>
        
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>Tipo</th>
                        <th>Razón Social/Nombre</th>
                        <th>RUC</th>
                        <th>Email</th>
                        <th>Teléfono</th>
                        <th>Ciudad</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="proveedoresTableBody">
                    <tr>
                        <td colspan="8" class="loading">Cargando proveedores...</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Modal de Proveedor -->
    <div class="modal" id="proveedorModal">
        <div class="modal-content modal-large">
            <div class="modal-header">
                <h2 id="proveedorModalTitle">Nuevo Proveedor</h2>
                <button class="modal-close" onclick="closeProveedorModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="proveedorForm">
                    <input type="hidden" name="id" id="proveedorId">
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Tipo de Proveedor *</label>
                            <select name="tipo" required>
                                <option value="servicio">Servicio</option>
                                <option value="producto">Producto</option>
                                <option value="mixto">Mixto</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>RUC</label>
                            <input type="text" name="ruc" placeholder="80012345-6">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Razón Social *</label>
                        <input type="text" name="razon_social" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Nombre Comercial</label>
                        <input type="text" name="nombre">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Email</label>
                            <input type="email" name="email">
                        </div>
                        <div class="form-group">
                            <label>Teléfono</label>
                            <input type="text" name="telefono" placeholder="(021) 123-456">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Celular</label>
                        <input type="text" name="celular" placeholder="0981 123 456">
                    </div>
                    
                    <div class="form-group">
                        <label>Dirección</label>
                        <textarea name="direccion" rows="2" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Ciudad</label>
                            <input type="text" name="ciudad" value="Asunción">
                        </div>
                        <div class="form-group">
                            <label>País</label>
                            <input type="text" name="pais" value="Paraguay">
                        </div>
                    </div>
                    
                    <hr style="margin: 20px 0;">
                    
                    <h3 style="margin-bottom: 15px;">Persona de Contacto</h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Nombre del Contacto</label>
                            <input type="text" name="contacto_nombre">
                        </div>
                        <div class="form-group">
                            <label>Cargo</label>
                            <input type="text" name="contacto_cargo">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Notas</label>
                        <textarea name="notas" rows="3" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="activo" value="1" checked>
                            Proveedor activo
                        </label>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                        Guardar Proveedor
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let proveedores = [];
        
        // Cargar proveedores
        async function loadProveedores() {
            try {
                const search = document.getElementById('searchInput').value;
                const tipo = document.getElementById('tipoFilter').value;
                const activo = document.getElementById('activoFilter').value;
                
                const params = new URLSearchParams();
                if (search) params.append('search', search);
                if (tipo) params.append('tipo', tipo);
                if (activo) params.append('activo', activo);
                
                const response = await fetch(`api.php?${params}`);
                const data = await response.json();
                
                if (data.success) {
                    proveedores = data.proveedores;
                    renderProveedores();
                }
            } catch (error) {
                console.error('Error al cargar proveedores:', error);
            }
        }
        
        // Renderizar proveedores
        function renderProveedores() {
            const tbody = document.getElementById('proveedoresTableBody');
            
            if (proveedores.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" class="loading">No se encontraron proveedores</td></tr>';
                return;
            }
            
            tbody.innerHTML = proveedores.map(proveedor => {
                const nombre = proveedor.razon_social || proveedor.nombre;
                
                return `
                    <tr>
                        <td><span class="badge badge-info">${proveedor.tipo.charAt(0).toUpperCase() + proveedor.tipo.slice(1)}</span></td>
                        <td><strong>${nombre}</strong></td>
                        <td>${proveedor.ruc || '-'}</td>
                        <td>${proveedor.email || '-'}</td>
                        <td>${proveedor.celular || proveedor.telefono || '-'}</td>
                        <td>${proveedor.ciudad || '-'}</td>
                        <td><span class="badge badge-${proveedor.activo == 1 ? 'success' : 'danger'}">${proveedor.activo == 1 ? 'Activo' : 'Inactivo'}</span></td>
                        <td>
                            <div class="actions">
                                <button class="btn btn-sm btn-edit" onclick="editProveedor(${proveedor.id})">Editar</button>
                                <button class="btn btn-sm btn-delete" onclick="deleteProveedor(${proveedor.id})">Eliminar</button>
                            </div>
                        </td>
                    </tr>
                `;
            }).join('');
        }
        
        // Abrir modal
        function openProveedorModal(proveedorId = null) {
            const modal = document.getElementById('proveedorModal');
            const form = document.getElementById('proveedorForm');
            const title = document.getElementById('proveedorModalTitle');
            
            form.reset();
            
            if (proveedorId) {
                title.textContent = 'Editar Proveedor';
                const proveedor = proveedores.find(p => p.id == proveedorId);
                if (proveedor) {
                    Object.keys(proveedor).forEach(key => {
                        const field = form.elements[key];
                        if (field) {
                            if (field.type === 'checkbox') {
                                field.checked = proveedor[key] == 1;
                            } else {
                                field.value = proveedor[key] || '';
                            }
                        }
                    });
                }
            } else {
                title.textContent = 'Nuevo Proveedor';
            }
            
            modal.classList.add('active');
        }
        
        // Cerrar modal
        function closeProveedorModal() {
            document.getElementById('proveedorModal').classList.remove('active');
        }
        
        // Editar proveedor
        function editProveedor(id) {
            openProveedorModal(id);
        }
        
        // Eliminar proveedor
        async function deleteProveedor(id) {
            if (!confirm('¿Estás seguro de desactivar este proveedor?')) return;
            
            try {
                const response = await fetch('api.php', {
                    method: 'DELETE',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({id})
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    loadProveedores();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al eliminar proveedor');
                console.error(error);
            }
        }
        
        // Guardar proveedor
        document.getElementById('proveedorForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {};
            
            for (let [key, value] of formData.entries()) {
                if (key === 'activo') {
                    data[key] = formData.get(key) ? 1 : 0;
                } else {
                    data[key] = value;
                }
            }
            
            try {
                const method = data.id ? 'PUT' : 'POST';
                const response = await fetch('api.php', {
                    method,
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    closeProveedorModal();
                    loadProveedores();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar proveedor');
                console.error(error);
            }
        });
        
        // Event listeners
        document.getElementById('searchInput').addEventListener('input', LexiPro.debounce(loadProveedores, 300));
        document.getElementById('tipoFilter').addEventListener('change', loadProveedores);
        document.getElementById('activoFilter').addEventListener('change', loadProveedores);
        
        // Cargar al inicio
        loadProveedores();
    </script>
</body>
</html>
