<?php
/**
 * LexiPro - API de Proveedores
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Listar proveedores
    if ($method === 'GET') {
        $auth->requirePermission('proveedores', 'ver');
        
        if (isset($_GET['id'])) {
            // Proveedor específico
            $proveedor = $db->fetchOne("SELECT * FROM proveedores WHERE id = ?", [$_GET['id']]);
            jsonSuccess(['proveedor' => $proveedor]);
        } else {
            // Listar todos
            $search = $_GET['search'] ?? '';
            $tipo = $_GET['tipo'] ?? '';
            $activo = $_GET['activo'] ?? '';
            
            $sql = "SELECT * FROM proveedores WHERE 1=1";
            $params = [];
            
            if ($search) {
                $sql .= " AND (nombre LIKE ? OR razon_social LIKE ? OR ruc LIKE ? OR email LIKE ?)";
                $searchParam = "%{$search}%";
                $params = array_fill(0, 4, $searchParam);
            }
            
            if ($tipo) {
                $sql .= " AND tipo = ?";
                $params[] = $tipo;
            }
            
            if ($activo !== '') {
                $sql .= " AND activo = ?";
                $params[] = (int)$activo;
            }
            
            $sql .= " ORDER BY razon_social, nombre LIMIT 200";
            
            $proveedores = $db->fetchAll($sql, $params);
            jsonSuccess(['proveedores' => $proveedores]);
        }
    }
    
    // POST - Crear proveedor
    elseif ($method === 'POST') {
        $auth->requirePermission('proveedores', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validaciones
        if (empty($data['razon_social']) && empty($data['nombre'])) {
            jsonError('Razón social o nombre es obligatorio');
        }
        
        // Validar RUC si existe
        if (!empty($data['ruc'])) {
            $existente = $db->fetchOne("SELECT id FROM proveedores WHERE ruc = ? AND id != ?", [
                $data['ruc'],
                $data['id'] ?? 0
            ]);
            
            if ($existente) {
                jsonError('Ya existe un proveedor con este RUC');
            }
        }
        
        $proveedorId = $db->insert('proveedores', [
            'tipo' => $data['tipo'] ?? 'servicio',
            'nombre' => sanitize($data['nombre'] ?? null),
            'razon_social' => sanitize($data['razon_social'] ?? null),
            'ruc' => sanitize($data['ruc'] ?? null),
            'email' => sanitize($data['email'] ?? null),
            'telefono' => sanitize($data['telefono'] ?? null),
            'celular' => sanitize($data['celular'] ?? null),
            'direccion' => sanitize($data['direccion'] ?? null),
            'ciudad' => sanitize($data['ciudad'] ?? null),
            'pais' => sanitize($data['pais'] ?? 'Paraguay'),
            'contacto_nombre' => sanitize($data['contacto_nombre'] ?? null),
            'contacto_cargo' => sanitize($data['contacto_cargo'] ?? null),
            'notas' => sanitize($data['notas'] ?? null),
            'activo' => (int)($data['activo'] ?? 1)
        ]);
        
        // Log de actividad
        logActivity($userId, 'crear', 'proveedores', $proveedorId, "Proveedor creado: " . ($data['razon_social'] ?? $data['nombre']));
        
        jsonSuccess(['id' => $proveedorId], 'Proveedor creado exitosamente');
    }
    
    // PUT - Actualizar proveedor
    elseif ($method === 'PUT') {
        $auth->requirePermission('proveedores', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de proveedor requerido');
        }
        
        // Validar RUC si existe
        if (!empty($data['ruc'])) {
            $existente = $db->fetchOne("SELECT id FROM proveedores WHERE ruc = ? AND id != ?", [
                $data['ruc'],
                $data['id']
            ]);
            
            if ($existente) {
                jsonError('Ya existe un proveedor con este RUC');
            }
        }
        
        $db->update('proveedores', [
            'tipo' => $data['tipo'] ?? 'servicio',
            'nombre' => sanitize($data['nombre'] ?? null),
            'razon_social' => sanitize($data['razon_social'] ?? null),
            'ruc' => sanitize($data['ruc'] ?? null),
            'email' => sanitize($data['email'] ?? null),
            'telefono' => sanitize($data['telefono'] ?? null),
            'celular' => sanitize($data['celular'] ?? null),
            'direccion' => sanitize($data['direccion'] ?? null),
            'ciudad' => sanitize($data['ciudad'] ?? null),
            'pais' => sanitize($data['pais'] ?? 'Paraguay'),
            'contacto_nombre' => sanitize($data['contacto_nombre'] ?? null),
            'contacto_cargo' => sanitize($data['contacto_cargo'] ?? null),
            'notas' => sanitize($data['notas'] ?? null),
            'activo' => (int)($data['activo'] ?? 1)
        ], 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'actualizar', 'proveedores', $data['id'], "Proveedor actualizado");
        
        jsonSuccess([], 'Proveedor actualizado exitosamente');
    }
    
    // DELETE - Eliminar proveedor (desactivar)
    elseif ($method === 'DELETE') {
        $auth->requirePermission('proveedores', 'eliminar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de proveedor requerido');
        }
        
        // Verificar si tiene facturas asociadas
        $facturas = $db->fetchOne("SELECT COUNT(*) as total FROM facturas_compras WHERE proveedor_id = ?", [$data['id']]);
        
        if ($facturas['total'] > 0) {
            jsonError('No se puede eliminar un proveedor con facturas asociadas. Desactívalo en su lugar.');
        }
        
        // Desactivar
        $db->update('proveedores', ['activo' => 0], 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'desactivar', 'proveedores', $data['id'], "Proveedor desactivado");
        
        jsonSuccess([], 'Proveedor desactivado exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de proveedores: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
