<?php
/**
 * LexiPro - Gestión de Permisos
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

// Solo administradores
if ($_SESSION['user_rol_nivel'] < 100) {
    header('Location: ../../lexipro.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Permisos - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .container {
            max-width: 100%;
            overflow-x: auto;
        }
        
        .permissions-table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .permissions-table th,
        .permissions-table td {
            padding: 12px;
            border: 1px solid #e5e7eb;
            text-align: center;
        }
        
        .permissions-table th {
            background: #f9fafb;
            font-weight: 600;
            color: #374151;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .permissions-table th:first-child {
            position: sticky;
            left: 0;
            z-index: 20;
            background: #f9fafb;
            text-align: left;
            min-width: 150px;
        }
        
        .permissions-table td:first-child {
            position: sticky;
            left: 0;
            background: white;
            font-weight: 600;
            text-align: left;
            z-index: 5;
        }
        
        .module-row:hover td {
            background-color: #f3f4f6;
        }
        
        .check-group {
            display: flex;
            justify-content: center;
            gap: 8px;
        }
        
        .perm-check {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        .role-header {
            min-width: 120px;
        }
        
        .action-label {
            font-size: 10px;
            color: #6b7280;
            display: block;
            margin-top: 2px;
        }
        
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255,255,255,0.8);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
            font-size: 20px;
            color: #1e3c72;
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container" style="padding: 20px;">
        <div class="header" style="margin-bottom: 20px;">
            <h1>🛡️ Gestión de Permisos</h1>
            <p style="color: #6b7280;">Configura los accesos de cada rol a los módulos del sistema</p>
        </div>
        
        <div id="tableContainer">
            <!-- La tabla se generará dinámicamente -->
        </div>
    </div>
    
    <div id="loading" class="loading-overlay">
        Cargando permisos...
    </div>

    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let roles = [];
        let modulos = [];
        let permisos = {};
        
        // Cargar datos
        async function loadData() {
            try {
                const response = await fetch('api.php');
                const data = await response.json();
                
                if (data.success) {
                    roles = data.roles;
                    modulos = data.modulos;
                    permisos = data.permisos;
                    renderTable();
                } else {
                    alert('Error al cargar datos: ' + data.error);
                }
            } catch (error) {
                console.error(error);
                alert('Error de conexión');
            } finally {
                document.getElementById('loading').style.display = 'none';
            }
        }
        
        // Renderizar tabla
        function renderTable() {
            let html = '<table class="permissions-table">';
            
            // Header
            html += '<thead><tr><th>Módulo</th>';
            roles.forEach(rol => {
                html += `<th class="role-header">
                    ${rol.nombre}
                    <div style="font-size: 11px; font-weight: normal; color: #6b7280;">Nivel ${rol.nivel}</div>
                </th>`;
            });
            html += '</tr></thead>';
            
            // Body
            html += '<tbody>';
            modulos.forEach(modulo => {
                html += `<tr class="module-row">
                    <td>${capitalize(modulo)}</td>`;
                
                roles.forEach(rol => {
                    // Si es admin (nivel >= 100), mostrar check deshabilitado y marcado
                    const isAdmin = parseInt(rol.nivel) >= 100;
                    const rolPermisos = permisos[rol.id] && permisos[rol.id][modulo] ? permisos[rol.id][modulo] : {ver:0, crear:0, editar:0, eliminar:0};
                    
                    html += `<td>
                        <div class="check-group">
                            <div title="Ver">
                                <input type="checkbox" class="perm-check" 
                                    ${isAdmin || rolPermisos.ver ? 'checked' : ''} 
                                    ${isAdmin ? 'disabled' : ''}
                                    onchange="updatePermiso(${rol.id}, '${modulo}', 'ver', this.checked)">
                                <span class="action-label">Ver</span>
                            </div>
                            <div title="Crear">
                                <input type="checkbox" class="perm-check" 
                                    ${isAdmin || rolPermisos.crear ? 'checked' : ''} 
                                    ${isAdmin ? 'disabled' : ''}
                                    onchange="updatePermiso(${rol.id}, '${modulo}', 'crear', this.checked)">
                                <span class="action-label">Crear</span>
                            </div>
                            <div title="Editar">
                                <input type="checkbox" class="perm-check" 
                                    ${isAdmin || rolPermisos.editar ? 'checked' : ''} 
                                    ${isAdmin ? 'disabled' : ''}
                                    onchange="updatePermiso(${rol.id}, '${modulo}', 'editar', this.checked)">
                                <span class="action-label">Edit</span>
                            </div>
                            <div title="Eliminar">
                                <input type="checkbox" class="perm-check" 
                                    ${isAdmin || rolPermisos.eliminar ? 'checked' : ''} 
                                    ${isAdmin ? 'disabled' : ''}
                                    onchange="updatePermiso(${rol.id}, '${modulo}', 'eliminar', this.checked)">
                                <span class="action-label">Del</span>
                            </div>
                        </div>
                    </td>`;
                });
                
                html += '</tr>';
            });
            html += '</tbody></table>';
            
            document.getElementById('tableContainer').innerHTML = html;
        }
        
        // Actualizar permiso
        async function updatePermiso(rolId, modulo, accion, valor) {
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        rol_id: rolId,
                        modulo: modulo,
                        accion: accion,
                        valor: valor ? 1 : 0
                    })
                });
                
                const result = await response.json();
                
                if (!result.success) {
                    alert('Error al guardar: ' + result.error);
                    // Revertir cambio visualmente (recargar tabla es drástico, mejor solo alertar)
                    loadData(); 
                }
            } catch (error) {
                console.error(error);
                alert('Error de conexión al guardar');
            }
        }
        
        function capitalize(str) {
            return str.charAt(0).toUpperCase() + str.slice(1).replace('_', ' ');
        }
        
        // Iniciar
        loadData();
    </script>
</body>
</html>
