<?php
/**
 * LexiPro - API de Gestión de Permisos
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

// Solo administradores pueden gestionar permisos
if ($_SESSION['user_rol_nivel'] < 100) {
    jsonError('Acceso denegado. Se requieren permisos de administrador.', 403);
}

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Obtener matriz de permisos
    if ($method === 'GET') {
        // Obtener roles
        $roles = $db->fetchAll("SELECT * FROM roles ORDER BY nivel DESC");
        
        // Obtener módulos (definidos en el sistema)
        $modulos = [
            'usuarios', 'casos', 'clientes', 'agenda', 'tareas', 
            'chat', 'facturacion', 'compras', 'proveedores', 
            'contabilidad', 'rrhh', 'ia', 'biblioteca', 
            'reportes', 'archivos', 'configuracion', 'contenido_web', 'notificaciones'
        ];
        
        // Obtener permisos actuales
        $permisos_db = $db->fetchAll("SELECT * FROM permisos");
        
        // Organizar permisos en una matriz [rol_id][modulo] = {ver, crear, ...}
        $matriz = [];
        foreach ($permisos_db as $p) {
            $matriz[$p['rol_id']][$p['modulo']] = [
                'ver' => (int)$p['ver'],
                'crear' => (int)$p['crear'],
                'editar' => (int)$p['editar'],
                'eliminar' => (int)$p['eliminar']
            ];
        }
        
        jsonSuccess([
            'roles' => $roles,
            'modulos' => $modulos,
            'permisos' => $matriz
        ]);
    }
    
    // POST - Actualizar permiso
    elseif ($method === 'POST') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $rolId = $data['rol_id'] ?? 0;
        $modulo = $data['modulo'] ?? '';
        $accion = $data['accion'] ?? ''; // ver, crear, editar, eliminar
        $valor = isset($data['valor']) ? (int)$data['valor'] : 0;
        
        if (empty($rolId) || empty($modulo) || empty($accion)) {
            jsonError('Datos incompletos');
        }
        
        // Verificar si existe el registro
        $existe = $db->fetchOne(
            "SELECT id FROM permisos WHERE rol_id = ? AND modulo = ?",
            [$rolId, $modulo]
        );
        
        if ($existe) {
            // Actualizar
            $db->update('permisos', [$accion => $valor], 'id = ?', [$existe['id']]);
        } else {
            // Insertar nuevo registro con el permiso activado y los demás en 0
            $nuevoPermiso = [
                'rol_id' => $rolId,
                'modulo' => $modulo,
                'ver' => 0,
                'crear' => 0,
                'editar' => 0,
                'eliminar' => 0
            ];
            $nuevoPermiso[$accion] = $valor;
            
            $db->insert('permisos', $nuevoPermiso);
        }
        
        // Log
        logActivity($userId, 'actualizar', 'permisos', $rolId, "Permiso $modulo.$accion = $valor");
        
        jsonSuccess([], 'Permiso actualizado');
    }
    
} catch (Exception $e) {
    error_log("Error en API permisos: " . $e->getMessage());
    jsonError('Error del servidor: ' . $e->getMessage(), 500);
}
