<?php
/**
 * LexiPro - Asistente IA
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';
require_once 'prompts.php';

$auth = new Auth();
$auth->requireLogin();

$prompts = PromptManager::getPrompts();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Asistente IA - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <script src="https://cdn.jsdelivr.net/npm/marked/marked.min.js"></script>
    <style>
        .ia-container {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 20px;
            height: calc(100vh - 140px);
        }
        
        .sidebar-ia {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            display: flex;
            flex-direction: column;
        }
        
        .chat-area {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }
        
        .chat-messages {
            flex: 1;
            padding: 20px;
            overflow-y: auto;
            display: flex;
            flex-direction: column;
            gap: 20px;
            background: #f9fafb;
        }
        
        .message {
            max-width: 80%;
            padding: 15px;
            border-radius: 12px;
            line-height: 1.5;
            position: relative;
        }
        
        .message.user {
            align-self: flex-end;
            background: #3b82f6;
            color: white;
            border-bottom-right-radius: 2px;
        }
        
        .message.model {
            align-self: flex-start;
            background: white;
            border: 1px solid #e5e7eb;
            color: #1f2937;
            border-bottom-left-radius: 2px;
        }
        
        .message.model h1, .message.model h2, .message.model h3 {
            margin-top: 10px;
            margin-bottom: 5px;
            font-size: 1.1em;
        }
        
        .message.model ul, .message.model ol {
            padding-left: 20px;
        }
        
        .chat-input-area {
            padding: 20px;
            background: white;
            border-top: 1px solid #e5e7eb;
            display: flex;
            gap: 10px;
        }
        
        .chat-input {
            flex: 1;
            padding: 12px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            resize: none;
            font-family: inherit;
        }
        
        .chat-input:focus {
            outline: none;
            border-color: #3b82f6;
        }
        
        .tool-btn {
            display: block;
            width: 100%;
            padding: 10px;
            text-align: left;
            background: none;
            border: 1px solid #e5e7eb;
            border-radius: 6px;
            margin-bottom: 8px;
            cursor: pointer;
            transition: all 0.2s;
            font-size: 14px;
        }
        
        .tool-btn:hover {
            background: #f3f4f6;
            border-color: #d1d5db;
        }
        
        .tool-btn span {
            margin-right: 8px;
        }
        
        .typing-indicator {
            display: none;
            padding: 10px;
            color: #6b7280;
            font-style: italic;
            font-size: 13px;
        }
        
        /* Markdown styles */
        .markdown-body {
            font-size: 15px;
        }
        .markdown-body pre {
            background: #f3f4f6;
            padding: 10px;
            border-radius: 6px;
            overflow-x: auto;
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container" style="max-width: 1400px;">
        <div class="header-flex">
            <div>
                <h1>🤖 Asistente Jurídico IA</h1>
                <p>Potenciado por Gemini 1.5</p>
            </div>
            <button class="btn btn-secondary" onclick="clearChat()">🗑️ Limpiar Chat</button>
        </div>
        
        <div class="ia-container">
            <!-- Sidebar -->
            <aside class="sidebar-ia">
                <h3 style="margin-bottom: 15px; font-size: 16px;">🛠️ Herramientas Rápidas</h3>
                
                <div style="flex: 1; overflow-y: auto;">
                    <div style="margin-bottom: 20px;">
                        <h4 style="font-size: 12px; text-transform: uppercase; color: #6b7280; margin-bottom: 10px;">Generar Documentos</h4>
                        <?php foreach ($prompts as $key => $prompt): ?>
                            <button class="tool-btn" onclick="usePrompt('<?= $key ?>')">
                                <span>📄</span> <?= $prompt['label'] ?>
                            </button>
                        <?php endforeach; ?>
                    </div>
                    
                    <div>
                        <h4 style="font-size: 12px; text-transform: uppercase; color: #6b7280; margin-bottom: 10px;">Análisis</h4>
                        <button class="tool-btn" onclick="document.getElementById('fileInput').click()">
                            <span>🔍</span> Analizar Documento (Texto)
                        </button>
                        <input type="file" id="fileInput" accept=".txt" style="display: none;" onchange="handleFileUpload(this)">
                    </div>
                </div>
                
                <div class="alert alert-info" style="font-size: 12px; margin-top: 20px;">
                    <strong>Nota:</strong> La IA puede cometer errores. Verifica siempre la información legal generada.
                </div>
            </aside>
            
            <!-- Chat Area -->
            <main class="chat-area">
                <div id="chatMessages" class="chat-messages">
                    <div class="message model">
                        Hola, soy tu asistente jurídico virtual. ¿En qué puedo ayudarte hoy?
                        <br><br>
                        Puedo ayudarte a:
                        <ul>
                            <li>Redactar borradores de escritos.</li>
                            <li>Analizar textos legales.</li>
                            <li>Buscar conceptos jurídicos.</li>
                        </ul>
                    </div>
                </div>
                
                <div class="typing-indicator" id="typingIndicator">
                    Gemini está escribiendo...
                </div>
                
                <div class="chat-input-area">
                    <textarea id="userInput" class="chat-input" rows="2" placeholder="Escribe tu consulta aquí..." onkeydown="if(event.key === 'Enter' && !event.shiftKey) { event.preventDefault(); sendMessage(); }"></textarea>
                    <button class="btn btn-primary" onclick="sendMessage()" style="height: auto;">Enviar ➤</button>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Modal Contexto -->
    <div class="modal" id="contextModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Contexto para Generación</h2>
                <button class="modal-close" onclick="closeContextModal()">×</button>
            </div>
            <div class="modal-body">
                <p style="margin-bottom: 15px;">Proporciona los detalles del caso para generar el documento:</p>
                <input type="hidden" id="currentPromptType">
                <textarea id="contextInput" rows="6" style="width: 100%; padding: 10px; border: 2px solid #e5e7eb; border-radius: 8px;" placeholder="Ej: Cliente Juan Pérez, demandado Empresa X, monto 50.000.000 Gs, concepto facturas impagas..."></textarea>
                <button class="btn btn-primary" style="width: 100%; margin-top: 15px;" onclick="confirmGeneration()">Generar Documento</button>
            </div>
        </div>
    </div>

    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let chatHistory = [];
        
        function scrollToBottom() {
            const container = document.getElementById('chatMessages');
            container.scrollTop = container.scrollHeight;
        }
        
        function addMessage(role, content) {
            const container = document.getElementById('chatMessages');
            const div = document.createElement('div');
            div.className = `message ${role}`;
            
            if (role === 'model') {
                div.innerHTML = marked.parse(content);
                div.classList.add('markdown-body');
            } else {
                div.textContent = content;
            }
            
            container.appendChild(div);
            scrollToBottom();
            
            // Guardar en historial (limitado a últimos 10 mensajes para contexto)
            chatHistory.push({ role, content });
            if (chatHistory.length > 10) chatHistory.shift();
        }
        
        async function sendMessage() {
            const input = document.getElementById('userInput');
            const message = input.value.trim();
            
            if (!message) return;
            
            input.value = '';
            addMessage('user', message);
            
            document.getElementById('typingIndicator').style.display = 'block';
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        action: 'chat',
                        message: message,
                        history: chatHistory.slice(0, -1) // Enviar historial previo
                    })
                });
                
                const data = await response.json();
                
                document.getElementById('typingIndicator').style.display = 'none';
                
                if (data.success) {
                    addMessage('model', data.response);
                } else {
                    addMessage('model', '❌ Error: ' + data.error);
                }
            } catch (error) {
                document.getElementById('typingIndicator').style.display = 'none';
                addMessage('model', '❌ Error de conexión');
            }
        }
        
        function usePrompt(type) {
            document.getElementById('currentPromptType').value = type;
            document.getElementById('contextModal').classList.add('active');
            document.getElementById('contextInput').focus();
        }
        
        function closeContextModal() {
            document.getElementById('contextModal').classList.remove('active');
            document.getElementById('contextInput').value = '';
        }
        
        async function confirmGeneration() {
            const type = document.getElementById('currentPromptType').value;
            const context = document.getElementById('contextInput').value;
            
            if (!context) {
                alert('Por favor ingresa el contexto del caso');
                return;
            }
            
            closeContextModal();
            addMessage('user', `Generar documento: ${type}\nContexto: ${context}`);
            document.getElementById('typingIndicator').style.display = 'block';
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        action: 'generate_document',
                        type: type,
                        context: context
                    })
                });
                
                const data = await response.json();
                document.getElementById('typingIndicator').style.display = 'none';
                
                if (data.success) {
                    addMessage('model', data.content);
                } else {
                    addMessage('model', '❌ Error: ' + data.error);
                }
            } catch (error) {
                document.getElementById('typingIndicator').style.display = 'none';
                addMessage('model', '❌ Error de conexión');
            }
        }
        
        function handleFileUpload(input) {
            const file = input.files[0];
            if (!file) return;
            
            const reader = new FileReader();
            reader.onload = async function(e) {
                const text = e.target.result;
                addMessage('user', `Analizar documento adjunto (${file.name})`);
                document.getElementById('typingIndicator').style.display = 'block';
                
                try {
                    const response = await fetch('api.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify({
                            action: 'analyze_text',
                            text: text
                        })
                    });
                    
                    const data = await response.json();
                    document.getElementById('typingIndicator').style.display = 'none';
                    
                    if (data.success) {
                        addMessage('model', data.analysis);
                    } else {
                        addMessage('model', '❌ Error: ' + data.error);
                    }
                } catch (error) {
                    document.getElementById('typingIndicator').style.display = 'none';
                    addMessage('model', '❌ Error de conexión');
                }
            };
            reader.readAsText(file);
            input.value = ''; // Reset
        }
        
        function clearChat() {
            if (confirm('¿Borrar historial de chat?')) {
                document.getElementById('chatMessages').innerHTML = '';
                chatHistory = [];
                addMessage('model', 'Chat reiniciado. ¿En qué puedo ayudarte?');
            }
        }
    </script>
</body>
</html>
