<?php
/**
 * LexiPro - Cliente de Google Gemini API
 */

class GeminiClient {
    private $apiKey;
    private $model;
    private $baseUrl = 'https://generativelanguage.googleapis.com/v1beta/models/';
    private $temperature;
    private $maxTokens;
    private $systemPrompt;

    public function __construct() {
        $db = Database::getInstance();
        
        // Cargar configuración
        $config = $db->fetchAll("SELECT clave, valor FROM configuracion WHERE clave LIKE 'gemini_%' OR clave = 'system_prompt'");
        $settings = [];
        foreach ($config as $row) {
            $settings[$row['clave']] = $row['valor'];
        }
        
        $this->apiKey = $settings['gemini_api_key'] ?? '';
        $this->model = $settings['gemini_model'] ?? 'gemini-1.5-flash';
        $this->temperature = (float)($settings['gemini_temperature'] ?? 0.7);
        $this->maxTokens = (int)($settings['gemini_max_tokens'] ?? 2048);
        $this->systemPrompt = $settings['system_prompt'] ?? 'Eres un asistente útil.';
    }

    public function isConfigured() {
        return !empty($this->apiKey);
    }

    public function generateContent($prompt, $history = []) {
        if (!$this->isConfigured()) {
            throw new Exception("API Key de Gemini no configurada");
        }

        $url = $this->baseUrl . $this->model . ':generateContent?key=' . $this->apiKey;

        // Construir el cuerpo de la solicitud
        $contents = [];
        
        // Agregar historial si existe
        if (!empty($history)) {
            foreach ($history as $msg) {
                $contents[] = [
                    'role' => $msg['role'] === 'user' ? 'user' : 'model',
                    'parts' => [['text' => $msg['content']]]
                ];
            }
        }
        
        // Agregar prompt actual
        $contents[] = [
            'role' => 'user',
            'parts' => [['text' => $prompt]]
        ];

        $data = [
            'contents' => $contents,
            'generationConfig' => [
                'temperature' => $this->temperature,
                'maxOutputTokens' => $this->maxTokens,
            ],
            'systemInstruction' => [
                'parts' => [['text' => $this->systemPrompt]]
            ]
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_errno($ch)) {
            throw new Exception('Error de conexión: ' . curl_error($ch));
        }
        
        curl_close($ch);
        
        $result = json_decode($response, true);
        
        if ($httpCode !== 200) {
            $errorMsg = $result['error']['message'] ?? 'Error desconocido de la API';
            throw new Exception("Error API ($httpCode): $errorMsg");
        }
        
        return $result['candidates'][0]['content']['parts'][0]['text'] ?? '';
    }
}
