<?php
/**
 * LexiPro - Gestión de Facturas
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Breadcrumbs
$breadcrumbs = [
    ['label' => 'Facturación']
];

// Obtener clientes para el selector
$clientes = $db->fetchAll("SELECT id, nombre, apellido, razon_social, tipo_persona, ruc FROM clientes WHERE activo = 1 ORDER BY nombre LIMIT 200");

// Obtener casos para vincular
$casos = $db->fetchAll("SELECT id, numero_caso, nombre FROM casos WHERE estado != 'archivado' ORDER BY created_at DESC LIMIT 100");
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Facturación - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .invoice-header {
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            color: white;
            padding: 30px;
            border-radius: 12px 12px 0 0;
            margin-bottom: 0;
        }
        
        .invoice-stats {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .stat-box {
            background: white;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
        }
        
        .stat-value {
            font-size: 28px;
            font-weight: 700;
            color: #1e3c72;
            margin-bottom: 5px;
        }
        
        .stat-label {
            font-size: 13px;
            color: #6b7280;
        }
        
        .detalle-row {
            display: grid;
            grid-template-columns: 3fr 1fr 1fr 1fr 1fr 50px;
            gap: 10px;
            margin-bottom: 10px;
            align-items: center;
        }
        
        .detalle-row input,
        .detalle-row select {
            padding: 10px;
            border: 2px solid #e5e7eb;
            border-radius: 6px;
        }
        
        .btn-remove {
            background: #ef4444;
            color: white;
            border: none;
            padding: 10px;
            border-radius: 6px;
            cursor: pointer;
        }
        
        .totales-box {
            background: #f9fafb;
            padding: 20px;
            border-radius: 8px;
            margin-top: 20px;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #e5e7eb;
        }
        
        .total-row:last-child {
            border-bottom: none;
            font-size: 18px;
            font-weight: 700;
            color: #1e3c72;
            padding-top: 15px;
            border-top: 2px solid #1e3c72;
        }
        
        .estado-badge {
            padding: 6px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .estado-pagada { background: #d1fae5; color: #065f46; }
        .estado-pendiente { background: #fef3c7; color: #92400e; }
        .estado-parcial { background: #dbeafe; color: #1e40af; }
        .estado-anulada { background: #fee2e2; color: #991b1b; }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="invoice-header">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <h1>📄 Facturación</h1>
                    <p style="opacity: 0.9; margin-top: 5px;">Sistema de facturación RG90 Paraguay</p>
                </div>
                <button class="btn btn-primary" onclick="openInvoiceModal()" style="background: white; color: #1e3c72;">
                    + Nueva Factura
                </button>
            </div>
        </div>
        
        <div class="invoice-stats">
            <div class="stat-box">
                <div class="stat-value" id="totalFacturado">Gs. 0</div>
                <div class="stat-label">Total Facturado</div>
            </div>
            <div class="stat-box">
                <div class="stat-value" id="totalCobrado">Gs. 0</div>
                <div class="stat-label">Total Cobrado</div>
            </div>
            <div class="stat-box">
                <div class="stat-value" id="totalPendiente">Gs. 0</div>
                <div class="stat-label">Por Cobrar</div>
            </div>
            <div class="stat-box">
                <div class="stat-value" id="cantidadFacturas">0</div>
                <div class="stat-label">Facturas del Mes</div>
            </div>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>🔍 Buscar</label>
                <input type="text" id="searchInput" placeholder="Número de factura, cliente...">
            </div>
            <div class="filter-group">
                <label>Cliente</label>
                <select id="clienteFilter">
                    <option value="">Todos los clientes</option>
                    <?php foreach ($clientes as $cliente): ?>
                        <?php 
                        $nombreCliente = $cliente['tipo_persona'] === 'juridica' 
                            ? $cliente['razon_social'] 
                            : $cliente['nombre'] . ' ' . ($cliente['apellido'] ?? '');
                        ?>
                        <option value="<?= $cliente['id'] ?>"><?= e($nombreCliente) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label>Estado</label>
                <select id="estadoFilter">
                    <option value="">Todos</option>
                    <option value="pagada">Pagada</option>
                    <option value="pendiente">Pendiente</option>
                    <option value="parcial">Parcial</option>
                    <option value="anulada">Anulada</option>
                </select>
            </div>
            <div class="filter-group">
                <label>Desde</label>
                <input type="date" id="fechaDesde">
            </div>
            <div class="filter-group">
                <label>Hasta</label>
                <input type="date" id="fechaHasta">
            </div>
        </div>
        
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>Número</th>
                        <th>Fecha</th>
                        <th>Cliente</th>
                        <th>RUC</th>
                        <th>Condición</th>
                        <th>Total</th>
                        <th>Cobrado</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="facturasTableBody">
                    <tr>
                        <td colspan="9" class="loading">Cargando facturas...</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Modal de Factura -->
    <div class="modal" id="invoiceModal">
        <div class="modal-content modal-large" style="max-width: 1000px;">
            <div class="modal-header">
                <h2>Nueva Factura</h2>
                <button class="modal-close" onclick="closeInvoiceModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="invoiceForm">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Cliente *</label>
                            <select name="cliente_id" required id="clienteSelect" onchange="loadClienteData()">
                                <option value="">Seleccionar cliente...</option>
                                <?php foreach ($clientes as $cliente): ?>
                                    <?php 
                                    $nombreCliente = $cliente['tipo_persona'] === 'juridica' 
                                        ? $cliente['razon_social'] 
                                        : $cliente['nombre'] . ' ' . ($cliente['apellido'] ?? '');
                                    ?>
                                    <option value="<?= $cliente['id'] ?>" data-ruc="<?= $cliente['ruc'] ?? '' ?>">
                                        <?= e($nombreCliente) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>RUC Cliente</label>
                            <input type="text" id="clienteRuc" readonly style="background: #f3f4f6;">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Fecha *</label>
                            <input type="date" name="fecha" required value="<?= date('Y-m-d') ?>">
                        </div>
                        <div class="form-group">
                            <label>Condición de Venta *</label>
                            <select name="condicion_venta" required id="condicionVenta" onchange="toggleFormaPago()">
                                <option value="contado">Contado</option>
                                <option value="credito">Crédito</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Caso (opcional)</label>
                            <select name="caso_id">
                                <option value="">Sin caso asociado</option>
                                <?php foreach ($casos as $caso): ?>
                                    <option value="<?= $caso['id'] ?>"><?= e($caso['numero_caso'] . ' - ' . $caso['nombre']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div id="formaPagoSection">
                        <div class="form-row">
                            <div class="form-group">
                                <label>Forma de Pago *</label>
                                <select name="forma_pago" id="formaPago">
                                    <option value="efectivo">Efectivo</option>
                                    <option value="transferencia">Transferencia</option>
                                    <option value="cheque">Cheque</option>
                                    <option value="tarjeta">Tarjeta</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Referencia</label>
                                <input type="text" name="referencia_pago" placeholder="Nro. de transferencia, cheque, etc.">
                            </div>
                        </div>
                    </div>
                    
                    <hr style="margin: 20px 0;">
                    
                    <h3 style="margin-bottom: 15px;">Detalles de la Factura</h3>
                    
                    <div id="detallesContainer">
                        <div class="detalle-row">
                            <div><strong>Descripción</strong></div>
                            <div><strong>Cantidad</strong></div>
                            <div><strong>Precio Unit.</strong></div>
                            <div><strong>IVA</strong></div>
                            <div><strong>Subtotal</strong></div>
                            <div></div>
                        </div>
                    </div>
                    
                    <button type="button" class="btn btn-secondary" onclick="addDetalle()" style="margin-top: 10px;">
                        + Agregar Ítem
                    </button>
                    
                    <div class="totales-box">
                        <div class="total-row">
                            <span>Subtotal Gravado 5%:</span>
                            <span id="subtotal5">Gs. 0</span>
                        </div>
                        <div class="total-row">
                            <span>Subtotal Gravado 10%:</span>
                            <span id="subtotal10">Gs. 0</span>
                        </div>
                        <div class="total-row">
                            <span>Subtotal Exento:</span>
                            <span id="subtotalExento">Gs. 0</span>
                        </div>
                        <div class="total-row">
                            <span>IVA 5%:</span>
                            <span id="iva5">Gs. 0</span>
                        </div>
                        <div class="total-row">
                            <span>IVA 10%:</span>
                            <span id="iva10">Gs. 0</span>
                        </div>
                        <div class="total-row">
                            <span>TOTAL GENERAL:</span>
                            <span id="totalGeneral">Gs. 0</span>
                        </div>
                    </div>
                    
                    <div class="form-group" style="margin-top: 20px;">
                        <label>Observaciones</label>
                        <textarea name="observaciones" rows="2" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                        Generar Factura
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let facturas = [];
        let detalleCounter = 0;
        
        // Cargar facturas
        async function loadFacturas() {
            try {
                const params = new URLSearchParams();
                const search = document.getElementById('searchInput').value;
                const cliente = document.getElementById('clienteFilter').value;
                const estado = document.getElementById('estadoFilter').value;
                const fechaDesde = document.getElementById('fechaDesde').value;
                const fechaHasta = document.getElementById('fechaHasta').value;
                
                if (search) params.append('search', search);
                if (cliente) params.append('cliente', cliente);
                if (estado) params.append('estado', estado);
                if (fechaDesde) params.append('fecha_desde', fechaDesde);
                if (fechaHasta) params.append('fecha_hasta', fechaHasta);
                
                const response = await fetch(`api.php?${params}`);
                const data = await response.json();
                
                if (data.success) {
                    facturas = data.facturas;
                    renderFacturas();
                    updateStats();
                }
            } catch (error) {
                console.error('Error al cargar facturas:', error);
            }
        }
        
        // Renderizar facturas
        function renderFacturas() {
            const tbody = document.getElementById('facturasTableBody');
            
            if (facturas.length === 0) {
                tbody.innerHTML = '<tr><td colspan="9" class="loading">No se encontraron facturas</td></tr>';
                return;
            }
            
            tbody.innerHTML = facturas.map(factura => {
                const nombreCliente = factura.razon_social || `${factura.cliente_nombre} ${factura.cliente_apellido || ''}`;
                const cobrado = parseFloat(factura.total_cobrado || 0);
                const total = parseFloat(factura.total_general);
                const saldo = total - cobrado;
                
                return `
                    <tr>
                        <td><strong>${factura.numero_factura}</strong></td>
                        <td>${LexiPro.formatDate(factura.fecha)}</td>
                        <td>${nombreCliente}</td>
                        <td>${factura.ruc || '-'}</td>
                        <td><span class="badge badge-info">${factura.condicion_venta === 'contado' ? 'Contado' : 'Crédito'}</span></td>
                        <td>${LexiPro.formatMoney(total)}</td>
                        <td>${LexiPro.formatMoney(cobrado)}</td>
                        <td><span class="estado-badge estado-${factura.estado}">${factura.estado.toUpperCase()}</span></td>
                        <td>
                            <div class="actions">
                                <button class="btn btn-sm btn-edit" onclick="viewFactura(${factura.id})">Ver</button>
                                ${factura.estado !== 'anulada' && factura.estado !== 'pagada' ? 
                                    `<button class="btn btn-sm btn-delete" onclick="anularFactura(${factura.id})">Anular</button>` : 
                                    ''}
                            </div>
                        </td>
                    </tr>
                `;
            }).join('');
        }
        
        // Actualizar estadísticas
        function updateStats() {
            const mesActual = new Date().getMonth();
            const añoActual = new Date().getFullYear();
            
            const facturasMes = facturas.filter(f => {
                const fecha = new Date(f.fecha);
                return fecha.getMonth() === mesActual && 
                       fecha.getFullYear() === añoActual &&
                       f.estado !== 'anulada';
            });
            
            const totalFacturado = facturasMes.reduce((sum, f) => sum + parseFloat(f.total_general), 0);
            const totalCobrado = facturasMes.reduce((sum, f) => sum + parseFloat(f.total_cobrado || 0), 0);
            const totalPendiente = totalFacturado - totalCobrado;
            
            document.getElementById('totalFacturado').textContent = LexiPro.formatMoney(totalFacturado);
            document.getElementById('totalCobrado').textContent = LexiPro.formatMoney(totalCobrado);
            document.getElementById('totalPendiente').textContent = LexiPro.formatMoney(totalPendiente);
            document.getElementById('cantidadFacturas').textContent = facturasMes.length;
        }
        
        // Abrir modal
        function openInvoiceModal() {
            document.getElementById('invoiceModal').classList.add('active');
            addDetalle(); // Agregar primera línea
        }
        
        // Cerrar modal
        function closeInvoiceModal() {
            document.getElementById('invoiceModal').classList.remove('active');
            document.getElementById('invoiceForm').reset();
            document.getElementById('detallesContainer').innerHTML = `
                <div class="detalle-row">
                    <div><strong>Descripción</strong></div>
                    <div><strong>Cantidad</strong></div>
                    <div><strong>Precio Unit.</strong></div>
                    <div><strong>IVA</strong></div>
                    <div><strong>Subtotal</strong></div>
                    <div></div>
                </div>
            `;
            detalleCounter = 0;
        }
        
        // Cargar datos del cliente
        function loadClienteData() {
            const select = document.getElementById('clienteSelect');
            const option = select.options[select.selectedIndex];
            document.getElementById('clienteRuc').value = option.dataset.ruc || '';
        }
        
        // Toggle forma de pago
        function toggleFormaPago() {
            const condicion = document.getElementById('condicionVenta').value;
            const section = document.getElementById('formaPagoSection');
            const formaPago = document.getElementById('formaPago');
            
            if (condicion === 'contado') {
                section.style.display = 'block';
                formaPago.required = true;
            } else {
                section.style.display = 'none';
                formaPago.required = false;
            }
        }
        
        // Agregar detalle
        function addDetalle() {
            detalleCounter++;
            const container = document.getElementById('detallesContainer');
            const row = document.createElement('div');
            row.className = 'detalle-row';
            row.id = `detalle-${detalleCounter}`;
            row.innerHTML = `
                <input type="text" placeholder="Descripción del servicio" data-field="descripcion" required>
                <input type="number" placeholder="1" value="1" min="1" step="0.01" data-field="cantidad" onchange="calcularTotales()">
                <input type="number" placeholder="0" min="0" step="1000" data-field="precio_unitario" onchange="calcularTotales()">
                <select data-field="tipo_iva" onchange="calcularTotales()">
                    <option value="10">10%</option>
                    <option value="5">5%</option>
                    <option value="exento">Exento</option>
                </select>
                <input type="number" readonly style="background: #f3f4f6;" data-field="subtotal" value="0">
                <button type="button" class="btn-remove" onclick="removeDetalle(${detalleCounter})">×</button>
            `;
            container.appendChild(row);
        }
        
        // Remover detalle
        function removeDetalle(id) {
            const row = document.getElementById(`detalle-${id}`);
            if (row) {
                row.remove();
                calcularTotales();
            }
        }
        
        // Calcular totales
        function calcularTotales() {
            let subtotal5 = 0, subtotal10 = 0, subtotalExento = 0;
            
            document.querySelectorAll('.detalle-row').forEach((row, index) => {
                if (index === 0) return; // Skip header
                
                const cantidad = parseFloat(row.querySelector('[data-field="cantidad"]')?.value || 0);
                const precio = parseFloat(row.querySelector('[data-field="precio_unitario"]')?.value || 0);
                const tipoIva = row.querySelector('[data-field="tipo_iva"]')?.value;
                const subtotal = cantidad * precio;
                
                const subtotalField = row.querySelector('[data-field="subtotal"]');
                if (subtotalField) subtotalField.value = subtotal.toFixed(0);
                
                switch(tipoIva) {
                    case '5': subtotal5 += subtotal; break;
                    case '10': subtotal10 += subtotal; break;
                    case 'exento': subtotalExento += subtotal; break;
                }
            });
            
            const iva5 = subtotal5 * 0.05;
            const iva10 = subtotal10 * 0.10;
            const totalGeneral = subtotal5 + subtotal10 + subtotalExento + iva5 + iva10;
            
            document.getElementById('subtotal5').textContent = LexiPro.formatMoney(subtotal5);
            document.getElementById('subtotal10').textContent = LexiPro.formatMoney(subtotal10);
            document.getElementById('subtotalExento').textContent = LexiPro.formatMoney(subtotalExento);
            document.getElementById('iva5').textContent = LexiPro.formatMoney(iva5);
            document.getElementById('iva10').textContent = LexiPro.formatMoney(iva10);
            document.getElementById('totalGeneral').textContent = LexiPro.formatMoney(totalGeneral);
        }
        
        // Guardar factura
        document.getElementById('invoiceForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {
                cliente_id: formData.get('cliente_id'),
                fecha: formData.get('fecha'),
                condicion_venta: formData.get('condicion_venta'),
                caso_id: formData.get('caso_id') || null,
                forma_pago: formData.get('forma_pago'),
                referencia_pago: formData.get('referencia_pago'),
                observaciones: formData.get('observaciones'),
                detalles: []
            };
            
            // Recopilar detalles
            document.querySelectorAll('.detalle-row').forEach((row, index) => {
                if (index === 0) return; // Skip header
                
                const detalle = {
                    descripcion: row.querySelector('[data-field="descripcion"]').value,
                    cantidad: parseFloat(row.querySelector('[data-field="cantidad"]').value),
                    precio_unitario: parseFloat(row.querySelector('[data-field="precio_unitario"]').value),
                    tipo_iva: row.querySelector('[data-field="tipo_iva"]').value
                };
                
                if (detalle.descripcion && detalle.cantidad > 0 && detalle.precio_unitario >= 0) {
                    data.detalles.push(detalle);
                }
            });
            
            if (data.detalles.length === 0) {
                alert('Debe agregar al menos un ítem a la factura');
                return;
            }
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(`Factura ${result.numero_factura} creada exitosamente`);
                    closeInvoiceModal();
                    loadFacturas();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al crear factura');
                console.error(error);
            }
        });
        
        // Event listeners
        document.getElementById('searchInput').addEventListener('input', LexiPro.debounce(loadFacturas, 300));
        document.getElementById('clienteFilter').addEventListener('change', loadFacturas);
        document.getElementById('estadoFilter').addEventListener('change', loadFacturas);
        document.getElementById('fechaDesde').addEventListener('change', loadFacturas);
        document.getElementById('fechaHasta').addEventListener('change', loadFacturas);
        
        // Cargar al inicio
        loadFacturas();
    </script>
</body>
</html>
