<?php
/**
 * LexiPro - API de Facturación
 * Sistema de facturación según RG90 Paraguay
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Listar facturas o obtener una específica
    if ($method === 'GET') {
        $auth->requirePermission('facturacion', 'ver');
        
        if (isset($_GET['id'])) {
            // Factura específica con detalles
            $factura = $db->fetchOne("SELECT * FROM facturas_ventas WHERE id = ?", [$_GET['id']]);
            
            if ($factura) {
                $detalles = $db->fetchAll("SELECT * FROM facturas_ventas_detalle WHERE factura_id = ?", [$_GET['id']]);
                $factura['detalles'] = $detalles;
                
                // Obtener cobros
                $cobros = $db->fetchAll("SELECT * FROM cobros WHERE factura_id = ? ORDER BY fecha", [$_GET['id']]);
                $factura['cobros'] = $cobros;
            }
            
            jsonSuccess(['factura' => $factura]);
        } else {
            // Listar facturas
            $search = $_GET['search'] ?? '';
            $cliente = $_GET['cliente'] ?? '';
            $estado = $_GET['estado'] ?? '';
            $fechaDesde = $_GET['fecha_desde'] ?? '';
            $fechaHasta = $_GET['fecha_hasta'] ?? '';
            
            $sql = "SELECT fv.*, c.nombre as cliente_nombre, c.apellido as cliente_apellido, c.razon_social,
                           (SELECT SUM(monto) FROM cobros WHERE factura_id = fv.id) as total_cobrado
                    FROM facturas_ventas fv
                    INNER JOIN clientes c ON fv.cliente_id = c.id
                    WHERE 1=1";
            $params = [];
            
            if ($search) {
                $sql .= " AND (fv.numero_factura LIKE ? OR c.nombre LIKE ? OR c.razon_social LIKE ?)";
                $searchParam = "%{$search}%";
                $params = array_fill(0, 3, $searchParam);
            }
            
            if ($cliente) {
                $sql .= " AND fv.cliente_id = ?";
                $params[] = $cliente;
            }
            
            if ($estado) {
                $sql .= " AND fv.estado = ?";
                $params[] = $estado;
            }
            
            if ($fechaDesde) {
                $sql .= " AND fv.fecha >= ?";
                $params[] = $fechaDesde;
            }
            
            if ($fechaHasta) {
                $sql .= " AND fv.fecha <= ?";
                $params[] = $fechaHasta;
            }
            
            $sql .= " ORDER BY fv.fecha DESC, fv.id DESC LIMIT 100";
            
            $facturas = $db->fetchAll($sql, $params);
            jsonSuccess(['facturas' => $facturas]);
        }
    }
    
    // POST - Crear factura
    elseif ($method === 'POST') {
        $auth->requirePermission('facturacion', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validaciones
        if (empty($data['cliente_id'])) {
            jsonError('Cliente es obligatorio');
        }
        
        if (empty($data['detalles']) || count($data['detalles']) == 0) {
            jsonError('Debe agregar al menos un ítem a la factura');
        }
        
        // Obtener configuración de timbrado activo
        $timbrado = $db->fetchOne("SELECT * FROM configuracion WHERE clave = 'timbrado_activo'");
        if (!$timbrado) {
            jsonError('No hay timbrado configurado. Configure el timbrado en Configuración.');
        }
        
        $timbradoData = json_decode($timbrado['valor'], true);
        
        // Generar número de factura
        $ultimaFactura = $db->fetchOne(
            "SELECT numero_factura FROM facturas_ventas WHERE timbrado = ? ORDER BY id DESC LIMIT 1",
            [$timbradoData['numero']]
        );
        
        if ($ultimaFactura) {
            // Extraer el número y sumar 1
            $partes = explode('-', $ultimaFactura['numero_factura']);
            $numero = (int)$partes[2] + 1;
        } else {
            $numero = 1;
        }
        
        $numeroFactura = sprintf(
            "%03d-%03d-%07d",
            $timbradoData['establecimiento'],
            $timbradoData['punto_expedicion'],
            $numero
        );
        
        // Iniciar transacción
        $db->beginTransaction();
        
        try {
            // Calcular totales
            $subtotal5 = 0;
            $subtotal10 = 0;
            $subtotalExento = 0;
            
            foreach ($data['detalles'] as $detalle) {
                $cantidad = (float)$detalle['cantidad'];
                $precioUnitario = (float)$detalle['precio_unitario'];
                $subtotal = $cantidad * $precioUnitario;
                
                switch ($detalle['tipo_iva']) {
                    case '10':
                        $subtotal10 += $subtotal;
                        break;
                    case '5':
                        $subtotal5 += $subtotal;
                        break;
                    case 'exento':
                        $subtotalExento += $subtotal;
                        break;
                }
            }
            
            $iva5 = $subtotal5 * 0.05;
            $iva10 = $subtotal10 * 0.10;
            $totalIva = $iva5 + $iva10;
            $totalGeneral = $subtotal5 + $subtotal10 + $subtotalExento + $totalIva;
            
            // Insertar factura
            $facturaId = $db->insert('facturas_ventas', [
                'numero_factura' => $numeroFactura,
                'timbrado' => $timbradoData['numero'],
                'fecha' => $data['fecha'] ?? date('Y-m-d'),
                'cliente_id' => $data['cliente_id'],
                'condicion_venta' => $data['condicion_venta'] ?? 'contado',
                'subtotal_5' => $subtotal5,
                'subtotal_10' => $subtotal10,
                'subtotal_exento' => $subtotalExento,
                'iva_5' => $iva5,
                'iva_10' => $iva10,
                'total_iva' => $totalIva,
                'total_general' => $totalGeneral,
                'estado' => $data['condicion_venta'] === 'contado' ? 'pagada' : 'pendiente',
                'caso_id' => $data['caso_id'] ?? null,
                'observaciones' => sanitize($data['observaciones'] ?? null),
                'usuario_id' => $userId
            ]);
            
            // Insertar detalles
            foreach ($data['detalles'] as $detalle) {
                $cantidad = (float)$detalle['cantidad'];
                $precioUnitario = (float)$detalle['precio_unitario'];
                $subtotal = $cantidad * $precioUnitario;
                
                $db->insert('facturas_ventas_detalle', [
                    'factura_id' => $facturaId,
                    'descripcion' => sanitize($detalle['descripcion']),
                    'cantidad' => $cantidad,
                    'precio_unitario' => $precioUnitario,
                    'tipo_iva' => $detalle['tipo_iva'],
                    'subtotal' => $subtotal
                ]);
            }
            
            // Si es contado, registrar cobro automático
            if ($data['condicion_venta'] === 'contado' && !empty($data['forma_pago'])) {
                $db->insert('cobros', [
                    'factura_id' => $facturaId,
                    'fecha' => $data['fecha'] ?? date('Y-m-d'),
                    'monto' => $totalGeneral,
                    'forma_pago' => $data['forma_pago'],
                    'referencia' => sanitize($data['referencia_pago'] ?? null),
                    'observaciones' => 'Cobro automático - Venta al contado',
                    'usuario_id' => $userId
                ]);
            }
            
            $db->commit();
            
            // Log de actividad
            logActivity($userId, 'crear', 'facturas', $facturaId, "Factura creada: {$numeroFactura}");
            
            jsonSuccess(['id' => $facturaId, 'numero_factura' => $numeroFactura], 'Factura creada exitosamente');
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    // PUT - Anular factura
    elseif ($method === 'PUT') {
        $auth->requirePermission('facturacion', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de factura requerido');
        }
        
        // Verificar que no tenga cobros
        $cobros = $db->fetchOne("SELECT COUNT(*) as total FROM cobros WHERE factura_id = ?", [$data['id']]);
        
        if ($cobros['total'] > 0) {
            jsonError('No se puede anular una factura con cobros registrados');
        }
        
        $db->update('facturas_ventas', [
            'estado' => 'anulada',
            'motivo_anulacion' => sanitize($data['motivo'] ?? 'Sin motivo especificado')
        ], 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'anular', 'facturas', $data['id'], "Factura anulada");
        
        jsonSuccess([], 'Factura anulada exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de facturación: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
