<?php
/**
 * LexiPro - Balance General
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Balance General - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .balance-container {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
        }
        
        .balance-section {
            background: white;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            padding: 20px;
        }
        
        .balance-header {
            padding-bottom: 15px;
            border-bottom: 2px solid #e5e7eb;
            margin-bottom: 15px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .balance-header h2 {
            margin: 0;
            font-size: 20px;
        }
        
        .account-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #f3f4f6;
        }
        
        .account-row:last-child {
            border-bottom: none;
        }
        
        .account-code {
            color: #6b7280;
            margin-right: 10px;
            font-family: monospace;
        }
        
        .total-row {
            margin-top: 20px;
            padding-top: 15px;
            border-top: 2px solid #e5e7eb;
            display: flex;
            justify-content: space-between;
            font-weight: 700;
            font-size: 18px;
        }
        
        .activo-header { color: #059669; border-color: #059669; }
        .pasivo-header { color: #dc2626; border-color: #dc2626; }
        .patrimonio-header { color: #2563eb; border-color: #2563eb; }
        
        @media print {
            .filters, .btn { display: none; }
            .container { width: 100%; max-width: none; }
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="header-flex">
            <div>
                <h1>⚖️ Balance General</h1>
                <p>Situación patrimonial de la empresa</p>
            </div>
            <button class="btn btn-secondary" onclick="window.print()">Imprimir Reporte</button>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>Fecha de Corte</label>
                <input type="date" id="fechaCorte" value="<?= date('Y-m-d') ?>">
            </div>
            <div class="filter-group" style="align-self: flex-end;">
                <button class="btn btn-primary" onclick="loadBalance()">Generar</button>
            </div>
        </div>
        
        <div id="balanceContent" style="display: none;">
            <div class="balance-container">
                <!-- Columna Izquierda: ACTIVO -->
                <div class="balance-section">
                    <div class="balance-header activo-header">
                        <h2>ACTIVO</h2>
                    </div>
                    <div id="activoList"></div>
                    <div class="total-row" style="color: #059669;">
                        <span>TOTAL ACTIVO</span>
                        <span id="totalActivo">Gs. 0</span>
                    </div>
                </div>
                
                <!-- Columna Derecha: PASIVO + PATRIMONIO -->
                <div style="display: flex; flex-direction: column; gap: 30px;">
                    <div class="balance-section">
                        <div class="balance-header pasivo-header">
                            <h2>PASIVO</h2>
                        </div>
                        <div id="pasivoList"></div>
                        <div class="total-row" style="color: #dc2626;">
                            <span>TOTAL PASIVO</span>
                            <span id="totalPasivo">Gs. 0</span>
                        </div>
                    </div>
                    
                    <div class="balance-section">
                        <div class="balance-header patrimonio-header">
                            <h2>PATRIMONIO</h2>
                        </div>
                        <div id="patrimonioList"></div>
                        <div class="total-row" style="color: #2563eb;">
                            <span>TOTAL PATRIMONIO</span>
                            <span id="totalPatrimonio">Gs. 0</span>
                        </div>
                    </div>
                    
                    <div class="balance-section" style="background: #f9fafb;">
                        <div class="total-row" style="margin-top: 0; border-top: none;">
                            <span>TOTAL PASIVO + PATRIMONIO</span>
                            <span id="totalPasivoPatrimonio">Gs. 0</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div id="loading" class="loading">Cargando balance...</div>
    </div>

    <script src="../../assets/js/lexipro.js"></script>
    <script>
        async function loadBalance() {
            const fecha = document.getElementById('fechaCorte').value;
            document.getElementById('balanceContent').style.display = 'none';
            document.getElementById('loading').style.display = 'block';
            
            try {
                const response = await fetch(`api.php?action=balance&fecha=${fecha}`);
                const data = await response.json();
                
                if (data.success) {
                    renderBalance(data);
                }
            } catch (error) {
                console.error(error);
                alert('Error al cargar el balance');
            } finally {
                document.getElementById('loading').style.display = 'none';
            }
        }
        
        function renderBalance(data) {
            document.getElementById('balanceContent').style.display = 'block';
            
            // Render Activo
            renderList('activoList', data.activo);
            document.getElementById('totalActivo').textContent = LexiPro.formatMoney(data.total_activo);
            
            // Render Pasivo
            renderList('pasivoList', data.pasivo);
            document.getElementById('totalPasivo').textContent = LexiPro.formatMoney(data.total_pasivo);
            
            // Render Patrimonio
            renderList('patrimonioList', data.patrimonio);
            document.getElementById('totalPatrimonio').textContent = LexiPro.formatMoney(data.total_patrimonio);
            
            // Total Pasivo + Patrimonio
            const totalPP = parseFloat(data.total_pasivo) + parseFloat(data.total_patrimonio);
            document.getElementById('totalPasivoPatrimonio').textContent = LexiPro.formatMoney(totalPP);
        }
        
        function renderList(elementId, items) {
            const container = document.getElementById(elementId);
            if (items.length === 0) {
                container.innerHTML = '<div style="text-align: center; color: #9ca3af; padding: 10px;">Sin movimientos</div>';
                return;
            }
            
            container.innerHTML = items.map(item => `
                <div class="account-row">
                    <div>
                        <span class="account-code">${item.codigo}</span>
                        <span>${item.nombre}</span>
                    </div>
                    <strong>${LexiPro.formatMoney(item.saldo)}</strong>
                </div>
            `).join('');
        }
        
        // Iniciar
        loadBalance();
    </script>
</body>
</html>
