<?php
/**
 * LexiPro - Libro Diario
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();

// Obtener cuentas para el selector del modal
$cuentas = $db->fetchAll("SELECT * FROM plan_cuentas WHERE es_movimiento = 1 ORDER BY codigo");
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Libro Diario - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .asiento-card {
            background: white;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            margin-bottom: 20px;
            overflow: hidden;
        }
        
        .asiento-header {
            background: #f9fafb;
            padding: 15px 20px;
            border-bottom: 1px solid #e5e7eb;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .asiento-body {
            padding: 0;
        }
        
        .asiento-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .asiento-table th, .asiento-table td {
            padding: 10px 20px;
            text-align: left;
            border-bottom: 1px solid #f3f4f6;
        }
        
        .asiento-table th {
            background: #f9fafb;
            font-size: 12px;
            text-transform: uppercase;
            color: #6b7280;
        }
        
        .asiento-table tr:last-child td {
            border-bottom: none;
        }
        
        .debe-col, .haber-col {
            text-align: right !important;
            font-family: monospace;
            font-size: 14px;
        }
        
        .total-row {
            background: #f3f4f6;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="header-flex">
            <div>
                <h1>📖 Libro Diario</h1>
                <p>Registro cronológico de asientos contables</p>
            </div>
            <button class="btn btn-primary" onclick="openAsientoModal()">+ Nuevo Asiento Manual</button>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>Desde</label>
                <input type="date" id="fechaDesde" value="<?= date('Y-m-01') ?>">
            </div>
            <div class="filter-group">
                <label>Hasta</label>
                <input type="date" id="fechaHasta" value="<?= date('Y-m-t') ?>">
            </div>
            <div class="filter-group" style="align-self: flex-end;">
                <button class="btn btn-secondary" onclick="loadAsientos()">Filtrar</button>
            </div>
        </div>
        
        <div id="asientosContainer">
            <div class="loading">Cargando asientos...</div>
        </div>
    </div>
    
    <!-- Modal Nuevo Asiento -->
    <div class="modal" id="asientoModal">
        <div class="modal-content modal-large">
            <div class="modal-header">
                <h2>Nuevo Asiento Manual</h2>
                <button class="modal-close" onclick="closeAsientoModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="asientoForm">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Fecha *</label>
                            <input type="date" name="fecha" required value="<?= date('Y-m-d') ?>">
                        </div>
                        <div class="form-group" style="flex: 2;">
                            <label>Descripción General *</label>
                            <input type="text" name="descripcion" required placeholder="Ej: Ajuste de caja chica">
                        </div>
                    </div>
                    
                    <table class="table" style="margin-top: 15px;">
                        <thead>
                            <tr>
                                <th width="40%">Cuenta</th>
                                <th width="25%">Descripción (Opcional)</th>
                                <th width="15%">Debe</th>
                                <th width="15%">Haber</th>
                                <th width="5%"></th>
                            </tr>
                        </thead>
                        <tbody id="detallesBody">
                            <!-- Filas dinámicas -->
                        </tbody>
                        <tfoot>
                            <tr style="background: #f9fafb; font-weight: bold;">
                                <td colspan="2" style="text-align: right;">TOTALES:</td>
                                <td id="totalDebe" style="text-align: right;">0</td>
                                <td id="totalHaber" style="text-align: right;">0</td>
                                <td></td>
                            </tr>
                        </tfoot>
                    </table>
                    
                    <button type="button" class="btn btn-secondary btn-sm" onclick="addDetalleRow()" style="margin-top: 10px;">+ Agregar Línea</button>
                    
                    <div id="errorBalance" class="alert alert-danger" style="display: none; margin-top: 15px;">
                        El asiento no cuadra. La diferencia debe ser 0.
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">Guardar Asiento</button>
                </form>
            </div>
        </div>
    </div>

    <script src="../../assets/js/lexipro.js"></script>
    <script>
        const cuentas = <?= json_encode($cuentas) ?>;
        let rowCount = 0;
        
        async function loadAsientos() {
            const desde = document.getElementById('fechaDesde').value;
            const hasta = document.getElementById('fechaHasta').value;
            const container = document.getElementById('asientosContainer');
            
            container.innerHTML = '<div class="loading">Cargando...</div>';
            
            try {
                const response = await fetch(`api.php?action=asientos&fecha_desde=${desde}&fecha_hasta=${hasta}`);
                const data = await response.json();
                
                if (data.success) {
                    renderAsientos(data.asientos);
                }
            } catch (error) {
                console.error(error);
                container.innerHTML = '<div class="alert alert-danger">Error al cargar asientos</div>';
            }
        }
        
        async function renderAsientos(asientos) {
            const container = document.getElementById('asientosContainer');
            
            if (asientos.length === 0) {
                container.innerHTML = '<p style="text-align: center; color: #6b7280; padding: 40px;">No hay asientos en este período.</p>';
                return;
            }
            
            container.innerHTML = '';
            
            for (const asiento of asientos) {
                // Obtener detalles de cada asiento
                const resp = await fetch(`api.php?action=asiento_detalle&id=${asiento.id}`);
                const detData = await resp.json();
                const detalles = detData.detalles || [];
                
                let totalDebe = 0;
                let totalHaber = 0;
                
                const detallesHtml = detalles.map(d => {
                    totalDebe += parseFloat(d.debe);
                    totalHaber += parseFloat(d.haber);
                    return `
                        <tr>
                            <td>${d.codigo} - ${d.cuenta_nombre}</td>
                            <td>${d.descripcion || ''}</td>
                            <td class="debe-col">${parseFloat(d.debe) > 0 ? LexiPro.formatMoney(d.debe) : ''}</td>
                            <td class="haber-col">${parseFloat(d.haber) > 0 ? LexiPro.formatMoney(d.haber) : ''}</td>
                        </tr>
                    `;
                }).join('');
                
                const html = `
                    <div class="asiento-card">
                        <div class="asiento-header">
                            <div>
                                <strong>#${asiento.id}</strong> - ${LexiPro.formatDate(asiento.fecha)}
                                <div style="font-size: 13px; color: #6b7280;">${asiento.descripcion}</div>
                            </div>
                            <span class="badge badge-info">${asiento.tipo.toUpperCase()}</span>
                        </div>
                        <div class="asiento-body">
                            <table class="asiento-table">
                                <thead>
                                    <tr>
                                        <th>Cuenta</th>
                                        <th>Detalle</th>
                                        <th style="text-align: right;">Debe</th>
                                        <th style="text-align: right;">Haber</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${detallesHtml}
                                    <tr class="total-row">
                                        <td colspan="2" style="text-align: right;">TOTALES</td>
                                        <td class="debe-col">${LexiPro.formatMoney(totalDebe)}</td>
                                        <td class="haber-col">${LexiPro.formatMoney(totalHaber)}</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                `;
                container.innerHTML += html;
            }
        }
        
        function openAsientoModal() {
            document.getElementById('asientoModal').classList.add('active');
            document.getElementById('detallesBody').innerHTML = '';
            rowCount = 0;
            addDetalleRow();
            addDetalleRow(); // Mínimo 2 líneas
            calculateTotals();
        }
        
        function closeAsientoModal() {
            document.getElementById('asientoModal').classList.remove('active');
            document.getElementById('asientoForm').reset();
        }
        
        function addDetalleRow() {
            rowCount++;
            const tbody = document.getElementById('detallesBody');
            const tr = document.createElement('tr');
            
            let options = '<option value="">Seleccionar cuenta...</option>';
            cuentas.forEach(c => {
                options += `<option value="${c.id}">${c.codigo} - ${c.nombre}</option>`;
            });
            
            tr.innerHTML = `
                <td>
                    <select name="detalles[${rowCount}][cuenta_id]" required style="width: 100%; padding: 8px; border: 1px solid #d1d5db; border-radius: 4px;">
                        ${options}
                    </select>
                </td>
                <td>
                    <input type="text" name="detalles[${rowCount}][descripcion]" style="width: 100%; padding: 8px; border: 1px solid #d1d5db; border-radius: 4px;">
                </td>
                <td>
                    <input type="number" name="detalles[${rowCount}][debe]" class="monto-input" step="0.01" min="0" value="0" onchange="calculateTotals()" style="width: 100%; padding: 8px; border: 1px solid #d1d5db; border-radius: 4px; text-align: right;">
                </td>
                <td>
                    <input type="number" name="detalles[${rowCount}][haber]" class="monto-input" step="0.01" min="0" value="0" onchange="calculateTotals()" style="width: 100%; padding: 8px; border: 1px solid #d1d5db; border-radius: 4px; text-align: right;">
                </td>
                <td>
                    <button type="button" onclick="this.closest('tr').remove(); calculateTotals();" style="color: red; border: none; background: none; cursor: pointer;">×</button>
                </td>
            `;
            tbody.appendChild(tr);
        }
        
        function calculateTotals() {
            let totalDebe = 0;
            let totalHaber = 0;
            
            document.querySelectorAll('input[name*="[debe]"]').forEach(i => totalDebe += parseFloat(i.value || 0));
            document.querySelectorAll('input[name*="[haber]"]').forEach(i => totalHaber += parseFloat(i.value || 0));
            
            document.getElementById('totalDebe').textContent = LexiPro.formatMoney(totalDebe);
            document.getElementById('totalHaber').textContent = LexiPro.formatMoney(totalHaber);
            
            const diff = Math.abs(totalDebe - totalHaber);
            const errorDiv = document.getElementById('errorBalance');
            
            if (diff > 0.01) {
                errorDiv.style.display = 'block';
                errorDiv.textContent = `El asiento no cuadra. Diferencia: ${LexiPro.formatMoney(diff)}`;
                return false;
            } else {
                errorDiv.style.display = 'none';
                return true;
            }
        }
        
        document.getElementById('asientoForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            if (!calculateTotals()) return;
            
            // Construir objeto JSON manualmente para manejar array de detalles
            const formData = new FormData(e.target);
            const data = {
                fecha: formData.get('fecha'),
                descripcion: formData.get('descripcion'),
                detalles: []
            };
            
            // Recorrer filas
            const rows = document.querySelectorAll('#detallesBody tr');
            rows.forEach(row => {
                const cuentaId = row.querySelector('select').value;
                const desc = row.querySelector('input[type="text"]').value;
                const debe = row.querySelector('input[name*="[debe]"]').value;
                const haber = row.querySelector('input[name*="[haber]"]').value;
                
                if (cuentaId) {
                    data.detalles.push({
                        cuenta_id: cuentaId,
                        descripcion: desc,
                        debe: parseFloat(debe),
                        haber: parseFloat(haber)
                    });
                }
            });
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('Asiento guardado exitosamente');
                    closeAsientoModal();
                    loadAsientos();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar asiento');
            }
        });
        
        // Iniciar
        loadAsientos();
    </script>
</body>
</html>
