<?php
/**
 * LexiPro - API de Contabilidad
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../includes/contabilidad_functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    $action = $_GET['action'] ?? '';
    
    // GET - Obtener datos contables
    if ($method === 'GET') {
        $auth->requirePermission('contabilidad', 'ver');
        
        switch ($action) {
            case 'plan_cuentas':
                $cuentas = $db->fetchAll("SELECT * FROM plan_cuentas ORDER BY codigo");
                jsonSuccess(['cuentas' => $cuentas]);
                break;
                
            case 'asientos':
                $fechaDesde = $_GET['fecha_desde'] ?? '';
                $fechaHasta = $_GET['fecha_hasta'] ?? '';
                
                $sql = "SELECT * FROM asientos_contables WHERE 1=1";
                $params = [];
                
                if ($fechaDesde) {
                    $sql .= " AND fecha >= ?";
                    $params[] = $fechaDesde;
                }
                
                if ($fechaHasta) {
                    $sql .= " AND fecha <= ?";
                    $params[] = $fechaHasta;
                }
                
                $sql .= " ORDER BY fecha DESC, id DESC LIMIT 100";
                
                $asientos = $db->fetchAll($sql, $params);
                jsonSuccess(['asientos' => $asientos]);
                break;
                
            case 'asiento_detalle':
                $asientoId = $_GET['id'] ?? 0;
                $detalles = $db->fetchAll(
                    "SELECT ad.*, pc.codigo, pc.nombre as cuenta_nombre
                     FROM asientos_detalle ad
                     INNER JOIN plan_cuentas pc ON ad.cuenta_id = pc.id
                     WHERE ad.asiento_id = ?
                     ORDER BY ad.id",
                    [$asientoId]
                );
                jsonSuccess(['detalles' => $detalles]);
                break;
                
            case 'mayor':
                $cuentaId = $_GET['cuenta_id'] ?? 0;
                $fechaDesde = $_GET['fecha_desde'] ?? '';
                $fechaHasta = $_GET['fecha_hasta'] ?? '';
                
                $sql = "SELECT ac.fecha, ac.descripcion, ad.debe, ad.haber, ad.descripcion as detalle_desc
                        FROM asientos_detalle ad
                        INNER JOIN asientos_contables ac ON ad.asiento_id = ac.id
                        WHERE ad.cuenta_id = ?";
                
                $params = [$cuentaId];
                
                if ($fechaDesde) {
                    $sql .= " AND ac.fecha >= ?";
                    $params[] = $fechaDesde;
                }
                
                if ($fechaHasta) {
                    $sql .= " AND ac.fecha <= ?";
                    $params[] = $fechaHasta;
                }
                
                $sql .= " ORDER BY ac.fecha, ac.id";
                
                $movimientos = $db->fetchAll($sql, $params);
                
                // Calcular saldos acumulados
                $saldo = 0;
                foreach ($movimientos as &$mov) {
                    $saldo += ($mov['debe'] - $mov['haber']);
                    $mov['saldo'] = $saldo;
                }
                
                jsonSuccess(['movimientos' => $movimientos]);
                break;
                
            case 'balance':
                $fecha = $_GET['fecha'] ?? date('Y-m-d');
                $balance = getBalance($fecha);
                jsonSuccess($balance);
                break;
                
            case 'resultados':
                $fechaDesde = $_GET['fecha_desde'] ?? date('Y-m-01');
                $fechaHasta = $_GET['fecha_hasta'] ?? date('Y-m-d');
                $resultados = getEstadoResultados($fechaDesde, $fechaHasta);
                jsonSuccess($resultados);
                break;
                
            default:
                jsonError('Acción no válida');
        }
    }
    
    // POST - Crear asiento manual
    elseif ($method === 'POST') {
        $auth->requirePermission('contabilidad', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['fecha'])) {
            jsonError('Fecha es obligatoria');
        }
        
        if (empty($data['detalles']) || count($data['detalles']) < 2) {
            jsonError('Debe agregar al menos 2 líneas al asiento');
        }
        
        // Validar que cuadre
        $totalDebe = 0;
        $totalHaber = 0;
        
        foreach ($data['detalles'] as $detalle) {
            $totalDebe += (float)($detalle['debe'] ?? 0);
            $totalHaber += (float)($detalle['haber'] ?? 0);
        }
        
        if (abs($totalDebe - $totalHaber) > 0.01) {
            jsonError('El asiento no cuadra. Debe = ' . formatMoney($totalDebe) . ', Haber = ' . formatMoney($totalHaber));
        }
        
        $db->beginTransaction();
        
        try {
            // Crear asiento
            $asientoId = $db->insert('asientos_contables', [
                'fecha' => $data['fecha'],
                'descripcion' => sanitize($data['descripcion']),
                'tipo' => 'manual',
                'usuario_id' => $userId
            ]);
            
            // Insertar detalles
            foreach ($data['detalles'] as $detalle) {
                $db->insert('asientos_detalle', [
                    'asiento_id' => $asientoId,
                    'cuenta_id' => $detalle['cuenta_id'],
                    'debe' => (float)($detalle['debe'] ?? 0),
                    'haber' => (float)($detalle['haber'] ?? 0),
                    'descripcion' => sanitize($detalle['descripcion'] ?? '')
                ]);
            }
            
            $db->commit();
            
            logActivity($userId, 'crear', 'asientos', $asientoId, "Asiento contable creado");
            
            jsonSuccess(['id' => $asientoId], 'Asiento creado exitosamente');
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
} catch (Exception $e) {
    error_log("Error en API de contabilidad: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
