<?php
/**
 * LexiPro - API de Configuración
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Obtener configuración
    if ($method === 'GET') {
        $auth->requirePermission('configuracion', 'ver');
        
        $clave = $_GET['clave'] ?? null;
        
        if ($clave) {
            // Configuración específica
            $config = $db->fetchOne("SELECT * FROM configuracion WHERE clave = ?", [$clave]);
            
            if ($config && $config['tipo'] === 'json') {
                $config['valor'] = json_decode($config['valor'], true);
            }
            
            jsonSuccess(['configuracion' => $config]);
        } else {
            // Todas las configuraciones
            $configs = $db->fetchAll("SELECT * FROM configuracion ORDER BY categoria, clave");
            
            // Decodificar JSON
            foreach ($configs as &$config) {
                if ($config['tipo'] === 'json') {
                    $config['valor'] = json_decode($config['valor'], true);
                }
            }
            
            jsonSuccess(['configuraciones' => $configs]);
        }
    }
    
    // POST - Guardar configuración
    elseif ($method === 'POST') {
        $auth->requirePermission('configuracion', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['clave'])) {
            jsonError('Clave de configuración requerida');
        }
        
        // Preparar valor según tipo
        $valor = $data['valor'];
        if ($data['tipo'] === 'json') {
            $valor = json_encode($valor);
        }
        
        // Verificar si existe
        $existe = $db->fetchOne("SELECT id FROM configuracion WHERE clave = ?", [$data['clave']]);
        
        if ($existe) {
            // Actualizar
            $db->update('configuracion', [
                'valor' => $valor,
                'descripcion' => sanitize($data['descripcion'] ?? null)
            ], 'clave = ?', [$data['clave']]);
        } else {
            // Insertar
            $db->insert('configuracion', [
                'clave' => $data['clave'],
                'valor' => $valor,
                'tipo' => $data['tipo'] ?? 'string',
                'categoria' => $data['categoria'] ?? 'general',
                'descripcion' => sanitize($data['descripcion'] ?? null)
            ]);
        }
        
        // Log de actividad
        logActivity($userId, 'actualizar', 'configuracion', 0, "Configuración actualizada: {$data['clave']}");
        
        jsonSuccess([], 'Configuración guardada exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de configuración: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
