<?php
/**
 * LexiPro - Gestión de Compras (Facturas de Proveedores)
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Breadcrumbs
$breadcrumbs = [
    ['label' => 'Compras']
];

// Obtener proveedores
$proveedores = $db->fetchAll("SELECT id, nombre, razon_social, ruc FROM proveedores WHERE activo = 1 ORDER BY razon_social LIMIT 200");
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Compras - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .purchase-header {
            background: linear-gradient(135deg, #7e22ce 0%, #9333ea 100%);
            color: white;
            padding: 30px;
            border-radius: 12px 12px 0 0;
            margin-bottom: 0;
        }
        
        .purchase-stats {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .stat-box {
            background: white;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
        }
        
        .stat-value {
            font-size: 28px;
            font-weight: 700;
            color: #7e22ce;
            margin-bottom: 5px;
        }
        
        .stat-label {
            font-size: 13px;
            color: #6b7280;
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="purchase-header">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <h1>🛒 Compras</h1>
                    <p style="opacity: 0.9; margin-top: 5px;">Gestión de facturas de proveedores</p>
                </div>
                <button class="btn btn-primary" onclick="openPurchaseModal()" style="background: white; color: #7e22ce;">
                    + Registrar Factura
                </button>
            </div>
        </div>
        
        <div class="purchase-stats">
            <div class="stat-box">
                <div class="stat-value" id="totalComprado">Gs. 0</div>
                <div class="stat-label">Total Comprado</div>
            </div>
            <div class="stat-box">
                <div class="stat-value" id="totalPagado">Gs. 0</div>
                <div class="stat-label">Total Pagado</div>
            </div>
            <div class="stat-box">
                <div class="stat-value" id="totalPendiente">Gs. 0</div>
                <div class="stat-label">Por Pagar</div>
            </div>
            <div class="stat-box">
                <div class="stat-value" id="cantidadFacturas">0</div>
                <div class="stat-label">Facturas del Mes</div>
            </div>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>🔍 Buscar</label>
                <input type="text" id="searchInput" placeholder="Número de factura, proveedor...">
            </div>
            <div class="filter-group">
                <label>Proveedor</label>
                <select id="proveedorFilter">
                    <option value="">Todos los proveedores</option>
                    <?php foreach ($proveedores as $proveedor): ?>
                        <option value="<?= $proveedor['id'] ?>"><?= e($proveedor['razon_social'] ?? $proveedor['nombre']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label>Estado</label>
                <select id="estadoFilter">
                    <option value="">Todos</option>
                    <option value="pagada">Pagada</option>
                    <option value="pendiente">Pendiente</option>
                    <option value="parcial">Parcial</option>
                </select>
            </div>
            <div class="filter-group">
                <label>Desde</label>
                <input type="date" id="fechaDesde">
            </div>
            <div class="filter-group">
                <label>Hasta</label>
                <input type="date" id="fechaHasta">
            </div>
        </div>
        
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>Número</th>
                        <th>Fecha</th>
                        <th>Proveedor</th>
                        <th>Timbrado</th>
                        <th>Condición</th>
                        <th>Total</th>
                        <th>Pagado</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="comprasTableBody">
                    <tr>
                        <td colspan="9" class="loading">Cargando facturas...</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Modal de Factura de Compra -->
    <div class="modal" id="purchaseModal">
        <div class="modal-content modal-large" style="max-width: 1000px;">
            <div class="modal-header">
                <h2>Registrar Factura de Compra</h2>
                <button class="modal-close" onclick="closePurchaseModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="purchaseForm">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Proveedor *</label>
                            <select name="proveedor_id" required id="proveedorSelect" onchange="loadProveedorData()">
                                <option value="">Seleccionar proveedor...</option>
                                <?php foreach ($proveedores as $proveedor): ?>
                                    <option value="<?= $proveedor['id'] ?>" data-ruc="<?= $proveedor['ruc'] ?? '' ?>">
                                        <?= e($proveedor['razon_social'] ?? $proveedor['nombre']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>RUC Proveedor</label>
                            <input type="text" id="proveedorRuc" readonly style="background: #f3f4f6;">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Número de Factura *</label>
                            <input type="text" name="numero_factura" required placeholder="001-001-0000001">
                        </div>
                        <div class="form-group">
                            <label>Timbrado</label>
                            <input type="text" name="timbrado" placeholder="12345678" maxlength="8">
                        </div>
                        <div class="form-group">
                            <label>Fecha *</label>
                            <input type="date" name="fecha" required value="<?= date('Y-m-d') ?>">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Condición de Compra *</label>
                            <select name="condicion_compra" required id="condicionCompra" onchange="toggleFormaPago()">
                                <option value="contado">Contado</option>
                                <option value="credito">Crédito</option>
                            </select>
                        </div>
                    </div>
                    
                    <div id="formaPagoSection">
                        <div class="form-row">
                            <div class="form-group">
                                <label>Forma de Pago *</label>
                                <select name="forma_pago" id="formaPago">
                                    <option value="efectivo">Efectivo</option>
                                    <option value="transferencia">Transferencia</option>
                                    <option value="cheque">Cheque</option>
                                    <option value="tarjeta">Tarjeta</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Referencia</label>
                                <input type="text" name="referencia_pago" placeholder="Nro. de transferencia, cheque, etc.">
                            </div>
                        </div>
                    </div>
                    
                    <hr style="margin: 20px 0;">
                    
                    <h3 style="margin-bottom: 15px;">Detalles de la Factura</h3>
                    
                    <div id="detallesContainer">
                        <div class="detalle-row" style="display: grid; grid-template-columns: 3fr 1fr 1fr 1fr 1fr 50px; gap: 10px; margin-bottom: 10px;">
                            <div><strong>Descripción</strong></div>
                            <div><strong>Cantidad</strong></div>
                            <div><strong>Precio Unit.</strong></div>
                            <div><strong>IVA</strong></div>
                            <div><strong>Subtotal</strong></div>
                            <div></div>
                        </div>
                    </div>
                    
                    <button type="button" class="btn btn-secondary" onclick="addDetalle()" style="margin-top: 10px;">
                        + Agregar Ítem
                    </button>
                    
                    <div class="totales-box" style="background: #f9fafb; padding: 20px; border-radius: 8px; margin-top: 20px;">
                        <div class="total-row" style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #e5e7eb;">
                            <span>Subtotal Gravado 5%:</span>
                            <span id="subtotal5">Gs. 0</span>
                        </div>
                        <div class="total-row" style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #e5e7eb;">
                            <span>Subtotal Gravado 10%:</span>
                            <span id="subtotal10">Gs. 0</span>
                        </div>
                        <div class="total-row" style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #e5e7eb;">
                            <span>Subtotal Exento:</span>
                            <span id="subtotalExento">Gs. 0</span>
                        </div>
                        <div class="total-row" style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #e5e7eb;">
                            <span>IVA 5%:</span>
                            <span id="iva5">Gs. 0</span>
                        </div>
                        <div class="total-row" style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #e5e7eb;">
                            <span>IVA 10%:</span>
                            <span id="iva10">Gs. 0</span>
                        </div>
                        <div class="total-row" style="display: flex; justify-content: space-between; padding-top: 15px; border-top: 2px solid #7e22ce; font-size: 18px; font-weight: 700; color: #7e22ce;">
                            <span>TOTAL GENERAL:</span>
                            <span id="totalGeneral">Gs. 0</span>
                        </div>
                    </div>
                    
                    <div class="form-group" style="margin-top: 20px;">
                        <label>Observaciones</label>
                        <textarea name="observaciones" rows="2" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px; background: linear-gradient(135deg, #7e22ce 0%, #9333ea 100%);">
                        Registrar Factura
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let facturas = [];
        let detalleCounter = 0;
        
        // Cargar facturas
        async function loadFacturas() {
            try {
                const params = new URLSearchParams();
                const search = document.getElementById('searchInput').value;
                const proveedor = document.getElementById('proveedorFilter').value;
                const estado = document.getElementById('estadoFilter').value;
                const fechaDesde = document.getElementById('fechaDesde').value;
                const fechaHasta = document.getElementById('fechaHasta').value;
                
                if (search) params.append('search', search);
                if (proveedor) params.append('proveedor', proveedor);
                if (estado) params.append('estado', estado);
                if (fechaDesde) params.append('fecha_desde', fechaDesde);
                if (fechaHasta) params.append('fecha_hasta', fechaHasta);
                
                const response = await fetch(`api.php?${params}`);
                const data = await response.json();
                
                if (data.success) {
                    facturas = data.facturas;
                    renderFacturas();
                    updateStats();
                }
            } catch (error) {
                console.error('Error al cargar facturas:', error);
            }
        }
        
        // Renderizar facturas
        function renderFacturas() {
            const tbody = document.getElementById('comprasTableBody');
            
            if (facturas.length === 0) {
                tbody.innerHTML = '<tr><td colspan="9" class="loading">No se encontraron facturas</td></tr>';
                return;
            }
            
            tbody.innerHTML = facturas.map(factura => {
                const pagado = parseFloat(factura.total_pagado || 0);
                const total = parseFloat(factura.total_general);
                const saldo = total - pagado;
                
                return `
                    <tr>
                        <td><strong>${factura.numero_factura}</strong></td>
                        <td>${LexiPro.formatDate(factura.fecha)}</td>
                        <td>${factura.proveedor_nombre}</td>
                        <td>${factura.timbrado || '-'}</td>
                        <td><span class="badge badge-info">${factura.condicion_compra === 'contado' ? 'Contado' : 'Crédito'}</span></td>
                        <td>${LexiPro.formatMoney(total)}</td>
                        <td>${LexiPro.formatMoney(pagado)}</td>
                        <td><span class="estado-badge estado-${factura.estado}">${factura.estado.toUpperCase()}</span></td>
                        <td>
                            <div class="actions">
                                ${factura.estado !== 'pagada' ? 
                                    `<button class="btn btn-sm btn-edit" onclick="registrarPago(${factura.id}, ${saldo})">Pagar</button>` : 
                                    '<span style="color: #10b981;">✓ Pagada</span>'}
                            </div>
                        </td>
                    </tr>
                `;
            }).join('');
        }
        
        // Actualizar estadísticas
        function updateStats() {
            const mesActual = new Date().getMonth();
            const añoActual = new Date().getFullYear();
            
            const facturasMes = facturas.filter(f => {
                const fecha = new Date(f.fecha);
                return fecha.getMonth() === mesActual && fecha.getFullYear() === añoActual;
            });
            
            const totalComprado = facturasMes.reduce((sum, f) => sum + parseFloat(f.total_general), 0);
            const totalPagado = facturasMes.reduce((sum, f) => sum + parseFloat(f.total_pagado || 0), 0);
            const totalPendiente = totalComprado - totalPagado;
            
            document.getElementById('totalComprado').textContent = LexiPro.formatMoney(totalComprado);
            document.getElementById('totalPagado').textContent = LexiPro.formatMoney(totalPagado);
            document.getElementById('totalPendiente').textContent = LexiPro.formatMoney(totalPendiente);
            document.getElementById('cantidadFacturas').textContent = facturasMes.length;
        }
        
        // Abrir modal
        function openPurchaseModal() {
            document.getElementById('purchaseModal').classList.add('active');
            addDetalle();
        }
        
        // Cerrar modal
        function closePurchaseModal() {
            document.getElementById('purchaseModal').classList.remove('active');
            document.getElementById('purchaseForm').reset();
            document.getElementById('detallesContainer').innerHTML = `
                <div class="detalle-row" style="display: grid; grid-template-columns: 3fr 1fr 1fr 1fr 1fr 50px; gap: 10px; margin-bottom: 10px;">
                    <div><strong>Descripción</strong></div>
                    <div><strong>Cantidad</strong></div>
                    <div><strong>Precio Unit.</strong></div>
                    <div><strong>IVA</strong></div>
                    <div><strong>Subtotal</strong></div>
                    <div></div>
                </div>
            `;
            detalleCounter = 0;
        }
        
        // Cargar datos del proveedor
        function loadProveedorData() {
            const select = document.getElementById('proveedorSelect');
            const option = select.options[select.selectedIndex];
            document.getElementById('proveedorRuc').value = option.dataset.ruc || '';
        }
        
        // Toggle forma de pago
        function toggleFormaPago() {
            const condicion = document.getElementById('condicionCompra').value;
            const section = document.getElementById('formaPagoSection');
            const formaPago = document.getElementById('formaPago');
            
            if (condicion === 'contado') {
                section.style.display = 'block';
                formaPago.required = true;
            } else {
                section.style.display = 'none';
                formaPago.required = false;
            }
        }
        
        // Agregar detalle
        function addDetalle() {
            detalleCounter++;
            const container = document.getElementById('detallesContainer');
            const row = document.createElement('div');
            row.className = 'detalle-row';
            row.id = `detalle-${detalleCounter}`;
            row.style.cssText = 'display: grid; grid-template-columns: 3fr 1fr 1fr 1fr 1fr 50px; gap: 10px; margin-bottom: 10px; align-items: center;';
            row.innerHTML = `
                <input type="text" placeholder="Descripción del producto/servicio" data-field="descripcion" required style="padding: 10px; border: 2px solid #e5e7eb; border-radius: 6px;">
                <input type="number" placeholder="1" value="1" min="1" step="0.01" data-field="cantidad" onchange="calcularTotales()" style="padding: 10px; border: 2px solid #e5e7eb; border-radius: 6px;">
                <input type="number" placeholder="0" min="0" step="1000" data-field="precio_unitario" onchange="calcularTotales()" style="padding: 10px; border: 2px solid #e5e7eb; border-radius: 6px;">
                <select data-field="tipo_iva" onchange="calcularTotales()" style="padding: 10px; border: 2px solid #e5e7eb; border-radius: 6px;">
                    <option value="10">10%</option>
                    <option value="5">5%</option>
                    <option value="exento">Exento</option>
                </select>
                <input type="number" readonly style="background: #f3f4f6; padding: 10px; border: 2px solid #e5e7eb; border-radius: 6px;" data-field="subtotal" value="0">
                <button type="button" onclick="removeDetalle(${detalleCounter})" style="background: #ef4444; color: white; border: none; padding: 10px; border-radius: 6px; cursor: pointer;">×</button>
            `;
            container.appendChild(row);
        }
        
        // Remover detalle
        function removeDetalle(id) {
            const row = document.getElementById(`detalle-${id}`);
            if (row) {
                row.remove();
                calcularTotales();
            }
        }
        
        // Calcular totales (igual que en facturación)
        function calcularTotales() {
            let subtotal5 = 0, subtotal10 = 0, subtotalExento = 0;
            
            document.querySelectorAll('.detalle-row').forEach((row, index) => {
                if (index === 0) return;
                
                const cantidad = parseFloat(row.querySelector('[data-field="cantidad"]')?.value || 0);
                const precio = parseFloat(row.querySelector('[data-field="precio_unitario"]')?.value || 0);
                const tipoIva = row.querySelector('[data-field="tipo_iva"]')?.value;
                const subtotal = cantidad * precio;
                
                const subtotalField = row.querySelector('[data-field="subtotal"]');
                if (subtotalField) subtotalField.value = subtotal.toFixed(0);
                
                switch(tipoIva) {
                    case '5': subtotal5 += subtotal; break;
                    case '10': subtotal10 += subtotal; break;
                    case 'exento': subtotalExento += subtotal; break;
                }
            });
            
            const iva5 = subtotal5 * 0.05;
            const iva10 = subtotal10 * 0.10;
            const totalGeneral = subtotal5 + subtotal10 + subtotalExento + iva5 + iva10;
            
            document.getElementById('subtotal5').textContent = LexiPro.formatMoney(subtotal5);
            document.getElementById('subtotal10').textContent = LexiPro.formatMoney(subtotal10);
            document.getElementById('subtotalExento').textContent = LexiPro.formatMoney(subtotalExento);
            document.getElementById('iva5').textContent = LexiPro.formatMoney(iva5);
            document.getElementById('iva10').textContent = LexiPro.formatMoney(iva10);
            document.getElementById('totalGeneral').textContent = LexiPro.formatMoney(totalGeneral);
        }
        
        // Guardar factura
        document.getElementById('purchaseForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {
                proveedor_id: formData.get('proveedor_id'),
                numero_factura: formData.get('numero_factura'),
                timbrado: formData.get('timbrado'),
                fecha: formData.get('fecha'),
                condicion_compra: formData.get('condicion_compra'),
                forma_pago: formData.get('forma_pago'),
                referencia_pago: formData.get('referencia_pago'),
                observaciones: formData.get('observaciones'),
                detalles: []
            };
            
            // Recopilar detalles
            document.querySelectorAll('.detalle-row').forEach((row, index) => {
                if (index === 0) return;
                
                const detalle = {
                    descripcion: row.querySelector('[data-field="descripcion"]').value,
                    cantidad: parseFloat(row.querySelector('[data-field="cantidad"]').value),
                    precio_unitario: parseFloat(row.querySelector('[data-field="precio_unitario"]').value),
                    tipo_iva: row.querySelector('[data-field="tipo_iva"]').value
                };
                
                if (detalle.descripcion && detalle.cantidad > 0 && detalle.precio_unitario >= 0) {
                    data.detalles.push(detalle);
                }
            });
            
            if (data.detalles.length === 0) {
                alert('Debe agregar al menos un ítem a la factura');
                return;
            }
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('Factura de compra registrada exitosamente');
                    closePurchaseModal();
                    loadFacturas();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al registrar factura');
                console.error(error);
            }
        });
        
        // Registrar pago
        async function registrarPago(facturaId, saldo) {
            const monto = prompt(`Ingrese el monto a pagar (Saldo: ${LexiPro.formatMoney(saldo)}):`);
            if (!monto) return;
            
            const formaPago = prompt('Forma de pago (efectivo/transferencia/cheque/tarjeta):');
            if (!formaPago) return;
            
            try {
                const response = await fetch('api.php', {
                    method: 'PUT',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        id: facturaId,
                        registrar_pago: true,
                        monto: parseFloat(monto),
                        forma_pago: formaPago,
                        fecha_pago: new Date().toISOString().split('T')[0]
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('Pago registrado exitosamente');
                    loadFacturas();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al registrar pago');
                console.error(error);
            }
        }
        
        // Event listeners
        document.getElementById('searchInput').addEventListener('input', LexiPro.debounce(loadFacturas, 300));
        document.getElementById('proveedorFilter').addEventListener('change', loadFacturas);
        document.getElementById('estadoFilter').addEventListener('change', loadFacturas);
        document.getElementById('fechaDesde').addEventListener('change', loadFacturas);
        document.getElementById('fechaHasta').addEventListener('change', loadFacturas);
        
        // Cargar al inicio
        loadFacturas();
    </script>
</body>
</html>
