<?php
/**
 * LexiPro - API de Compras (Facturas de Proveedores)
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Listar facturas de compras
    if ($method === 'GET') {
        $auth->requirePermission('compras', 'ver');
        
        if (isset($_GET['id'])) {
            // Factura específica con detalles
            $factura = $db->fetchOne("SELECT * FROM facturas_compras WHERE id = ?", [$_GET['id']]);
            
            if ($factura) {
                $detalles = $db->fetchAll("SELECT * FROM facturas_compras_detalle WHERE factura_id = ?", [$_GET['id']]);
                $factura['detalles'] = $detalles;
                
                // Obtener pagos
                $pagos = $db->fetchAll("SELECT * FROM pagos WHERE factura_compra_id = ? ORDER BY fecha", [$_GET['id']]);
                $factura['pagos'] = $pagos;
            }
            
            jsonSuccess(['factura' => $factura]);
        } else {
            // Listar facturas
            $search = $_GET['search'] ?? '';
            $proveedor = $_GET['proveedor'] ?? '';
            $estado = $_GET['estado'] ?? '';
            $fechaDesde = $_GET['fecha_desde'] ?? '';
            $fechaHasta = $_GET['fecha_hasta'] ?? '';
            
            $sql = "SELECT fc.*, p.razon_social as proveedor_nombre,
                           (SELECT SUM(monto) FROM pagos WHERE factura_compra_id = fc.id) as total_pagado
                    FROM facturas_compras fc
                    INNER JOIN proveedores p ON fc.proveedor_id = p.id
                    WHERE 1=1";
            $params = [];
            
            if ($search) {
                $sql .= " AND (fc.numero_factura LIKE ? OR p.razon_social LIKE ?)";
                $searchParam = "%{$search}%";
                $params = [$searchParam, $searchParam];
            }
            
            if ($proveedor) {
                $sql .= " AND fc.proveedor_id = ?";
                $params[] = $proveedor;
            }
            
            if ($estado) {
                $sql .= " AND fc.estado = ?";
                $params[] = $estado;
            }
            
            if ($fechaDesde) {
                $sql .= " AND fc.fecha >= ?";
                $params[] = $fechaDesde;
            }
            
            if ($fechaHasta) {
                $sql .= " AND fc.fecha <= ?";
                $params[] = $fechaHasta;
            }
            
            $sql .= " ORDER BY fc.fecha DESC, fc.id DESC LIMIT 100";
            
            $facturas = $db->fetchAll($sql, $params);
            jsonSuccess(['facturas' => $facturas]);
        }
    }
    
    // POST - Registrar factura de compra
    elseif ($method === 'POST') {
        $auth->requirePermission('compras', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validaciones
        if (empty($data['proveedor_id'])) {
            jsonError('Proveedor es obligatorio');
        }
        
        if (empty($data['numero_factura'])) {
            jsonError('Número de factura es obligatorio');
        }
        
        if (empty($data['detalles']) || count($data['detalles']) == 0) {
            jsonError('Debe agregar al menos un ítem a la factura');
        }
        
        // Verificar que no exista el mismo número de factura del mismo proveedor
        $existe = $db->fetchOne(
            "SELECT id FROM facturas_compras WHERE proveedor_id = ? AND numero_factura = ?",
            [$data['proveedor_id'], $data['numero_factura']]
        );
        
        if ($existe) {
            jsonError('Ya existe una factura con este número del mismo proveedor');
        }
        
        // Iniciar transacción
        $db->beginTransaction();
        
        try {
            // Calcular totales
            $subtotal5 = 0;
            $subtotal10 = 0;
            $subtotalExento = 0;
            
            foreach ($data['detalles'] as $detalle) {
                $cantidad = (float)$detalle['cantidad'];
                $precioUnitario = (float)$detalle['precio_unitario'];
                $subtotal = $cantidad * $precioUnitario;
                
                switch ($detalle['tipo_iva']) {
                    case '10':
                        $subtotal10 += $subtotal;
                        break;
                    case '5':
                        $subtotal5 += $subtotal;
                        break;
                    case 'exento':
                        $subtotalExento += $subtotal;
                        break;
                }
            }
            
            $iva5 = $subtotal5 * 0.05;
            $iva10 = $subtotal10 * 0.10;
            $totalIva = $iva5 + $iva10;
            $totalGeneral = $subtotal5 + $subtotal10 + $subtotalExento + $totalIva;
            
            // Insertar factura
            $facturaId = $db->insert('facturas_compras', [
                'numero_factura' => sanitize($data['numero_factura']),
                'timbrado' => sanitize($data['timbrado'] ?? null),
                'fecha' => $data['fecha'] ?? date('Y-m-d'),
                'proveedor_id' => $data['proveedor_id'],
                'condicion_compra' => $data['condicion_compra'] ?? 'contado',
                'subtotal_5' => $subtotal5,
                'subtotal_10' => $subtotal10,
                'subtotal_exento' => $subtotalExento,
                'iva_5' => $iva5,
                'iva_10' => $iva10,
                'total_iva' => $totalIva,
                'total_general' => $totalGeneral,
                'estado' => $data['condicion_compra'] === 'contado' ? 'pagada' : 'pendiente',
                'observaciones' => sanitize($data['observaciones'] ?? null),
                'usuario_id' => $userId
            ]);
            
            // Insertar detalles
            foreach ($data['detalles'] as $detalle) {
                $cantidad = (float)$detalle['cantidad'];
                $precioUnitario = (float)$detalle['precio_unitario'];
                $subtotal = $cantidad * $precioUnitario;
                
                $db->insert('facturas_compras_detalle', [
                    'factura_id' => $facturaId,
                    'descripcion' => sanitize($detalle['descripcion']),
                    'cantidad' => $cantidad,
                    'precio_unitario' => $precioUnitario,
                    'tipo_iva' => $detalle['tipo_iva'],
                    'subtotal' => $subtotal
                ]);
            }
            
            // Si es contado, registrar pago automático
            if ($data['condicion_compra'] === 'contado' && !empty($data['forma_pago'])) {
                $db->insert('pagos', [
                    'factura_compra_id' => $facturaId,
                    'fecha' => $data['fecha'] ?? date('Y-m-d'),
                    'monto' => $totalGeneral,
                    'forma_pago' => $data['forma_pago'],
                    'referencia' => sanitize($data['referencia_pago'] ?? null),
                    'observaciones' => 'Pago automático - Compra al contado',
                    'usuario_id' => $userId
                ]);
            }
            
            $db->commit();
            
            // Log de actividad
            logActivity($userId, 'crear', 'facturas_compras', $facturaId, "Factura de compra registrada: {$data['numero_factura']}");
            
            jsonSuccess(['id' => $facturaId], 'Factura de compra registrada exitosamente');
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    // PUT - Actualizar estado o registrar pago
    elseif ($method === 'PUT') {
        $auth->requirePermission('compras', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de factura requerido');
        }
        
        if (isset($data['registrar_pago'])) {
            // Registrar pago
            if (empty($data['monto']) || $data['monto'] <= 0) {
                jsonError('Monto de pago inválido');
            }
            
            $factura = $db->fetchOne("SELECT * FROM facturas_compras WHERE id = ?", [$data['id']]);
            $totalPagado = $db->fetchOne("SELECT COALESCE(SUM(monto), 0) as total FROM pagos WHERE factura_compra_id = ?", [$data['id']]);
            
            $saldo = $factura['total_general'] - $totalPagado['total'];
            
            if ($data['monto'] > $saldo) {
                jsonError('El monto del pago excede el saldo pendiente');
            }
            
            // Registrar pago
            $db->insert('pagos', [
                'factura_compra_id' => $data['id'],
                'fecha' => $data['fecha_pago'] ?? date('Y-m-d'),
                'monto' => $data['monto'],
                'forma_pago' => $data['forma_pago'],
                'referencia' => sanitize($data['referencia'] ?? null),
                'observaciones' => sanitize($data['observaciones_pago'] ?? null),
                'usuario_id' => $userId
            ]);
            
            // Actualizar estado de factura
            $nuevoTotal = $totalPagado['total'] + $data['monto'];
            if ($nuevoTotal >= $factura['total_general']) {
                $db->update('facturas_compras', ['estado' => 'pagada'], 'id = ?', [$data['id']]);
            } else {
                $db->update('facturas_compras', ['estado' => 'parcial'], 'id = ?', [$data['id']]);
            }
            
            logActivity($userId, 'pago', 'facturas_compras', $data['id'], "Pago registrado: " . formatMoney($data['monto']));
            
            jsonSuccess([], 'Pago registrado exitosamente');
        } else {
            jsonError('Acción no válida');
        }
    }
    
} catch (Exception $e) {
    error_log("Error en API de compras: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
