<?php
/**
 * LexiPro - Gestión de Clientes
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Clientes - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .tipo-toggle {
            display: inline-flex;
            background: #f3f4f6;
            border-radius: 8px;
            padding: 4px;
            margin-bottom: 20px;
        }
        
        .tipo-btn {
            padding: 8px 20px;
            border: none;
            background: transparent;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s;
        }
        
        .tipo-btn.active {
            background: white;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .form-section {
            display: none;
        }
        
        .form-section.active {
            display: block;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>👤 Gestión de Clientes</h1>
                <p style="color: #6b7280; margin-top: 5px;">Administra la base de clientes del estudio</p>
            </div>
            <button class="btn btn-primary" onclick="openClientModal()">+ Nuevo Cliente</button>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>🔍 Buscar</label>
                <input type="text" id="searchInput" placeholder="Nombre, RUC, email...">
            </div>
            <div class="filter-group">
                <label>Tipo</label>
                <select id="tipoFilter">
                    <option value="">Todos</option>
                    <option value="fisica">Persona Física</option>
                    <option value="juridica">Persona Jurídica</option>
                </select>
            </div>
            <div class="filter-group">
                <label>Estado</label>
                <select id="activoFilter">
                    <option value="">Todos</option>
                    <option value="1" selected>Activos</option>
                    <option value="0">Inactivos</option>
                </select>
            </div>
        </div>
        
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>Tipo</th>
                        <th>Nombre/Razón Social</th>
                        <th>RUC/Cédula</th>
                        <th>Email</th>
                        <th>Teléfono</th>
                        <th>Ciudad</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="clientesTableBody">
                    <tr>
                        <td colspan="8" class="loading">Cargando clientes...</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Modal de Cliente -->
    <div class="modal" id="clientModal">
        <div class="modal-content modal-large">
            <div class="modal-header">
                <h2 id="clientModalTitle">Nuevo Cliente</h2>
                <button class="modal-close" onclick="closeClientModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="clientForm">
                    <input type="hidden" name="id" id="clientId">
                    
                    <div class="form-group">
                        <label>Tipo de Persona *</label>
                        <div class="tipo-toggle">
                            <button type="button" class="tipo-btn active" onclick="switchTipo('fisica')">Persona Física</button>
                            <button type="button" class="tipo-btn" onclick="switchTipo('juridica')">Persona Jurídica</button>
                        </div>
                        <input type="hidden" name="tipo_persona" id="tipoPersona" value="fisica">
                    </div>
                    
                    <!-- Persona Física -->
                    <div id="fisicaSection" class="form-section active">
                        <div class="form-row">
                            <div class="form-group">
                                <label>Nombre *</label>
                                <input type="text" name="nombre" id="nombre">
                            </div>
                            <div class="form-group">
                                <label>Apellido *</label>
                                <input type="text" name="apellido" id="apellido">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Cédula</label>
                                <input type="text" name="cedula" placeholder="1.234.567">
                            </div>
                            <div class="form-group">
                                <label>Fecha de Nacimiento</label>
                                <input type="date" name="fecha_nacimiento">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Profesión</label>
                                <input type="text" name="profesion">
                            </div>
                            <div class="form-group">
                                <label>Estado Civil</label>
                                <select name="estado_civil">
                                    <option value="">Seleccionar...</option>
                                    <option value="soltero">Soltero/a</option>
                                    <option value="casado">Casado/a</option>
                                    <option value="divorciado">Divorciado/a</option>
                                    <option value="viudo">Viudo/a</option>
                                    <option value="union_libre">Unión Libre</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Persona Jurídica -->
                    <div id="juridicaSection" class="form-section">
                        <div class="form-group">
                            <label>Razón Social *</label>
                            <input type="text" name="razon_social" id="razonSocial">
                        </div>
                    </div>
                    
                    <!-- Campos Comunes -->
                    <div class="form-group">
                        <label>RUC</label>
                        <input type="text" name="ruc" placeholder="80012345-6">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Email</label>
                            <input type="email" name="email">
                        </div>
                        <div class="form-group">
                            <label>Teléfono</label>
                            <input type="text" name="telefono" placeholder="(021) 123-456">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Celular</label>
                        <input type="text" name="celular" placeholder="0981 123 456">
                    </div>
                    
                    <div class="form-group">
                        <label>Dirección</label>
                        <textarea name="direccion" rows="2" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Ciudad</label>
                            <input type="text" name="ciudad" value="Asunción">
                        </div>
                        <div class="form-group">
                            <label>País</label>
                            <input type="text" name="pais" value="Paraguay">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Notas</label>
                        <textarea name="notas" rows="3" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="activo" value="1" checked>
                            Cliente activo
                        </label>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                        Guardar Cliente
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let clientes = [];
        
        // Cargar clientes
        async function loadClientes() {
            try {
                const search = document.getElementById('searchInput').value;
                const tipo = document.getElementById('tipoFilter').value;
                const activo = document.getElementById('activoFilter').value;
                
                const params = new URLSearchParams();
                if (search) params.append('search', search);
                if (tipo) params.append('tipo', tipo);
                if (activo) params.append('activo', activo);
                
                const response = await fetch(`api.php?${params}`);
                const data = await response.json();
                
                if (data.success) {
                    clientes = data.clientes;
                    renderClientes();
                }
            } catch (error) {
                console.error('Error al cargar clientes:', error);
            }
        }
        
        // Renderizar clientes
        function renderClientes() {
            const tbody = document.getElementById('clientesTableBody');
            
            if (clientes.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" class="loading">No se encontraron clientes</td></tr>';
                return;
            }
            
            tbody.innerHTML = clientes.map(cliente => {
                const nombre = cliente.tipo_persona === 'juridica' 
                    ? cliente.razon_social 
                    : `${cliente.nombre} ${cliente.apellido || ''}`;
                
                const documento = cliente.ruc || cliente.cedula || '-';
                
                return `
                    <tr>
                        <td><span class="badge badge-${cliente.tipo_persona === 'fisica' ? 'info' : 'secondary'}">${cliente.tipo_persona === 'fisica' ? 'Física' : 'Jurídica'}</span></td>
                        <td><strong>${nombre}</strong></td>
                        <td>${documento}</td>
                        <td>${cliente.email || '-'}</td>
                        <td>${cliente.celular || cliente.telefono || '-'}</td>
                        <td>${cliente.ciudad || '-'}</td>
                        <td><span class="badge badge-${cliente.activo == 1 ? 'success' : 'danger'}">${cliente.activo == 1 ? 'Activo' : 'Inactivo'}</span></td>
                        <td>
                            <div class="actions">
                                <button class="btn btn-sm btn-edit" onclick="editCliente(${cliente.id})">Editar</button>
                                <button class="btn btn-sm btn-delete" onclick="deleteCliente(${cliente.id})">Eliminar</button>
                            </div>
                        </td>
                    </tr>
                `;
            }).join('');
        }
        
        // Cambiar tipo de persona
        function switchTipo(tipo) {
            document.querySelectorAll('.tipo-btn').forEach(btn => btn.classList.remove('active'));
            event.target.classList.add('active');
            
            document.getElementById('tipoPersona').value = tipo;
            
            if (tipo === 'fisica') {
                document.getElementById('fisicaSection').classList.add('active');
                document.getElementById('juridicaSection').classList.remove('active');
                document.getElementById('nombre').required = true;
                document.getElementById('apellido').required = true;
                document.getElementById('razonSocial').required = false;
            } else {
                document.getElementById('fisicaSection').classList.remove('active');
                document.getElementById('juridicaSection').classList.add('active');
                document.getElementById('nombre').required = false;
                document.getElementById('apellido').required = false;
                document.getElementById('razonSocial').required = true;
            }
        }
        
        // Abrir modal
        function openClientModal(clienteId = null) {
            const modal = document.getElementById('clientModal');
            const form = document.getElementById('clientForm');
            const title = document.getElementById('clientModalTitle');
            
            form.reset();
            switchTipo('fisica');
            
            if (clienteId) {
                title.textContent = 'Editar Cliente';
                const cliente = clientes.find(c => c.id == clienteId);
                if (cliente) {
                    Object.keys(cliente).forEach(key => {
                        const field = form.elements[key];
                        if (field) {
                            if (field.type === 'checkbox') {
                                field.checked = cliente[key] == 1;
                            } else {
                                field.value = cliente[key] || '';
                            }
                        }
                    });
                    switchTipo(cliente.tipo_persona);
                }
            } else {
                title.textContent = 'Nuevo Cliente';
            }
            
            modal.classList.add('active');
        }
        
        // Cerrar modal
        function closeClientModal() {
            document.getElementById('clientModal').classList.remove('active');
        }
        
        // Editar cliente
        function editCliente(id) {
            openClientModal(id);
        }
        
        // Eliminar cliente
        async function deleteCliente(id) {
            if (!confirm('¿Estás seguro de desactivar este cliente?')) return;
            
            try {
                const response = await fetch('api.php', {
                    method: 'DELETE',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({id})
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    loadClientes();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al eliminar cliente');
                console.error(error);
            }
        }
        
        // Guardar cliente
        document.getElementById('clientForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {};
            
            for (let [key, value] of formData.entries()) {
                if (key === 'activo') {
                    data[key] = formData.get(key) ? 1 : 0;
                } else {
                    data[key] = value;
                }
            }
            
            try {
                const method = data.id ? 'PUT' : 'POST';
                const response = await fetch('api.php', {
                    method,
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    closeClientModal();
                    loadClientes();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar cliente');
                console.error(error);
            }
        });
        
        // Event listeners
        document.getElementById('searchInput').addEventListener('input', LexiPro.debounce(loadClientes, 300));
        document.getElementById('tipoFilter').addEventListener('change', loadClientes);
        document.getElementById('activoFilter').addEventListener('change', loadClientes);
        
        // Cargar al inicio
        loadClientes();
    </script>
</body>
</html>
