<?php
/**
 * LexiPro - API de Clientes
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Listar clientes
    if ($method === 'GET') {
        $auth->requirePermission('clientes', 'ver');
        
        if (isset($_GET['id'])) {
            // Cliente específico
            $cliente = $db->fetchOne("SELECT * FROM clientes WHERE id = ?", [$_GET['id']]);
            jsonSuccess(['cliente' => $cliente]);
        } else {
            // Listar todos
            $search = $_GET['search'] ?? '';
            $tipo = $_GET['tipo'] ?? '';
            $activo = $_GET['activo'] ?? '';
            
            $sql = "SELECT * FROM clientes WHERE 1=1";
            $params = [];
            
            if ($search) {
                $sql .= " AND (nombre LIKE ? OR apellido LIKE ? OR razon_social LIKE ? OR ruc LIKE ? OR email LIKE ?)";
                $searchParam = "%{$search}%";
                $params = array_fill(0, 5, $searchParam);
            }
            
            if ($tipo) {
                $sql .= " AND tipo_persona = ?";
                $params[] = $tipo;
            }
            
            if ($activo !== '') {
                $sql .= " AND activo = ?";
                $params[] = (int)$activo;
            }
            
            $sql .= " ORDER BY created_at DESC LIMIT 100";
            
            $clientes = $db->fetchAll($sql, $params);
            jsonSuccess(['clientes' => $clientes]);
        }
    }
    
    // POST - Crear cliente
    elseif ($method === 'POST') {
        $auth->requirePermission('clientes', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validaciones básicas
        if (empty($data['tipo_persona'])) {
            jsonError('El tipo de persona es obligatorio');
        }
        
        if ($data['tipo_persona'] === 'fisica') {
            if (empty($data['nombre']) || empty($data['apellido'])) {
                jsonError('Nombre y apellido son obligatorios para personas físicas');
            }
        } else {
            if (empty($data['razon_social'])) {
                jsonError('Razón social es obligatoria para personas jurídicas');
            }
        }
        
        // Validar RUC si existe
        if (!empty($data['ruc'])) {
            // Verificar que no exista otro cliente con el mismo RUC
            $existente = $db->fetchOne("SELECT id FROM clientes WHERE ruc = ? AND id != ?", [
                $data['ruc'],
                $data['id'] ?? 0
            ]);
            
            if ($existente) {
                jsonError('Ya existe un cliente con este RUC');
            }
        }
        
        $clienteId = $db->insert('clientes', [
            'tipo_persona' => $data['tipo_persona'],
            'nombre' => sanitize($data['nombre'] ?? null),
            'apellido' => sanitize($data['apellido'] ?? null),
            'razon_social' => sanitize($data['razon_social'] ?? null),
            'ruc' => sanitize($data['ruc'] ?? null),
            'cedula' => sanitize($data['cedula'] ?? null),
            'email' => sanitize($data['email'] ?? null),
            'telefono' => sanitize($data['telefono'] ?? null),
            'celular' => sanitize($data['celular'] ?? null),
            'direccion' => sanitize($data['direccion'] ?? null),
            'ciudad' => sanitize($data['ciudad'] ?? null),
            'pais' => sanitize($data['pais'] ?? 'Paraguay'),
            'fecha_nacimiento' => $data['fecha_nacimiento'] ?? null,
            'profesion' => sanitize($data['profesion'] ?? null),
            'estado_civil' => $data['estado_civil'] ?? null,
            'notas' => sanitize($data['notas'] ?? null),
            'activo' => (int)($data['activo'] ?? 1)
        ]);
        
        // Log de actividad
        logActivity($userId, 'crear', 'clientes', $clienteId, "Cliente creado: " . ($data['razon_social'] ?? $data['nombre'] . ' ' . $data['apellido']));
        
        jsonSuccess(['id' => $clienteId], 'Cliente creado exitosamente');
    }
    
    // PUT - Actualizar cliente
    elseif ($method === 'PUT') {
        $auth->requirePermission('clientes', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de cliente requerido');
        }
        
        // Validar RUC si existe
        if (!empty($data['ruc'])) {
            $existente = $db->fetchOne("SELECT id FROM clientes WHERE ruc = ? AND id != ?", [
                $data['ruc'],
                $data['id']
            ]);
            
            if ($existente) {
                jsonError('Ya existe un cliente con este RUC');
            }
        }
        
        $db->update('clientes', [
            'tipo_persona' => $data['tipo_persona'],
            'nombre' => sanitize($data['nombre'] ?? null),
            'apellido' => sanitize($data['apellido'] ?? null),
            'razon_social' => sanitize($data['razon_social'] ?? null),
            'ruc' => sanitize($data['ruc'] ?? null),
            'cedula' => sanitize($data['cedula'] ?? null),
            'email' => sanitize($data['email'] ?? null),
            'telefono' => sanitize($data['telefono'] ?? null),
            'celular' => sanitize($data['celular'] ?? null),
            'direccion' => sanitize($data['direccion'] ?? null),
            'ciudad' => sanitize($data['ciudad'] ?? null),
            'pais' => sanitize($data['pais'] ?? 'Paraguay'),
            'fecha_nacimiento' => $data['fecha_nacimiento'] ?? null,
            'profesion' => sanitize($data['profesion'] ?? null),
            'estado_civil' => $data['estado_civil'] ?? null,
            'notas' => sanitize($data['notas'] ?? null),
            'activo' => (int)($data['activo'] ?? 1)
        ], 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'actualizar', 'clientes', $data['id'], "Cliente actualizado");
        
        jsonSuccess([], 'Cliente actualizado exitosamente');
    }
    
    // DELETE - Eliminar cliente (desactivar)
    elseif ($method === 'DELETE') {
        $auth->requirePermission('clientes', 'eliminar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de cliente requerido');
        }
        
        // Verificar si tiene casos asociados
        $casos = $db->fetchOne("SELECT COUNT(*) as total FROM casos WHERE cliente_id = ?", [$data['id']]);
        
        if ($casos['total'] > 0) {
            jsonError('No se puede eliminar un cliente con casos asociados. Desactívalo en su lugar.');
        }
        
        // Desactivar en lugar de eliminar
        $db->update('clientes', ['activo' => 0], 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'desactivar', 'clientes', $data['id'], "Cliente desactivado");
        
        jsonSuccess([], 'Cliente desactivado exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de clientes: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
