<?php
/**
 * LexiPro - Gestión de Chat Web
 * Interfaz para que secretarios/admins respondan a visitantes
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Obtener sesiones de chat web
$sesiones = $db->fetchAll(
    "SELECT 
        sesion_id,
        MAX(nombre) as nombre,
        MAX(apellido) as apellido,
        MAX(email) as email,
        MAX(created_at) as ultima_actividad,
        COUNT(*) as total_mensajes,
        SUM(CASE WHEN es_cliente = 1 AND leido = 0 THEN 1 ELSE 0 END) as no_leidos
     FROM chat_web
     GROUP BY sesion_id
     ORDER BY ultima_actividad DESC
     LIMIT 50"
);
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chat Web - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .chat-web-container {
            display: grid;
            grid-template-columns: 350px 1fr;
            gap: 0;
            height: calc(100vh - 100px);
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        .sessions-sidebar {
            border-right: 1px solid #e5e7eb;
            display: flex;
            flex-direction: column;
        }
        
        .sessions-header {
            padding: 20px;
            border-bottom: 1px solid #e5e7eb;
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            color: white;
        }
        
        .sessions-header h3 {
            margin: 0 0 5px 0;
            font-size: 18px;
        }
        
        .sessions-list {
            flex: 1;
            overflow-y: auto;
        }
        
        .session-item {
            padding: 15px 20px;
            border-bottom: 1px solid #f3f4f6;
            cursor: pointer;
            transition: background 0.2s;
        }
        
        .session-item:hover {
            background: #f9fafb;
        }
        
        .session-item.active {
            background: #eff6ff;
            border-left: 3px solid var(--primary);
        }
        
        .session-item.unread {
            background: #fef2f2;
        }
        
        .session-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 5px;
        }
        
        .session-name {
            font-weight: 600;
            color: #1f2937;
        }
        
        .session-badge {
            background: #ef4444;
            color: white;
            border-radius: 12px;
            padding: 2px 8px;
            font-size: 11px;
            font-weight: bold;
        }
        
        .session-email {
            font-size: 12px;
            color: #6b7280;
            margin-bottom: 3px;
        }
        
        .session-time {
            font-size: 11px;
            color: #9ca3af;
        }
        
        .chat-area {
            display: flex;
            flex-direction: column;
        }
        
        .chat-area-header {
            padding: 20px;
            border-bottom: 1px solid #e5e7eb;
            background: #f9fafb;
        }
        
        .visitor-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .visitor-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            font-weight: bold;
        }
        
        .visitor-details h3 {
            margin: 0 0 3px 0;
            font-size: 16px;
        }
        
        .visitor-details p {
            margin: 0;
            font-size: 13px;
            color: #6b7280;
        }
        
        .messages-area {
            flex: 1;
            overflow-y: auto;
            padding: 20px;
            background: #f9fafb;
        }
        
        .web-message {
            margin-bottom: 20px;
            display: flex;
            gap: 12px;
        }
        
        .web-message.agent {
            flex-direction: row-reverse;
        }
        
        .web-message-avatar {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            flex-shrink: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            font-size: 14px;
        }
        
        .web-message.visitor .web-message-avatar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .web-message.agent .web-message-avatar {
            background: var(--primary);
            color: white;
        }
        
        .web-message-content {
            max-width: 60%;
        }
        
        .web-message-bubble {
            padding: 12px 16px;
            border-radius: 12px;
            word-wrap: break-word;
        }
        
        .web-message.visitor .web-message-bubble {
            background: white;
            box-shadow: 0 1px 2px rgba(0,0,0,0.05);
        }
        
        .web-message.agent .web-message-bubble {
            background: var(--primary);
            color: white;
        }
        
        .web-message-time {
            font-size: 11px;
            color: #9ca3af;
            margin-top: 4px;
        }
        
        .reply-area {
            padding: 20px;
            border-top: 1px solid #e5e7eb;
            background: white;
        }
        
        .reply-input-wrapper {
            display: flex;
            gap: 10px;
        }
        
        .reply-input {
            flex: 1;
            padding: 12px 16px;
            border: 2px solid #e5e7eb;
            border-radius: 24px;
            font-size: 14px;
            resize: none;
            font-family: inherit;
        }
        
        .reply-btn {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: var(--primary);
            color: white;
            border: none;
            cursor: pointer;
            font-size: 20px;
        }
        
        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: #9ca3af;
        }
        
        .empty-icon {
            font-size: 64px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>🌐 Chat Web</h1>
                <p style="color: #6b7280; margin-top: 5px;">Gestión de consultas de visitantes del sitio web</p>
            </div>
        </div>
        
        <div class="chat-web-container">
            <!-- Sidebar de sesiones -->
            <div class="sessions-sidebar">
                <div class="sessions-header">
                    <h3>💬 Conversaciones</h3>
                    <p style="margin: 0; font-size: 13px; opacity: 0.9;">Visitantes del sitio web</p>
                </div>
                
                <div class="sessions-list">
                    <?php if (empty($sesiones)): ?>
                        <p style="padding: 20px; text-align: center; color: #9ca3af;">No hay conversaciones</p>
                    <?php else: ?>
                        <?php foreach ($sesiones as $sesion): ?>
                            <div class="session-item <?= $sesion['no_leidos'] > 0 ? 'unread' : '' ?>" 
                                 onclick="openSession('<?= e($sesion['sesion_id']) ?>', '<?= e($sesion['nombre']) ?>', '<?= e($sesion['apellido']) ?>', '<?= e($sesion['email']) ?>')">
                                <div class="session-header">
                                    <span class="session-name"><?= e($sesion['nombre'] . ' ' . $sesion['apellido']) ?></span>
                                    <?php if ($sesion['no_leidos'] > 0): ?>
                                        <span class="session-badge"><?= $sesion['no_leidos'] ?></span>
                                    <?php endif; ?>
                                </div>
                                <div class="session-email">📧 <?= e($sesion['email']) ?></div>
                                <div class="session-time">
                                    💬 <?= $sesion['total_mensajes'] ?> mensajes • <?= formatDateTime($sesion['ultima_actividad']) ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Área de chat -->
            <div class="chat-area">
                <div id="emptyState" class="empty-state">
                    <div class="empty-icon">💬</div>
                    <h3>Selecciona una conversación</h3>
                    <p>Elige un visitante de la lista para responder</p>
                </div>
                
                <div id="chatArea" style="display: none; height: 100%; display: flex; flex-direction: column;">
                    <div class="chat-area-header">
                        <div class="visitor-info">
                            <div class="visitor-avatar" id="visitorAvatar">?</div>
                            <div class="visitor-details">
                                <h3 id="visitorName">Visitante</h3>
                                <p id="visitorEmail">email@ejemplo.com</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="messages-area" id="messagesArea">
                        <!-- Los mensajes se cargarán aquí -->
                    </div>
                    
                    <div class="reply-area">
                        <div class="reply-input-wrapper">
                            <textarea class="reply-input" id="replyInput" placeholder="Escribe tu respuesta..." rows="1"></textarea>
                            <button class="reply-btn" onclick="sendReply()">➤</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let currentSession = null;
        let refreshInterval = null;
        
        // Abrir sesión
        function openSession(sesionId, nombre, apellido, email) {
            currentSession = { sesion_id: sesionId, nombre, apellido, email };
            
            document.getElementById('emptyState').style.display = 'none';
            document.getElementById('chatArea').style.display = 'flex';
            
            document.getElementById('visitorName').textContent = `${nombre} ${apellido}`;
            document.getElementById('visitorEmail').textContent = email;
            document.getElementById('visitorAvatar').textContent = (nombre[0] + apellido[0]).toUpperCase();
            
            // Marcar items activos
            document.querySelectorAll('.session-item').forEach(item => {
                item.classList.remove('active');
            });
            event.currentTarget.classList.add('active');
            event.currentTarget.classList.remove('unread');
            
            loadMessages();
            markAsRead();
            
            // Actualizar cada 3 segundos
            if (refreshInterval) clearInterval(refreshInterval);
            refreshInterval = setInterval(loadMessages, 3000);
        }
        
        // Cargar mensajes
        async function loadMessages() {
            if (!currentSession) return;
            
            try {
                const response = await fetch(`api_web.php?sesion_id=${currentSession.sesion_id}`);
                const data = await response.json();
                
                if (data.success) {
                    renderMessages(data.mensajes);
                }
            } catch (error) {
                console.error('Error al cargar mensajes:', error);
            }
        }
        
        // Renderizar mensajes
        function renderMessages(mensajes) {
            const container = document.getElementById('messagesArea');
            
            container.innerHTML = mensajes.map(msg => `
                <div class="web-message ${msg.es_cliente == 1 ? 'visitor' : 'agent'}">
                    <div class="web-message-avatar">
                        ${msg.es_cliente == 1 
                            ? (msg.nombre[0] + msg.apellido[0]).toUpperCase()
                            : (msg.usuario_nombre ? msg.usuario_nombre[0] + msg.usuario_apellido[0] : 'A').toUpperCase()
                        }
                    </div>
                    <div class="web-message-content">
                        <div class="web-message-bubble">${msg.mensaje}</div>
                        <div class="web-message-time">${formatTime(msg.created_at)}</div>
                    </div>
                </div>
            `).join('');
            
            container.scrollTop = container.scrollHeight;
        }
        
        // Enviar respuesta
        async function sendReply() {
            if (!currentSession) return;
            
            const input = document.getElementById('replyInput');
            const mensaje = input.value.trim();
            
            if (!mensaje) return;
            
            try {
                const response = await fetch('api_web.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        sesion_id: currentSession.sesion_id,
                        mensaje: mensaje,
                        es_cliente: 0,
                        usuario_id: <?= $currentUser['id'] ?>
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    input.value = '';
                    loadMessages();
                }
            } catch (error) {
                console.error('Error al enviar respuesta:', error);
            }
        }
        
        // Marcar como leído
        async function markAsRead() {
            if (!currentSession) return;
            
            try {
                await fetch('api_web.php', {
                    method: 'PUT',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        sesion_id: currentSession.sesion_id
                    })
                });
            } catch (error) {
                console.error('Error al marcar como leído:', error);
            }
        }
        
        // Helpers
        function formatTime(datetime) {
            const date = new Date(datetime);
            return date.toLocaleTimeString('es-PY', { hour: '2-digit', minute: '2-digit' });
        }
        
        // Event listener
        document.getElementById('replyInput').addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendReply();
            }
        });
    </script>
</body>
</html>
