<?php
/**
 * LexiPro - Chat Interno
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Obtener todos los usuarios para iniciar conversaciones
$usuarios = $db->fetchAll("SELECT id, nombre, apellido, cargo FROM usuarios WHERE activo = 1 AND id != ? ORDER BY nombre", [$currentUser['id']]);
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chat - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .chat-container {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 0;
            height: calc(100vh - 100px);
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        .chat-sidebar {
            border-right: 1px solid #e5e7eb;
            display: flex;
            flex-direction: column;
        }
        
        .chat-sidebar-header {
            padding: 20px;
            border-bottom: 1px solid #e5e7eb;
        }
        
        .chat-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .chat-tab {
            flex: 1;
            padding: 8px;
            background: #f3f4f6;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            font-size: 13px;
        }
        
        .chat-tab.active {
            background: var(--primary);
            color: white;
        }
        
        .chat-search {
            width: 100%;
            padding: 10px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .chat-list {
            flex: 1;
            overflow-y: auto;
        }
        
        .chat-item {
            padding: 15px 20px;
            border-bottom: 1px solid #f3f4f6;
            cursor: pointer;
            transition: background 0.2s;
        }
        
        .chat-item:hover {
            background: #f9fafb;
        }
        
        .chat-item.active {
            background: #eff6ff;
            border-left: 3px solid var(--primary);
        }
        
        .chat-item-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 5px;
        }
        
        .chat-item-name {
            font-weight: 600;
            color: #1f2937;
        }
        
        .chat-item-time {
            font-size: 11px;
            color: #9ca3af;
        }
        
        .chat-item-preview {
            font-size: 13px;
            color: #6b7280;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .chat-badge {
            background: #ef4444;
            color: white;
            border-radius: 12px;
            padding: 2px 8px;
            font-size: 11px;
            font-weight: bold;
        }
        
        .chat-main {
            display: flex;
            flex-direction: column;
        }
        
        .chat-header {
            padding: 20px;
            border-bottom: 1px solid #e5e7eb;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .chat-header h3 {
            margin: 0;
            font-size: 18px;
        }
        
        .chat-messages-container {
            flex: 1;
            overflow-y: auto;
            padding: 20px;
            background: #f9fafb;
        }
        
        .chat-message {
            margin-bottom: 20px;
            display: flex;
            gap: 12px;
        }
        
        .chat-message.own {
            flex-direction: row-reverse;
        }
        
        .message-avatar {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            background: var(--primary);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            flex-shrink: 0;
        }
        
        .message-content {
            max-width: 60%;
        }
        
        .message-bubble {
            background: white;
            padding: 12px 16px;
            border-radius: 12px;
            box-shadow: 0 1px 2px rgba(0,0,0,0.05);
        }
        
        .chat-message.own .message-bubble {
            background: var(--primary);
            color: white;
        }
        
        .message-meta {
            font-size: 11px;
            color: #9ca3af;
            margin-top: 4px;
        }
        
        .chat-input-container {
            padding: 20px;
            border-top: 1px solid #e5e7eb;
            background: white;
        }
        
        .chat-input-wrapper {
            display: flex;
            gap: 10px;
        }
        
        .chat-input {
            flex: 1;
            padding: 12px 16px;
            border: 2px solid #e5e7eb;
            border-radius: 24px;
            font-size: 14px;
            resize: none;
            font-family: inherit;
        }
        
        .chat-send-btn {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: var(--primary);
            color: white;
            border: none;
            cursor: pointer;
            font-size: 20px;
        }
        
        .empty-chat {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: #9ca3af;
        }
        
        .empty-chat-icon {
            font-size: 64px;
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .chat-container {
                grid-template-columns: 1fr;
            }
            
            .chat-sidebar {
                display: none;
            }
            
            .chat-sidebar.show {
                display: flex;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>💬 Chat</h1>
                <p style="color: #6b7280; margin-top: 5px;">Mensajería interna del equipo</p>
            </div>
            <button class="btn btn-primary" onclick="showNewChatModal()">+ Nueva Conversación</button>
        </div>
        
        <div class="chat-container">
            <!-- Sidebar -->
            <div class="chat-sidebar">
                <div class="chat-sidebar-header">
                    <div class="chat-tabs">
                        <button class="chat-tab active" onclick="switchTab('chats')">Chats</button>
                        <button class="chat-tab" onclick="switchTab('grupos')">Grupos</button>
                    </div>
                    <input type="text" class="chat-search" placeholder="Buscar..." id="chatSearch">
                </div>
                
                <div class="chat-list" id="chatList">
                    <!-- Las conversaciones se cargarán aquí -->
                </div>
            </div>
            
            <!-- Área principal -->
            <div class="chat-main">
                <div id="emptyState" class="empty-chat">
                    <div class="empty-chat-icon">💬</div>
                    <h3>Selecciona una conversación</h3>
                    <p>Elige un chat de la lista o inicia uno nuevo</p>
                </div>
                
                <div id="chatArea" style="display: none; height: 100%; display: flex; flex-direction: column;">
                    <div class="chat-header">
                        <div>
                            <h3 id="chatTitle">Chat</h3>
                            <p id="chatSubtitle" style="margin: 0; font-size: 13px; color: #6b7280;"></p>
                        </div>
                    </div>
                    
                    <div class="chat-messages-container" id="messagesContainer">
                        <!-- Los mensajes se cargarán aquí -->
                    </div>
                    
                    <div class="chat-input-container">
                        <div class="chat-input-wrapper">
                            <textarea class="chat-input" id="messageInput" placeholder="Escribe un mensaje..." rows="1"></textarea>
                            <button class="chat-send-btn" onclick="sendMessage()">➤</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Modal Nueva Conversación -->
    <div class="modal" id="newChatModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Nueva Conversación</h2>
                <button class="modal-close" onclick="closeNewChatModal()">×</button>
            </div>
            <div class="modal-body">
                <div class="form-group">
                    <label>Selecciona un usuario</label>
                    <select id="newChatUser" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;">
                        <option value="">Seleccionar...</option>
                        <?php foreach ($usuarios as $usuario): ?>
                            <option value="<?= $usuario['id'] ?>"><?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?> - <?= e($usuario['cargo'] ?? '') ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <button class="btn btn-primary" onclick="startNewChat()" style="width: 100%; margin-top: 20px;">Iniciar Chat</button>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let currentChat = null;
        let currentTab = 'chats';
        let refreshInterval = null;
        
        // Cargar conversaciones
        async function loadConversations() {
            try {
                const response = await fetch('api.php?conversaciones=1');
                const data = await response.json();
                
                if (data.success) {
                    renderConversations(data.conversaciones);
                }
            } catch (error) {
                console.error('Error al cargar conversaciones:', error);
            }
        }
        
        // Renderizar conversaciones
        function renderConversations(conversaciones) {
            const container = document.getElementById('chatList');
            
            if (conversaciones.length === 0) {
                container.innerHTML = '<p style="padding: 20px; text-align: center; color: #9ca3af;">No hay conversaciones</p>';
                return;
            }
            
            container.innerHTML = conversaciones.map(conv => `
                <div class="chat-item ${currentChat && currentChat.usuario_id == conv.usuario_id ? 'active' : ''}" 
                     onclick="openChat(${conv.usuario_id}, '${conv.nombre}', '${conv.apellido}')">
                    <div class="chat-item-header">
                        <span class="chat-item-name">${conv.nombre} ${conv.apellido}</span>
                        <span class="chat-item-time">${conv.ultima_fecha ? formatTime(conv.ultima_fecha) : ''}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <div class="chat-item-preview">${conv.ultimo_mensaje || 'Sin mensajes'}</div>
                        ${conv.no_leidos > 0 ? `<span class="chat-badge">${conv.no_leidos}</span>` : ''}
                    </div>
                </div>
            `).join('');
        }
        
        // Abrir chat
        async function openChat(usuarioId, nombre, apellido) {
            currentChat = { usuario_id: usuarioId, nombre, apellido };
            
            document.getElementById('emptyState').style.display = 'none';
            document.getElementById('chatArea').style.display = 'flex';
            document.getElementById('chatTitle').textContent = `${nombre} ${apellido}`;
            
            loadMessages();
            
            // Actualizar cada 3 segundos
            if (refreshInterval) clearInterval(refreshInterval);
            refreshInterval = setInterval(loadMessages, 3000);
        }
        
        // Cargar mensajes
        async function loadMessages() {
            if (!currentChat) return;
            
            try {
                const response = await fetch(`api.php?con=${currentChat.usuario_id}`);
                const data = await response.json();
                
                if (data.success) {
                    renderMessages(data.mensajes);
                }
            } catch (error) {
                console.error('Error al cargar mensajes:', error);
            }
        }
        
        // Renderizar mensajes
        function renderMessages(mensajes) {
            const container = document.getElementById('messagesContainer');
            
            container.innerHTML = mensajes.map(msg => `
                <div class="chat-message ${msg.emisor_id == <?= $currentUser['id'] ?> ? 'own' : ''}">
                    <div class="message-avatar">${getInitials(msg.emisor_nombre, msg.emisor_apellido)}</div>
                    <div class="message-content">
                        <div class="message-bubble">${msg.mensaje}</div>
                        <div class="message-meta">${formatTime(msg.created_at)}</div>
                    </div>
                </div>
            `).join('');
            
            container.scrollTop = container.scrollHeight;
        }
        
        // Enviar mensaje
        async function sendMessage() {
            if (!currentChat) return;
            
            const input = document.getElementById('messageInput');
            const mensaje = input.value.trim();
            
            if (!mensaje) return;
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        receptor_id: currentChat.usuario_id,
                        mensaje: mensaje
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    input.value = '';
                    loadMessages();
                    loadConversations();
                }
            } catch (error) {
                console.error('Error al enviar mensaje:', error);
            }
        }
        
        // Cambiar tab
        function switchTab(tab) {
            currentTab = tab;
            document.querySelectorAll('.chat-tab').forEach(t => t.classList.remove('active'));
            event.target.classList.add('active');
            
            if (tab === 'chats') {
                loadConversations();
            } else {
                // Cargar grupos (implementar después)
            }
        }
        
        // Modal nueva conversación
        function showNewChatModal() {
            document.getElementById('newChatModal').classList.add('active');
        }
        
        function closeNewChatModal() {
            document.getElementById('newChatModal').classList.remove('active');
        }
        
        function startNewChat() {
            const userId = document.getElementById('newChatUser').value;
            if (!userId) {
                alert('Selecciona un usuario');
                return;
            }
            
            const option = document.getElementById('newChatUser').selectedOptions[0];
            const fullName = option.text.split(' - ')[0].split(' ');
            const nombre = fullName[0];
            const apellido = fullName.slice(1).join(' ');
            
            closeNewChatModal();
            openChat(parseInt(userId), nombre, apellido);
        }
        
        // Helpers
        function getInitials(nombre, apellido) {
            return (nombre[0] + (apellido ? apellido[0] : '')).toUpperCase();
        }
        
        function formatTime(datetime) {
            const date = new Date(datetime);
            const now = new Date();
            const diff = now - date;
            
            if (diff < 86400000) { // Menos de 24 horas
                return date.toLocaleTimeString('es-PY', { hour: '2-digit', minute: '2-digit' });
            } else {
                return date.toLocaleDateString('es-PY', { day: '2-digit', month: '2-digit' });
            }
        }
        
        // Event listeners
        document.getElementById('messageInput').addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });
        
        // Cargar al inicio
        loadConversations();
    </script>
</body>
</html>
