<?php
/**
 * LexiPro - API de Chat Web (Público)
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$db = Database::getInstance();
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Obtener mensajes de una sesión
    if ($method === 'GET') {
        $sesionId = $_GET['sesion_id'] ?? '';
        
        if (empty($sesionId)) {
            jsonError('ID de sesión requerido');
        }
        
        $mensajes = $db->fetchAll(
            "SELECT cw.*, u.nombre as usuario_nombre, u.apellido as usuario_apellido
             FROM chat_web cw
             LEFT JOIN usuarios u ON cw.usuario_id = u.id
             WHERE cw.sesion_id = ?
             ORDER BY cw.created_at ASC",
            [$sesionId]
        );
        
        jsonSuccess(['mensajes' => $mensajes]);
    }
    
    // POST - Enviar mensaje
    elseif ($method === 'POST') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['sesion_id']) || empty($data['mensaje'])) {
            jsonError('Sesión y mensaje son obligatorios');
        }
        
        // Si es el primer mensaje, validar datos del visitante
        if ($data['es_cliente'] == 1) {
            if (empty($data['nombre']) || empty($data['apellido']) || empty($data['email'])) {
                jsonError('Nombre, apellido y email son obligatorios');
            }
        }
        
        $mensajeId = $db->insert('chat_web', [
            'sesion_id' => sanitize($data['sesion_id']),
            'nombre' => sanitize($data['nombre'] ?? ''),
            'apellido' => sanitize($data['apellido'] ?? ''),
            'email' => sanitize($data['email'] ?? ''),
            'mensaje' => sanitize($data['mensaje']),
            'es_cliente' => (int)($data['es_cliente'] ?? 1),
            'usuario_id' => $data['usuario_id'] ?? null
        ]);
        
        // Si es mensaje del visitante, notificar a secretarios/administradores
        if ($data['es_cliente'] == 1) {
            // Obtener usuarios con rol de secretario o admin
            $secretarios = $db->fetchAll(
                "SELECT u.id FROM usuarios u 
                 INNER JOIN roles r ON u.rol_id = r.id 
                 WHERE r.nombre IN ('Secretario', 'Administrador General') AND u.activo = 1"
            );
            
            foreach ($secretarios as $secretario) {
                createNotification(
                    $secretario['id'],
                    'chat',
                    'Nuevo mensaje en chat web',
                    "Mensaje de {$data['nombre']} {$data['apellido']}",
                    "/modules/chat/web.php"
                );
            }
        }
        
        jsonSuccess(['id' => $mensajeId], 'Mensaje enviado');
    }
    
    // PUT - Marcar mensajes como leídos
    elseif ($method === 'PUT') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['sesion_id'])) {
            jsonError('ID de sesión requerido');
        }
        
        $db->query(
            "UPDATE chat_web SET leido = 1 WHERE sesion_id = ? AND es_cliente = 1 AND leido = 0",
            [$data['sesion_id']]
        );
        
        jsonSuccess([], 'Mensajes marcados como leídos');
    }
    
} catch (Exception $e) {
    error_log("Error en API de chat web: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
