<?php
/**
 * LexiPro - API de Chat Interno
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Obtener conversaciones o mensajes
    if ($method === 'GET') {
        $auth->requirePermission('chat', 'ver');
        
        // Obtener conversaciones del usuario
        if (isset($_GET['conversaciones'])) {
            $sql = "SELECT DISTINCT
                        CASE 
                            WHEN ci.emisor_id = ? THEN ci.receptor_id
                            ELSE ci.emisor_id
                        END as usuario_id,
                        u.nombre, u.apellido, u.cargo,
                        (SELECT mensaje FROM chat_interno 
                         WHERE (emisor_id = ? AND receptor_id = usuario_id) 
                            OR (emisor_id = usuario_id AND receptor_id = ?)
                         ORDER BY created_at DESC LIMIT 1) as ultimo_mensaje,
                        (SELECT created_at FROM chat_interno 
                         WHERE (emisor_id = ? AND receptor_id = usuario_id) 
                            OR (emisor_id = usuario_id AND receptor_id = ?)
                         ORDER BY created_at DESC LIMIT 1) as ultima_fecha,
                        (SELECT COUNT(*) FROM chat_interno 
                         WHERE emisor_id = usuario_id AND receptor_id = ? AND leido = 0) as no_leidos
                    FROM chat_interno ci
                    INNER JOIN usuarios u ON u.id = CASE 
                        WHEN ci.emisor_id = ? THEN ci.receptor_id
                        ELSE ci.emisor_id
                    END
                    WHERE ci.emisor_id = ? OR ci.receptor_id = ?
                    ORDER BY ultima_fecha DESC";
            
            $conversaciones = $db->fetchAll($sql, [
                $userId, $userId, $userId, $userId, $userId, $userId, $userId, $userId, $userId
            ]);
            
            jsonSuccess(['conversaciones' => $conversaciones]);
        }
        
        // Obtener mensajes de una conversación
        elseif (isset($_GET['con'])) {
            $otroUsuario = (int)$_GET['con'];
            
            $mensajes = $db->fetchAll(
                "SELECT ci.*, 
                        e.nombre as emisor_nombre, e.apellido as emisor_apellido,
                        r.nombre as receptor_nombre, r.apellido as receptor_apellido
                 FROM chat_interno ci
                 INNER JOIN usuarios e ON ci.emisor_id = e.id
                 INNER JOIN usuarios r ON ci.receptor_id = r.id
                 WHERE (ci.emisor_id = ? AND ci.receptor_id = ?)
                    OR (ci.emisor_id = ? AND ci.receptor_id = ?)
                 ORDER BY ci.created_at ASC",
                [$userId, $otroUsuario, $otroUsuario, $userId]
            );
            
            // Marcar como leídos
            $db->query(
                "UPDATE chat_interno SET leido = 1 
                 WHERE emisor_id = ? AND receptor_id = ? AND leido = 0",
                [$otroUsuario, $userId]
            );
            
            jsonSuccess(['mensajes' => $mensajes]);
        }
        
        // Obtener grupos del usuario
        elseif (isset($_GET['grupos'])) {
            $grupos = $db->fetchAll(
                "SELECT cg.*, 
                        (SELECT COUNT(*) FROM chat_grupos_miembros WHERE grupo_id = cg.id) as total_miembros,
                        (SELECT COUNT(*) FROM chat_grupos_miembros WHERE grupo_id = cg.id AND usuario_id = ?) as es_miembro
                 FROM chat_grupos cg
                 WHERE cg.creado_por = ? 
                    OR cg.id IN (SELECT grupo_id FROM chat_grupos_miembros WHERE usuario_id = ?)
                 ORDER BY cg.created_at DESC",
                [$userId, $userId, $userId]
            );
            
            jsonSuccess(['grupos' => $grupos]);
        }
        
        // Obtener mensajes de un grupo
        elseif (isset($_GET['grupo'])) {
            $grupoId = (int)$_GET['grupo'];
            
            // Verificar que el usuario es miembro
            $esMiembro = $db->fetchOne(
                "SELECT id FROM chat_grupos_miembros WHERE grupo_id = ? AND usuario_id = ?",
                [$grupoId, $userId]
            );
            
            if (!$esMiembro) {
                jsonError('No tienes acceso a este grupo');
            }
            
            $mensajes = $db->fetchAll(
                "SELECT ci.*, u.nombre, u.apellido
                 FROM chat_interno ci
                 INNER JOIN usuarios u ON ci.emisor_id = u.id
                 WHERE ci.grupo_id = ?
                 ORDER BY ci.created_at ASC",
                [$grupoId]
            );
            
            jsonSuccess(['mensajes' => $mensajes]);
        }
        
        // Contar mensajes no leídos
        elseif (isset($_GET['no_leidos'])) {
            $count = $db->fetchOne(
                "SELECT COUNT(*) as total FROM chat_interno 
                 WHERE receptor_id = ? AND leido = 0",
                [$userId]
            );
            
            jsonSuccess(['no_leidos' => $count['total']]);
        }
    }
    
    // POST - Enviar mensaje
    elseif ($method === 'POST') {
        $auth->requirePermission('chat', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['mensaje'])) {
            jsonError('El mensaje no puede estar vacío');
        }
        
        // Mensaje individual
        if (!empty($data['receptor_id'])) {
            $mensajeId = $db->insert('chat_interno', [
                'emisor_id' => $userId,
                'receptor_id' => (int)$data['receptor_id'],
                'mensaje' => sanitize($data['mensaje'])
            ]);
            
            // Notificar al receptor
            createNotification(
                $data['receptor_id'],
                'chat',
                'Nuevo mensaje',
                'Tienes un nuevo mensaje en el chat',
                "/modules/chat/"
            );
        }
        // Mensaje grupal
        elseif (!empty($data['grupo_id'])) {
            $mensajeId = $db->insert('chat_interno', [
                'emisor_id' => $userId,
                'grupo_id' => (int)$data['grupo_id'],
                'mensaje' => sanitize($data['mensaje'])
            ]);
            
            // Notificar a todos los miembros del grupo
            $miembros = $db->fetchAll(
                "SELECT usuario_id FROM chat_grupos_miembros WHERE grupo_id = ? AND usuario_id != ?",
                [$data['grupo_id'], $userId]
            );
            
            $grupo = $db->fetchOne("SELECT nombre FROM chat_grupos WHERE id = ?", [$data['grupo_id']]);
            
            foreach ($miembros as $miembro) {
                createNotification(
                    $miembro['usuario_id'],
                    'chat',
                    "Mensaje en {$grupo['nombre']}",
                    'Nuevo mensaje en el grupo',
                    "/modules/chat/?grupo={$data['grupo_id']}"
                );
            }
        } else {
            jsonError('Debes especificar un receptor o grupo');
        }
        
        jsonSuccess(['id' => $mensajeId], 'Mensaje enviado');
    }
    
    // PUT - Marcar como leído
    elseif ($method === 'PUT') {
        $auth->requirePermission('chat', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!empty($data['emisor_id'])) {
            $db->query(
                "UPDATE chat_interno SET leido = 1 
                 WHERE emisor_id = ? AND receptor_id = ?",
                [$data['emisor_id'], $userId]
            );
        }
        
        jsonSuccess([], 'Mensajes marcados como leídos');
    }
    
} catch (Exception $e) {
    error_log("Error en API de chat interno: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
