<?php
/**
 * LexiPro - Vista Detallada de Caso
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();
$auth->requirePermission('casos', 'ver');

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Obtener ID del caso
$casoId = $_GET['id'] ?? null;
if (!$casoId) {
    header('Location: index.php');
    exit;
}

// Obtener datos del caso
$caso = $db->fetchOne(
    "SELECT c.*, cl.nombre as cliente_nombre, cl.apellido as cliente_apellido, cl.email as cliente_email,
            cl.telefono as cliente_telefono, u.nombre as responsable_nombre, u.apellido as responsable_apellido
     FROM casos c
     INNER JOIN clientes cl ON c.cliente_id = cl.id
     INNER JOIN usuarios u ON c.responsable_id = u.id
     WHERE c.id = ?",
    [$casoId]
);

if (!$caso) {
    header('Location: index.php');
    exit;
}

// Obtener usuarios asignados
$usuariosAsignados = $db->fetchAll(
    "SELECT cu.*, u.nombre, u.apellido, u.email, u.cargo
     FROM casos_usuarios cu
     INNER JOIN usuarios u ON cu.usuario_id = u.id
     WHERE cu.caso_id = ?",
    [$casoId]
);

// Obtener todos los usuarios para el selector
$todosUsuarios = $db->fetchAll("SELECT id, nombre, apellido FROM usuarios WHERE activo = 1 ORDER BY nombre");
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($caso['nombre']) ?> - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .page-header {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
            padding: 30px;
            border-radius: 12px;
            margin-bottom: 20px;
        }
        
        .breadcrumb {
            font-size: 14px;
            margin-bottom: 10px;
            opacity: 0.9;
        }
        
        .breadcrumb a {
            color: white;
            text-decoration: none;
        }
        
        .case-title {
            font-size: 28px;
            margin-bottom: 10px;
        }
        
        .case-meta {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .meta-item {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .content-grid {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 20px;
        }
        
        .info-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        .info-row {
            display: grid;
            grid-template-columns: 140px 1fr;
            padding: 12px 0;
            border-bottom: 1px solid #f3f4f6;
        }
        
        .info-row:last-child {
            border-bottom: none;
        }
        
        .info-label {
            font-weight: 600;
            color: #6b7280;
        }
        
        .info-value {
            color: #1f2937;
        }
        
        .team-member {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px;
            background: #f9fafb;
            border-radius: 8px;
            margin-bottom: 10px;
        }
        
        .team-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
        }
        
        .team-info {
            flex: 1;
        }
        
        .team-name {
            font-weight: 600;
            color: #1f2937;
        }
        
        .team-role {
            font-size: 13px;
            color: #6b7280;
        }
        
        .task-list {
            margin-top: 20px;
        }
        
        .task-item {
            background: white;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 10px;
            border-left: 4px solid #e5e7eb;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .task-item:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .task-item.plazo-juridico {
            border-left-color: #ef4444;
            background: #fef2f2;
        }
        
        .task-item.completada {
            opacity: 0.6;
        }
        
        .task-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 8px;
        }
        
        .task-title {
            font-weight: 600;
            color: #1f2937;
        }
        
        .task-meta {
            display: flex;
            gap: 15px;
            font-size: 13px;
            color: #6b7280;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #9ca3af;
        }
        
        .empty-state-icon {
            font-size: 48px;
            margin-bottom: 15px;
        }
        
        @media (max-width: 968px) {
            .content-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="page-header">
            <div class="breadcrumb">
                <a href="index.php">← Casos</a> / <?= e($caso['numero_caso']) ?>
            </div>
            <h1 class="case-title"><?= e($caso['nombre']) ?></h1>
            <div class="case-meta">
                <div class="meta-item">
                    <span>📋</span>
                    <span><?= e($caso['numero_caso']) ?></span>
                </div>
                <div class="meta-item">
                    <span>👤</span>
                    <span><?= e($caso['cliente_nombre'] . ' ' . ($caso['cliente_apellido'] ?? '')) ?></span>
                </div>
                <div class="meta-item">
                    <span>⚖️</span>
                    <span><?= e($caso['responsable_nombre'] . ' ' . $caso['responsable_apellido']) ?></span>
                </div>
                <div class="meta-item">
                    <span class="badge badge-<?= $caso['estado'] === 'cerrado' ? 'success' : 'info' ?>">
                        <?= ucfirst(str_replace('_', ' ', $caso['estado'])) ?>
                    </span>
                </div>
                <div class="meta-item">
                    <span class="badge badge-<?= $caso['prioridad'] === 'urgente' ? 'danger' : ($caso['prioridad'] === 'alta' ? 'warning' : 'secondary') ?>">
                        Prioridad: <?= ucfirst($caso['prioridad']) ?>
                    </span>
                </div>
            </div>
        </div>
        
        <div class="content-grid">
            <!-- Columna Principal -->
            <div>
                <!-- Tareas -->
                <div class="card">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                        <h2 style="font-size: 20px;">📝 Tareas</h2>
                        <?php if ($auth->hasPermission('casos', 'editar')): ?>
                            <button class="btn btn-primary btn-sm" onclick="openTaskModal()">+ Nueva Tarea</button>
                        <?php endif; ?>
                    </div>
                    
                    <div id="tasksList" class="task-list">
                        <div class="loading">Cargando tareas...</div>
                    </div>
                </div>
                
                <!-- Descripción -->
                <?php if ($caso['descripcion']): ?>
                <div class="card">
                    <h3 style="margin-bottom: 15px;">📄 Descripción</h3>
                    <p style="color: #4b5563; line-height: 1.6;"><?= nl2br(e($caso['descripcion'])) ?></p>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Columna Lateral -->
            <div>
                <!-- Información del Caso -->
                <div class="info-card">
                    <h3 style="margin-bottom: 15px;">ℹ️ Información</h3>
                    
                    <div class="info-row">
                        <div class="info-label">Tipo:</div>
                        <div class="info-value"><?= e($caso['tipo_caso'] ?? 'No especificado') ?></div>
                    </div>
                    
                    <div class="info-row">
                        <div class="info-label">Materia:</div>
                        <div class="info-value"><?= e($caso['materia'] ?? 'No especificado') ?></div>
                    </div>
                    
                    <?php if ($caso['juzgado']): ?>
                    <div class="info-row">
                        <div class="info-label">Juzgado:</div>
                        <div class="info-value"><?= e($caso['juzgado']) ?></div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($caso['expediente']): ?>
                    <div class="info-row">
                        <div class="info-label">Expediente:</div>
                        <div class="info-value"><?= e($caso['expediente']) ?></div>
                    </div>
                    <?php endif; ?>
                    
                    <div class="info-row">
                        <div class="info-label">Fecha Inicio:</div>
                        <div class="info-value"><?= formatDate($caso['fecha_inicio']) ?></div>
                    </div>
                    
                    <?php if ($caso['fecha_fin']): ?>
                    <div class="info-row">
                        <div class="info-label">Fecha Fin:</div>
                        <div class="info-value"><?= formatDate($caso['fecha_fin']) ?></div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($caso['monto_estimado']): ?>
                    <div class="info-row">
                        <div class="info-label">Monto:</div>
                        <div class="info-value"><?= formatMoney($caso['monto_estimado']) ?></div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Cliente -->
                <div class="info-card" style="margin-top: 20px;">
                    <h3 style="margin-bottom: 15px;">👤 Cliente</h3>
                    
                    <div class="info-row">
                        <div class="info-label">Nombre:</div>
                        <div class="info-value"><?= e($caso['cliente_nombre'] . ' ' . ($caso['cliente_apellido'] ?? '')) ?></div>
                    </div>
                    
                    <?php if ($caso['cliente_email']): ?>
                    <div class="info-row">
                        <div class="info-label">Email:</div>
                        <div class="info-value"><a href="mailto:<?= e($caso['cliente_email']) ?>"><?= e($caso['cliente_email']) ?></a></div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($caso['cliente_telefono']): ?>
                    <div class="info-row">
                        <div class="info-label">Teléfono:</div>
                        <div class="info-value"><a href="tel:<?= e($caso['cliente_telefono']) ?>"><?= e($caso['cliente_telefono']) ?></a></div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Equipo -->
                <div class="info-card" style="margin-top: 20px;">
                    <h3 style="margin-bottom: 15px;">👥 Equipo</h3>
                    
                    <!-- Responsable -->
                    <div class="team-member">
                        <img src="../../assets/img/default-avatar.png" alt="Responsable" class="team-avatar">
                        <div class="team-info">
                            <div class="team-name"><?= e($caso['responsable_nombre'] . ' ' . $caso['responsable_apellido']) ?></div>
                            <div class="team-role">Responsable Principal</div>
                        </div>
                    </div>
                    
                    <!-- Usuarios asignados -->
                    <?php foreach ($usuariosAsignados as $usuario): ?>
                    <div class="team-member">
                        <img src="../../assets/img/default-avatar.png" alt="<?= e($usuario['nombre']) ?>" class="team-avatar">
                        <div class="team-info">
                            <div class="team-name"><?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?></div>
                            <div class="team-role"><?= e($usuario['cargo'] ?? 'Miembro del equipo') ?></div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Modal de Tarea -->
    <div class="modal" id="taskModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="taskModalTitle">Nueva Tarea</h2>
                <button class="modal-close" onclick="closeTaskModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="taskForm">
                    <input type="hidden" name="id" id="taskId">
                    <input type="hidden" name="caso_id" value="<?= $casoId ?>">
                    
                    <div class="form-group">
                        <label>Título *</label>
                        <input type="text" name="titulo" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Descripción</label>
                        <textarea name="descripcion" rows="3" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Responsable *</label>
                            <select name="responsable_id" required>
                                <?php foreach ($todosUsuarios as $usuario): ?>
                                    <option value="<?= $usuario['id'] ?>"><?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Estado</label>
                            <select name="estado">
                                <option value="pendiente">Pendiente</option>
                                <option value="en_proceso">En Proceso</option>
                                <option value="completada">Completada</option>
                                <option value="cancelada">Cancelada</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Fecha Inicio</label>
                            <input type="date" name="fecha_inicio">
                        </div>
                        <div class="form-group">
                            <label>Fecha Límite</label>
                            <input type="date" name="fecha_limite">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Horas Estimadas</label>
                            <input type="number" name="horas_estimadas" step="0.5" min="0">
                        </div>
                        <div class="form-group">
                            <label>Horas Trabajadas</label>
                            <input type="number" name="horas_trabajadas" step="0.5" min="0" value="0">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Prioridad</label>
                        <select name="prioridad">
                            <option value="baja">Baja</option>
                            <option value="media" selected>Media</option>
                            <option value="alta">Alta</option>
                            <option value="urgente">Urgente</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="es_plazo_juridico" value="1">
                            <strong style="color: #ef4444;">⚠️ Es un plazo jurídico (se notificará a todo el equipo)</strong>
                        </label>
                    </div>
                    
                    <div class="form-group">
                        <label>Notas</label>
                        <textarea name="notas" rows="3" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                        Guardar Tarea
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let tareas = [];
        
        // Cargar tareas
        async function loadTasks() {
            try {
                const response = await fetch(`../tareas/api.php?caso_id=<?= $casoId ?>`);
                const data = await response.json();
                
                if (data.success) {
                    tareas = data.tareas;
                    renderTasks();
                }
            } catch (error) {
                console.error('Error al cargar tareas:', error);
            }
        }
        
        // Renderizar tareas
        function renderTasks() {
            const container = document.getElementById('tasksList');
            
            if (tareas.length === 0) {
                container.innerHTML = `
                    <div class="empty-state">
                        <div class="empty-state-icon">📋</div>
                        <p>No hay tareas creadas para este caso</p>
                    </div>
                `;
                return;
            }
            
            container.innerHTML = tareas.map(tarea => `
                <div class="task-item ${tarea.es_plazo_juridico == 1 ? 'plazo-juridico' : ''} ${tarea.estado === 'completada' ? 'completada' : ''}" onclick="editTask(${tarea.id})">
                    <div class="task-header">
                        <div>
                            ${tarea.es_plazo_juridico == 1 ? '<span style="color: #ef4444; font-weight: bold;">⚠️ PLAZO JURÍDICO</span><br>' : ''}
                            <span class="task-title">${tarea.titulo}</span>
                        </div>
                        <span class="badge badge-${getEstadoBadge(tarea.estado)}">${getEstadoLabel(tarea.estado)}</span>
                    </div>
                    ${tarea.descripcion ? `<p style="color: #6b7280; font-size: 14px; margin-bottom: 8px;">${tarea.descripcion}</p>` : ''}
                    <div class="task-meta">
                        <span>👤 ${tarea.responsable_nombre} ${tarea.responsable_apellido}</span>
                        ${tarea.fecha_limite ? `<span>📅 ${LexiPro.formatDate(tarea.fecha_limite)}</span>` : ''}
                        ${tarea.horas_trabajadas > 0 ? `<span>⏱️ ${tarea.horas_trabajadas}h trabajadas</span>` : ''}
                        <span class="badge badge-${getPrioridadBadge(tarea.prioridad)}">${tarea.prioridad}</span>
                    </div>
                </div>
            `).join('');
        }
        
                }
            }
            
            try {
                const method = data.id ? 'PUT' : 'POST';
                const response = await fetch('../tareas/api.php', {
                    method,
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    closeTaskModal();
                    loadTasks();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar tarea');
                console.error(error);
            }
        });
        
        // Helpers
        function getEstadoLabel(estado) {
            const labels = {
                'pendiente': 'Pendiente',
                'en_proceso': 'En Proceso',
                'completada': 'Completada',
                'cancelada': 'Cancelada'
            };
            return labels[estado] || estado;
        }
        
        function getEstadoBadge(estado) {
            const badges = {
                'pendiente': 'secondary',
                'en_proceso': 'info',
                'completada': 'success',
                'cancelada': 'danger'
            };
            return badges[estado] || 'secondary';
        }
        
        function getPrioridadBadge(prioridad) {
            const badges = {
                'baja': 'secondary',
                'media': 'info',
                'alta': 'warning',
                'urgente': 'danger'
            };
            return badges[prioridad] || 'secondary';
        }
        
        // Cargar al inicio
        loadTasks();
    </script>
</body>
</html>
