<?php
/**
 * LexiPro - Gestión de Casos
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
        }
        
        .view-btn {
            padding: 10px 20px;
            border: 2px solid #e5e7eb;
            background: white;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .view-btn.active {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }
        
        /* Vista Kanban */
        .kanban-board {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .kanban-column {
            background: #f9fafb;
            border-radius: 12px;
            padding: 15px;
            min-height: 400px;
        }
        
        .kanban-header {
            font-weight: 600;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #e5e7eb;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .kanban-count {
            background: var(--primary);
            color: white;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 12px;
        }
        
        .caso-card {
            background: white;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 10px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            cursor: pointer;
            transition: all 0.3s;
            border-left: 4px solid transparent;
        }
        
        .caso-card:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            transform: translateY(-2px);
        }
        
        .caso-card.prioridad-urgente { border-left-color: #ef4444; }
        .caso-card.prioridad-alta { border-left-color: #f59e0b; }
        .caso-card.prioridad-media { border-left-color: #3b82f6; }
        .caso-card.prioridad-baja { border-left-color: #6b7280; }
        
        .caso-numero {
            font-size: 12px;
            color: #6b7280;
            font-weight: 600;
        }
        
        .caso-nombre {
            font-weight: 600;
            margin: 5px 0;
            color: #1f2937;
        }
        
        .caso-cliente {
            font-size: 13px;
            color: #6b7280;
            margin-bottom: 10px;
        }
        
        .caso-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 10px;
            padding-top: 10px;
            border-top: 1px solid #f3f4f6;
        }
        
        .caso-avatar {
            width: 24px;
            height: 24px;
            border-radius: 50%;
            border: 2px solid white;
        }
        
        .caso-progress {
            font-size: 12px;
            color: #6b7280;
        }
        
        /* Vista Tabla */
        .table-view {
            display: none;
        }
        
        .table-view.active {
            display: block;
        }
        
        .kanban-view {
            display: none;
        }
        
        .kanban-view.active {
            display: block;
        }
        
        /* Modal grande para casos */
        .modal-content.modal-large {
            max-width: 900px;
        }
        
        .tabs {
            display: flex;
            border-bottom: 2px solid #e5e7eb;
            margin-bottom: 20px;
        }
        
        .tab {
            padding: 12px 24px;
            cursor: pointer;
            border-bottom: 3px solid transparent;
            margin-bottom: -2px;
            transition: all 0.3s;
        }
        
        .tab.active {
            border-bottom-color: var(--primary);
            color: var(--primary);
            font-weight: 600;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .form-row-3 {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr;
            gap: 15px;
        }
        
        .multi-select {
            min-height: 120px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>⚖️ Gestión de Casos</h1>
                <p style="color: #6b7280; margin-top: 5px;">Administra los casos jurídicos del estudio</p>
            </div>
            <div style="display: flex; gap: 10px;">
                <div class="view-toggle">
                    <button class="view-btn active" onclick="switchView('kanban')">📋 Kanban</button>
                    <button class="view-btn" onclick="switchView('table')">📊 Tabla</button>
                </div>
                <?php if ($auth->hasPermission('casos', 'crear')): ?>
                    <button class="btn btn-primary" onclick="openCaseModal()">+ Nuevo Caso</button>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <label>🔍 Buscar</label>
                <input type="text" id="searchInput" placeholder="Número, nombre, carátula...">
            </div>
            <div class="filter-group">
                <label>Estado</label>
                <select id="estadoFilter">
                    <option value="">Todos</option>
                    <option value="abierto">Abierto</option>
                    <option value="en_proceso">En Proceso</option>
                    <option value="suspendido">Suspendido</option>
                    <option value="cerrado">Cerrado</option>
                    <option value="archivado">Archivado</option>
                </select>
            </div>
            <div class="filter-group">
                <label>Prioridad</label>
                <select id="prioridadFilter">
                    <option value="">Todas</option>
                    <option value="urgente">Urgente</option>
                    <option value="alta">Alta</option>
                    <option value="media">Media</option>
                    <option value="baja">Baja</option>
                </select>
            </div>
            <div class="filter-group">
                <label>Responsable</label>
                <select id="responsableFilter">
                    <option value="">Todos</option>
                    <?php foreach ($usuarios as $usuario): ?>
                        <option value="<?= $usuario['id'] ?>"><?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
        
        <!-- Vista Kanban -->
        <div class="kanban-view active">
            <div class="kanban-board" id="kanbanBoard">
                <div class="kanban-column">
                    <div class="kanban-header">
                        <span>📂 Abierto</span>
                        <span class="kanban-count" id="count-abierto">0</span>
                    </div>
                    <div id="column-abierto"></div>
                </div>
                
                <div class="kanban-column">
                    <div class="kanban-header">
                        <span>⚙️ En Proceso</span>
                        <span class="kanban-count" id="count-en_proceso">0</span>
                    </div>
                    <div id="column-en_proceso"></div>
                </div>
                
                <div class="kanban-column">
                    <div class="kanban-header">
                        <span>⏸️ Suspendido</span>
                        <span class="kanban-count" id="count-suspendido">0</span>
                    </div>
                    <div id="column-suspendido"></div>
                </div>
                
                <div class="kanban-column">
                    <div class="kanban-header">
                        <span>✅ Cerrado</span>
                        <span class="kanban-count" id="count-cerrado">0</span>
                    </div>
                    <div id="column-cerrado"></div>
                </div>
            </div>
        </div>
        
        <!-- Vista Tabla -->
        <div class="table-view">
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Número</th>
                            <th>Caso</th>
                            <th>Cliente</th>
                            <th>Responsable</th>
                            <th>Estado</th>
                            <th>Prioridad</th>
                            <th>Tareas</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody id="casosTableBody">
                        <tr>
                            <td colspan="8" class="loading">Cargando casos...</td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- Modal de Caso -->
    <div class="modal" id="caseModal">
        <div class="modal-content modal-large">
            <div class="modal-header">
                <h2 id="caseModalTitle">Nuevo Caso</h2>
                <button class="modal-close" onclick="closeCaseModal()">×</button>
            </div>
            <div class="modal-body">
                <div class="tabs">
                    <div class="tab active" onclick="switchTab('datos')">Datos del Caso</div>
                    <div class="tab" onclick="switchTab('tareas')" id="tareasTab" style="display: none;">Tareas</div>
                    <div class="tab" onclick="switchTab('archivos')" id="archivosTab" style="display: none;">Archivos</div>
                </div>
                
                <!-- Tab: Datos del Caso -->
                <div class="tab-content active" id="tab-datos">
                    <form id="caseForm">
                        <input type="hidden" id="casoId" name="id">
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Nombre del Caso *</label>
                                <input type="text" name="nombre" required>
                            </div>
                            <div class="form-group">
                                <label>Número de Caso</label>
                                <input type="text" id="numeroCaso" readonly style="background: #f3f4f6;" placeholder="Se genera automáticamente">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Carátula</label>
                            <input type="text" name="caratula" placeholder="Ej: Juan Pérez c/ María González s/ Divorcio">
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Cliente *</label>
                                <select name="cliente_id" id="clienteSelect" required>
                                    <option value="">Seleccionar cliente...</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Responsable *</label>
                                <select name="responsable_id" required>
                                    <?php foreach ($usuarios as $usuario): ?>
                                        <option value="<?= $usuario['id'] ?>"><?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row-3">
                            <div class="form-group">
                                <label>Tipo de Caso</label>
                                <select name="tipo_caso">
                                    <option value="">Seleccionar...</option>
                                    <option value="civil">Civil</option>
                                    <option value="penal">Penal</option>
                                    <option value="laboral">Laboral</option>
                                    <option value="comercial">Comercial</option>
                                    <option value="familia">Familia</option>
                                    <option value="administrativo">Administrativo</option>
                                    <option value="otro">Otro</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Materia</label>
                                <input type="text" name="materia" placeholder="Ej: Divorcio, Desalojo, etc.">
                            </div>
                            <div class="form-group">
                                <label>Expediente</label>
                                <input type="text" name="expediente">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Juzgado</label>
                            <input type="text" name="juzgado" placeholder="Ej: Juzgado de Primera Instancia en lo Civil">
                        </div>
                        
                        <div class="form-row-3">
                            <div class="form-group">
                                <label>Fecha de Inicio *</label>
                                <input type="date" name="fecha_inicio" value="<?= date('Y-m-d') ?>" required>
                            </div>
                            <div class="form-group">
                                <label>Estado</label>
                                <select name="estado">
                                    <option value="abierto">Abierto</option>
                                    <option value="en_proceso">En Proceso</option>
                                    <option value="suspendido">Suspendido</option>
                                    <option value="cerrado">Cerrado</option>
                                    <option value="archivado">Archivado</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Prioridad</label>
                                <select name="prioridad">
                                    <option value="baja">Baja</option>
                                    <option value="media" selected>Media</option>
                                    <option value="alta">Alta</option>
                                    <option value="urgente">Urgente</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Monto Estimado (Gs.)</label>
                            <input type="number" name="monto_estimado" step="0.01">
                        </div>
                        
                        <div class="form-group">
                            <label>Usuarios Asignados</label>
                            <select name="usuarios_asignados[]" multiple class="multi-select">
                                <?php foreach ($usuarios as $usuario): ?>
                                    <option value="<?= $usuario['id'] ?>"><?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?></option>
                                <?php endforeach; ?>
                            </select>
                            <small style="color: #6b7280;">Mantén presionado Ctrl (Cmd en Mac) para seleccionar múltiples</small>
                        </div>
                        
                        <div class="form-group">
                            <label>Descripción</label>
                            <textarea name="descripcion" rows="4" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                        </div>
                        
                        <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                            Guardar Caso
                        </button>
                    </form>
                </div>
                
                <!-- Tab: Tareas -->
                <div class="tab-content" id="tab-tareas">
                    <p style="text-align: center; color: #6b7280; padding: 40px;">
                        Las tareas se gestionan después de crear el caso
                    </p>
                </div>
                
                <!-- Tab: Archivos -->
                <div class="tab-content" id="tab-archivos">
                    <p style="text-align: center; color: #6b7280; padding: 40px;">
                        Los archivos se gestionan después de crear el caso
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <script>
        let casos = [];
        let clientes = [];
        let currentView = 'kanban';
        
        // Cargar casos
        async function loadCasos() {
            try {
                const search = document.getElementById('searchInput').value;
                const estado = document.getElementById('estadoFilter').value;
                const prioridad = document.getElementById('prioridadFilter').value;
                const responsable = document.getElementById('responsableFilter').value;
                
                const params = new URLSearchParams();
                if (search) params.append('search', search);
                if (estado) params.append('estado', estado);
                if (prioridad) params.append('prioridad', prioridad);
                if (responsable) params.append('responsable', responsable);
                
                const response = await fetch(`api.php?${params}`);
                const data = await response.json();
                
                if (data.success) {
                    casos = data.casos;
                    if (currentView === 'kanban') {
                        renderKanban();
                    } else {
                        renderTable();
                    }
                }
            } catch (error) {
                console.error('Error al cargar casos:', error);
            }
        }
        
        // Cargar clientes
        async function loadClientes() {
            try {
                const response = await fetch('../clientes/api.php');
                const data = await response.json();
                
                if (data.success) {
                    clientes = data.clientes;
                    renderClienteSelect();
                }
            } catch (error) {
                console.error('Error al cargar clientes:', error);
            }
        }
        
        // Renderizar select de clientes
        function renderClienteSelect() {
            const select = document.getElementById('clienteSelect');
            select.innerHTML = '<option value="">Seleccionar cliente...</option>';
            
            clientes.forEach(cliente => {
                const nombre = cliente.tipo_persona === 'juridica' 
                    ? cliente.razon_social 
                    : `${cliente.nombre} ${cliente.apellido || ''}`;
                select.innerHTML += `<option value="${cliente.id}">${nombre}</option>`;
            });
        }
        
        // Renderizar Kanban
        function renderKanban() {
            const estados = ['abierto', 'en_proceso', 'suspendido', 'cerrado'];
            
            estados.forEach(estado => {
                const column = document.getElementById(`column-${estado}`);
                const casosFiltrados = casos.filter(c => c.estado === estado);
                
                document.getElementById(`count-${estado}`).textContent = casosFiltrados.length;
                
                if (casosFiltrados.length === 0) {
                    column.innerHTML = '<p style="text-align: center; color: #9ca3af; padding: 20px; font-size: 14px;">No hay casos</p>';
                    return;
                }
                
                column.innerHTML = casosFiltrados.map(caso => `
                    <div class="caso-card prioridad-${caso.prioridad}" onclick="viewCaso(${caso.id})">
                        <div class="caso-numero">${caso.numero_caso}</div>
                        <div class="caso-nombre">${caso.nombre}</div>
                        <div class="caso-cliente">👤 ${caso.cliente_nombre} ${caso.cliente_apellido || ''}</div>
                        <div class="caso-footer">
                            <div>
                                <img src="../../assets/img/default-avatar.png" alt="${caso.responsable_nombre}" class="caso-avatar" title="${caso.responsable_nombre} ${caso.responsable_apellido}">
                            </div>
                            <div class="caso-progress">
                                ${caso.tareas_completadas || 0}/${caso.total_tareas || 0} tareas
                            </div>
                        </div>
                    </div>
                `).join('');
            });
        }
        
        // Renderizar Tabla
        function renderTable() {
            const tbody = document.getElementById('casosTableBody');
            
            if (casos.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" class="loading">No se encontraron casos</td></tr>';
                return;
            }
            
            tbody.innerHTML = casos.map(caso => `
                <tr>
                    <td><strong>${caso.numero_caso}</strong></td>
                    <td>${caso.nombre}</td>
                    <td>${caso.cliente_nombre} ${caso.cliente_apellido || ''}</td>
                    <td>${caso.responsable_nombre} ${caso.responsable_apellido}</td>
                    <td><span class="badge badge-info">${getEstadoLabel(caso.estado)}</span></td>
                    <td><span class="badge ${getPrioridadClass(caso.prioridad)}">${getPrioridadLabel(caso.prioridad)}</span></td>
                    <td>${caso.tareas_completadas || 0}/${caso.total_tareas || 0}</td>
                    <td>
                        <div class="actions">
                            <button class="btn btn-sm btn-info" onclick="viewCaso(${caso.id})">Ver</button>
                            <?php if ($auth->hasPermission('casos', 'editar')): ?>
                                <button class="btn btn-sm btn-edit" onclick="editCaso(${caso.id})">Editar</button>
                            <?php endif; ?>
                            <?php if ($auth->hasPermission('casos', 'eliminar')): ?>
                                <button class="btn btn-sm btn-delete" onclick="deleteCaso(${caso.id})">Eliminar</button>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
            `).join('');
        }
        
        // Cambiar vista
        function switchView(view) {
            currentView = view;
            
            document.querySelectorAll('.view-btn').forEach(btn => btn.classList.remove('active'));
            event.target.classList.add('active');
            
            document.querySelector('.kanban-view').classList.toggle('active', view === 'kanban');
            document.querySelector('.table-view').classList.toggle('active', view === 'table');
            
            if (view === 'table') {
                renderTable();
            } else {
                renderKanban();
            }
        }
        
        // Abrir modal
        function openCaseModal(casoId = null) {
            const modal = document.getElementById('caseModal');
            const form = document.getElementById('caseForm');
            const title = document.getElementById('caseModalTitle');
            
            form.reset();
            
            if (casoId) {
                title.textContent = 'Editar Caso';
                const caso = casos.find(c => c.id == casoId);
                if (caso) {
                    Object.keys(caso).forEach(key => {
                        const field = form.elements[key];
                        if (field) {
                            field.value = caso[key] || '';
                        }
                    });
                    document.getElementById('numeroCaso').value = caso.numero_caso;
                    document.getElementById('tareasTab').style.display = 'block';
                    document.getElementById('archivosTab').style.display = 'block';
                }
            } else {
                title.textContent = 'Nuevo Caso';
                document.getElementById('numeroCaso').value = '';
                document.getElementById('tareasTab').style.display = 'none';
                document.getElementById('archivosTab').style.display = 'none';
            }
            
            modal.classList.add('active');
        }
        
        // Cerrar modal
        function closeCaseModal() {
            document.getElementById('caseModal').classList.remove('active');
        }
        
        // Cambiar tab
        function switchTab(tabName) {
            document.querySelectorAll('.tab').forEach(tab => tab.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
            
            event.target.classList.add('active');
            document.getElementById(`tab-${tabName}`).classList.add('active');
        }
        
        // Guardar caso
        document.getElementById('caseForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {};
            
            for (let [key, value] of formData.entries()) {
                if (key === 'usuarios_asignados[]') {
                    if (!data.usuarios_asignados) data.usuarios_asignados = [];
                    data.usuarios_asignados.push(value);
                } else {
                    data[key] = value;
                }
            }
            
            try {
                const method = data.id ? 'PUT' : 'POST';
                const response = await fetch('api.php', {
                    method,
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    closeCaseModal();
                    loadCasos();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar caso');
                console.error(error);
            }
        });
        
        // Ver caso
        function viewCaso(id) {
            window.location.href = `view.php?id=${id}`;
        }
        
        // Editar caso
        function editCaso(id) {
            openCaseModal(id);
        }
        
        // Eliminar caso
        async function deleteCaso(id) {
            if (!confirm('¿Estás seguro de eliminar este caso?')) return;
            
            try {
                const response = await fetch('api.php', {
                    method: 'DELETE',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({id})
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    loadCasos();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al eliminar caso');
                console.error(error);
            }
        }
        
        // Helpers
        function getEstadoLabel(estado) {
            const labels = {
                'abierto': 'Abierto',
                'en_proceso': 'En Proceso',
                'suspendido': 'Suspendido',
                'cerrado': 'Cerrado',
                'archivado': 'Archivado'
            };
            return labels[estado] || estado;
        }
        
        function getPrioridadLabel(prioridad) {
            const labels = {
                'baja': 'Baja',
                'media': 'Media',
                'alta': 'Alta',
                'urgente': 'Urgente'
            };
            return labels[prioridad] || prioridad;
        }
        
        function getPrioridadClass(prioridad) {
            const classes = {
                'baja': 'badge-secondary',
                'media': 'badge-info',
                'alta': 'badge-warning',
                'urgente': 'badge-danger'
            };
            return classes[prioridad] || 'badge-secondary';
        }
        
        // Event listeners
        document.getElementById('searchInput').addEventListener('input', LexiPro.debounce(loadCasos, 300));
        document.getElementById('estadoFilter').addEventListener('change', loadCasos);
        document.getElementById('prioridadFilter').addEventListener('change', loadCasos);
        document.getElementById('responsableFilter').addEventListener('change', loadCasos);
        
        // Cargar al inicio
        loadCasos();
        loadClientes();
    </script>
</body>
</html>
