<?php
/**
 * LexiPro - API de Casos
 */

session_start();
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$method = $_SERVER['REQUEST_METHOD'];
$userId = $_SESSION['user_id'];

try {
    // GET - Listar casos
    if ($method === 'GET') {
        $auth->requirePermission('casos', 'ver');
        
        if (isset($_GET['id'])) {
            // Obtener un caso específico
            $caso = $db->fetchOne(
                "SELECT c.*, cl.nombre as cliente_nombre, cl.apellido as cliente_apellido,
                        u.nombre as responsable_nombre, u.apellido as responsable_apellido
                 FROM casos c
                 INNER JOIN clientes cl ON c.cliente_id = cl.id
                 INNER JOIN usuarios u ON c.responsable_id = u.id
                 WHERE c.id = ?",
                [$_GET['id']]
            );
            
            if ($caso) {
                // Obtener usuarios asignados
                $caso['usuarios_asignados'] = $db->fetchAll(
                    "SELECT cu.*, u.nombre, u.apellido 
                     FROM casos_usuarios cu
                     INNER JOIN usuarios u ON cu.usuario_id = u.id
                     WHERE cu.caso_id = ?",
                    [$_GET['id']]
                );
                
                jsonSuccess(['caso' => $caso]);
            } else {
                jsonError('Caso no encontrado', 404);
            }
        } else {
            // Listar todos los casos
            $search = $_GET['search'] ?? '';
            $estado = $_GET['estado'] ?? '';
            $prioridad = $_GET['prioridad'] ?? '';
            $responsable = $_GET['responsable'] ?? '';
            
            $sql = "SELECT c.*, cl.nombre as cliente_nombre, cl.apellido as cliente_apellido,
                           u.nombre as responsable_nombre, u.apellido as responsable_apellido
                    FROM casos c
                    INNER JOIN clientes cl ON c.cliente_id = cl.id
                    INNER JOIN usuarios u ON c.responsable_id = u.id
                    WHERE 1=1";
            $params = [];
            
            if ($search) {
                $sql .= " AND (c.nombre LIKE ? OR c.numero_caso LIKE ? OR c.caratula LIKE ?)";
                $searchParam = "%{$search}%";
                $params = array_merge($params, [$searchParam, $searchParam, $searchParam]);
            }
            
            if ($estado) {
                $sql .= " AND c.estado = ?";
                $params[] = $estado;
            }
            
            if ($prioridad) {
                $sql .= " AND c.prioridad = ?";
                $params[] = $prioridad;
            }
            
            if ($responsable) {
                $sql .= " AND c.responsable_id = ?";
                $params[] = $responsable;
            }
            
            $sql .= " ORDER BY c.created_at DESC";
            
            $casos = $db->fetchAll($sql, $params);
            
            // Obtener conteo de tareas por caso
            foreach ($casos as &$caso) {
                $tareas = $db->fetchOne(
                    "SELECT COUNT(*) as total, 
                            SUM(CASE WHEN estado = 'completada' THEN 1 ELSE 0 END) as completadas
                     FROM tareas WHERE caso_id = ?",
                    [$caso['id']]
                );
                $caso['total_tareas'] = $tareas['total'];
                $caso['tareas_completadas'] = $tareas['completadas'];
            }
            
            jsonSuccess(['casos' => $casos]);
        }
    }
    
    // POST - Crear caso
    elseif ($method === 'POST') {
        $auth->requirePermission('casos', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validaciones
        if (empty($data['nombre']) || empty($data['cliente_id']) || empty($data['responsable_id'])) {
            jsonError('Nombre, cliente y responsable son obligatorios');
        }
        
        // Generar número de caso
        $numeroCaso = generateCaseNumber();
        
        // Crear caso
        $db->beginTransaction();
        
        try {
            $casoId = $db->insert('casos', [
                'numero_caso' => $numeroCaso,
                'nombre' => sanitize($data['nombre']),
                'caratula' => sanitize($data['caratula'] ?? null),
                'cliente_id' => (int)$data['cliente_id'],
                'responsable_id' => (int)$data['responsable_id'],
                'tipo_caso' => sanitize($data['tipo_caso'] ?? null),
                'materia' => sanitize($data['materia'] ?? null),
                'juzgado' => sanitize($data['juzgado'] ?? null),
                'expediente' => sanitize($data['expediente'] ?? null),
                'fecha_inicio' => $data['fecha_inicio'] ?? date('Y-m-d'),
                'estado' => $data['estado'] ?? 'abierto',
                'prioridad' => $data['prioridad'] ?? 'media',
                'monto_estimado' => $data['monto_estimado'] ?? null,
                'descripcion' => sanitize($data['descripcion'] ?? null)
            ]);
            
            // Asignar usuarios adicionales
            if (!empty($data['usuarios_asignados'])) {
                foreach ($data['usuarios_asignados'] as $usuarioId) {
                    $db->insert('casos_usuarios', [
                        'caso_id' => $casoId,
                        'usuario_id' => (int)$usuarioId
                    ]);
                }
            }
            
            $db->commit();
            
            // Log de actividad
            logActivity($userId, 'crear', 'casos', $casoId, "Caso {$numeroCaso} creado");
            
            // Notificar al responsable
            if ($data['responsable_id'] != $userId) {
                createNotification(
                    $data['responsable_id'],
                    'sistema',
                    'Nuevo caso asignado',
                    "Se te ha asignado el caso: {$data['nombre']}",
                    "/modules/casos/view.php?id={$casoId}"
                );
            }
            
            jsonSuccess(['id' => $casoId, 'numero_caso' => $numeroCaso], 'Caso creado exitosamente');
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    // PUT - Actualizar caso
    elseif ($method === 'PUT') {
        $auth->requirePermission('casos', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de caso requerido');
        }
        
        $db->beginTransaction();
        
        try {
            $updateData = [
                'nombre' => sanitize($data['nombre']),
                'caratula' => sanitize($data['caratula'] ?? null),
                'cliente_id' => (int)$data['cliente_id'],
                'responsable_id' => (int)$data['responsable_id'],
                'tipo_caso' => sanitize($data['tipo_caso'] ?? null),
                'materia' => sanitize($data['materia'] ?? null),
                'juzgado' => sanitize($data['juzgado'] ?? null),
                'expediente' => sanitize($data['expediente'] ?? null),
                'fecha_inicio' => $data['fecha_inicio'],
                'fecha_fin' => $data['fecha_fin'] ?? null,
                'estado' => $data['estado'],
                'prioridad' => $data['prioridad'],
                'monto_estimado' => $data['monto_estimado'] ?? null,
                'descripcion' => sanitize($data['descripcion'] ?? null)
            ];
            
            $db->update('casos', $updateData, 'id = ?', [$data['id']]);
            
            // Actualizar usuarios asignados
            $db->delete('casos_usuarios', 'caso_id = ?', [$data['id']]);
            
            if (!empty($data['usuarios_asignados'])) {
                foreach ($data['usuarios_asignados'] as $usuarioId) {
                    $db->insert('casos_usuarios', [
                        'caso_id' => $data['id'],
                        'usuario_id' => (int)$usuarioId
                    ]);
                }
            }
            
            $db->commit();
            
            // Log de actividad
            logActivity($userId, 'actualizar', 'casos', $data['id'], "Caso actualizado");
            
            jsonSuccess([], 'Caso actualizado exitosamente');
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    // DELETE - Eliminar caso
    elseif ($method === 'DELETE') {
        $auth->requirePermission('casos', 'eliminar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de caso requerido');
        }
        
        // Verificar si tiene tareas
        $tareas = $db->fetchOne("SELECT COUNT(*) as total FROM tareas WHERE caso_id = ?", [$data['id']]);
        
        if ($tareas['total'] > 0) {
            jsonError('No se puede eliminar un caso con tareas asociadas');
        }
        
        $db->delete('casos', 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'eliminar', 'casos', $data['id'], "Caso eliminado");
        
        jsonSuccess([], 'Caso eliminado exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de casos: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
