<?php
/**
 * LexiPro - Biblioteca Jurídica
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Biblioteca Jurídica - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .library-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .lib-card {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            transition: transform 0.2s;
            display: flex;
            flex-direction: column;
            position: relative;
        }
        
        .lib-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .lib-icon {
            height: 120px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 48px;
            background: #f3f4f6;
            color: #6b7280;
        }
        
        .lib-icon.pdf { color: #ef4444; background: #fee2e2; }
        .lib-icon.doc { color: #3b82f6; background: #dbeafe; }
        .lib-icon.link { color: #10b981; background: #d1fae5; }
        
        .lib-content {
            padding: 15px;
            flex: 1;
            display: flex;
            flex-direction: column;
        }
        
        .lib-category {
            font-size: 11px;
            text-transform: uppercase;
            font-weight: 700;
            color: var(--primary);
            margin-bottom: 5px;
        }
        
        .lib-title {
            font-weight: 600;
            margin-bottom: 5px;
            font-size: 15px;
            line-height: 1.4;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .lib-meta {
            font-size: 12px;
            color: #9ca3af;
            margin-top: auto;
            padding-top: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .lib-actions {
            position: absolute;
            top: 10px;
            right: 10px;
            opacity: 0;
            transition: opacity 0.2s;
        }
        
        .lib-card:hover .lib-actions {
            opacity: 1;
        }
        
        .action-btn {
            background: white;
            border: none;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #ef4444;
        }
        
        .filters-bar {
            background: white;
            padding: 15px;
            border-radius: 12px;
            margin-bottom: 20px;
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .search-box {
            flex: 1;
            min-width: 200px;
            position: relative;
        }
        
        .search-input {
            width: 100%;
            padding: 10px 10px 10px 35px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
        }
        
        .search-icon {
            position: absolute;
            left: 12px;
            top: 50%;
            transform: translateY(-50%);
            color: #9ca3af;
        }
    </style>
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    
    <div class="container">
        <div class="header-flex">
            <div>
                <h1>📚 Biblioteca Jurídica</h1>
                <p>Gestión de documentos, leyes y modelos</p>
            </div>
            <button class="btn btn-primary" onclick="openModal()">+ Nuevo Recurso</button>
        </div>
        
        <div class="filters-bar">
            <div class="search-box">
                <span class="search-icon">🔍</span>
                <input type="text" class="search-input" id="searchInput" placeholder="Buscar por título o etiqueta..." onkeyup="loadItems()">
            </div>
            
            <select id="filterType" class="form-control" style="width: auto;" onchange="loadItems()">
                <option value="">Todos los tipos</option>
                <option value="documento">Documentos</option>
                <option value="enlace">Enlaces</option>
            </select>
            
            <select id="filterCategory" class="form-control" style="width: auto;" onchange="loadItems()">
                <option value="">Todas las categorías</option>
                <!-- Se llena dinámicamente -->
            </select>
        </div>
        
        <div id="libraryGrid" class="libraryGrid library-grid">
            <!-- Items -->
        </div>
    </div>
    
    <!-- Modal Nuevo Item -->
    <div class="modal" id="itemModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Nuevo Recurso</h2>
                <button class="modal-close" onclick="closeModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="itemForm">
                    <input type="hidden" name="action" value="create">
                    
                    <div class="form-group">
                        <label>Tipo *</label>
                        <select name="tipo" id="itemType" onchange="toggleTypeFields()" required style="width: 100%; padding: 10px; border: 2px solid #e5e7eb; border-radius: 8px;">
                            <option value="documento">Documento (PDF, Word)</option>
                            <option value="enlace">Enlace Web</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Título *</label>
                        <input type="text" name="titulo" required placeholder="Ej: Código Civil Paraguayo">
                    </div>
                    
                    <div class="form-group">
                        <label>Categoría *</label>
                        <select name="categoria_id" id="categorySelect" required style="width: 100%; padding: 10px; border: 2px solid #e5e7eb; border-radius: 8px;">
                            <!-- Se llena dinámicamente -->
                        </select>
                    </div>
                    
                    <div class="form-group" id="fileField">
                        <label>Archivo *</label>
                        <input type="file" name="archivo" accept=".pdf,.doc,.docx,.txt,.rtf">
                    </div>
                    
                    <div class="form-group" id="urlField" style="display: none;">
                        <label>URL *</label>
                        <input type="url" name="url" placeholder="https://...">
                    </div>
                    
                    <div class="form-group">
                        <label>Etiquetas (separadas por coma)</label>
                        <input type="text" name="etiquetas" placeholder="civil, familia, modelo">
                    </div>
                    
                    <div class="form-group">
                        <label>Descripción</label>
                        <textarea name="descripcion" rows="3"></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%;">Guardar Recurso</button>
                </form>
            </div>
        </div>
    </div>

    <script src="../../assets/js/lexipro.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            loadCategories();
            loadItems();
        });
        
        async function loadCategories() {
            const response = await fetch('api.php?action=categorias');
            const categories = await response.json();
            
            if (categories.success) {
                const select = document.getElementById('categorySelect');
                const filter = document.getElementById('filterCategory');
                
                categories.data.forEach(cat => {
                    // Llenar select del modal
                    const opt = document.createElement('option');
                    opt.value = cat.id;
                    opt.textContent = cat.nombre;
                    select.appendChild(opt);
                    
                    // Llenar filtro
                    const optFilter = document.createElement('option');
                    optFilter.value = cat.id;
                    optFilter.textContent = cat.nombre;
                    filter.appendChild(optFilter);
                });
            }
        }
        
        async function loadItems() {
            const search = document.getElementById('searchInput').value;
            const type = document.getElementById('filterType').value;
            const category = document.getElementById('filterCategory').value;
            
            let url = `api.php?action=list`;
            if (search) url += `&search=${encodeURIComponent(search)}`;
            if (type) url += `&tipo=${type}`;
            if (category) url += `&categoria=${category}`;
            
            const response = await fetch(url);
            const result = await response.json();
            
            const grid = document.getElementById('libraryGrid');
            grid.innerHTML = '';
            
            if (result.success && result.data.length > 0) {
                result.data.forEach(item => {
                    const card = createCard(item);
                    grid.appendChild(card);
                });
            } else {
                grid.innerHTML = '<p style="grid-column: 1/-1; text-align: center; color: #9ca3af; padding: 40px;">No se encontraron recursos</p>';
            }
        }
        
        function createCard(item) {
            const div = document.createElement('div');
            div.className = 'lib-card';
            
            let iconClass = 'doc';
            let iconChar = '📄';
            let link = '#';
            let target = '';
            
            if (item.tipo === 'enlace') {
                iconClass = 'link';
                iconChar = '🔗';
                link = item.url;
                target = '_blank';
            } else {
                const ext = item.archivo_ruta.split('.').pop().toLowerCase();
                if (ext === 'pdf') {
                    iconClass = 'pdf';
                    iconChar = '📕';
                }
                link = '../../' + item.archivo_ruta;
                target = '_blank';
            }
            
            div.innerHTML = `
                <div class="lib-actions">
                    <button class="action-btn" onclick="deleteItem(${item.id})" title="Eliminar">🗑️</button>
                </div>
                <a href="${link}" target="${target}" style="text-decoration: none; color: inherit; display: flex; flex-direction: column; height: 100%;">
                    <div class="lib-icon ${iconClass}">
                        ${iconChar}
                    </div>
                    <div class="lib-content">
                        <div class="lib-category" style="color: ${item.categoria_color}">${item.categoria_nombre}</div>
                        <div class="lib-title" title="${item.titulo}">${item.titulo}</div>
                        <div class="lib-meta">
                            <span>${item.tipo === 'documento' ? 'Archivo' : 'Enlace'}</span>
                            <span>${new Date(item.fecha_creacion).toLocaleDateString()}</span>
                        </div>
                    </div>
                </a>
            `;
            
            return div;
        }
        
        function toggleTypeFields() {
            const type = document.getElementById('itemType').value;
            const fileField = document.getElementById('fileField');
            const urlField = document.getElementById('urlField');
            
            if (type === 'documento') {
                fileField.style.display = 'block';
                urlField.style.display = 'none';
                document.querySelector('input[name="archivo"]').required = true;
                document.querySelector('input[name="url"]').required = false;
            } else {
                fileField.style.display = 'none';
                urlField.style.display = 'block';
                document.querySelector('input[name="archivo"]').required = false;
                document.querySelector('input[name="url"]').required = true;
            }
        }
        
        function openModal() {
            document.getElementById('itemModal').classList.add('active');
        }
        
        function closeModal() {
            document.getElementById('itemModal').classList.remove('active');
            document.getElementById('itemForm').reset();
            toggleTypeFields();
        }
        
        document.getElementById('itemForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    closeModal();
                    loadItems();
                    alert('Recurso guardado correctamente');
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar');
            }
        });
        
        async function deleteItem(id) {
            if (!confirm('¿Estás seguro de eliminar este recurso?')) return;
            
            try {
                const formData = new FormData();
                formData.append('action', 'delete');
                formData.append('id', id);
                
                const response = await fetch('api.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    loadItems();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al eliminar');
            }
        }
    </script>
</body>
</html>
