<?php
/**
 * LexiPro - Gestor de Archivos
 */
session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$casoId = $_GET['caso_id'] ?? null;
if (!$casoId) {
    die("ID de caso requerido");
}

$db = Database::getInstance();
$caso = $db->fetchOne("SELECT * FROM casos WHERE id = ?", [$casoId]);
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Archivos - <?= e($caso['nombre']) ?></title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <style>
        .file-manager-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            background: var(--bg-card);
            padding: 15px;
            border-radius: 12px;
            border: 1px solid var(--border);
        }
        
        .breadcrumbs {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
        }
        
        .breadcrumb-item {
            cursor: pointer;
            color: var(--primary);
            text-decoration: none;
        }
        
        .breadcrumb-item:hover {
            text-decoration: underline;
        }
        
        .breadcrumb-separator {
            color: var(--text-muted);
        }
        
        .files-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
            gap: 15px;
        }
        
        .file-item {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 15px;
            text-align: center;
            cursor: pointer;
            transition: all 0.2s;
            position: relative;
        }
        
        .file-item:hover {
            background: rgba(59, 130, 246, 0.05);
            border-color: var(--primary);
            transform: translateY(-2px);
        }
        
        .file-icon {
            font-size: 40px;
            margin-bottom: 10px;
            display: block;
        }
        
        .file-name {
            font-size: 12px;
            color: var(--text-main);
            word-break: break-word;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            line-height: 1.3;
        }
        
        .file-meta {
            font-size: 10px;
            color: var(--text-muted);
            margin-top: 5px;
        }
        
        .context-menu {
            position: absolute;
            background: white;
            border: 1px solid var(--border);
            box-shadow: var(--shadow-hover);
            border-radius: 8px;
            padding: 5px 0;
            z-index: 1000;
            display: none;
            min-width: 150px;
        }
        
        .context-menu-item {
            padding: 8px 15px;
            font-size: 13px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .context-menu-item:hover {
            background: #f3f4f6;
        }
        
        .drop-zone {
            border: 2px dashed var(--border);
            border-radius: 12px;
            padding: 40px;
            text-align: center;
            color: var(--text-muted);
            margin-bottom: 20px;
            transition: all 0.2s;
            cursor: pointer;
        }
        
        .drop-zone.dragover {
            border-color: var(--primary);
            background: rgba(59, 130, 246, 0.05);
        }
    </style>
</head>
<body>
    <div style="padding: 20px;">
        <div class="file-manager-header">
            <div class="breadcrumbs" id="breadcrumbs">
                <span class="breadcrumb-item" onclick="loadContent(null)">📁 Raíz</span>
            </div>
            <div style="display: flex; gap: 10px;">
                <button class="btn btn-sm btn-secondary" onclick="createFolder()">➕ Nueva Carpeta</button>
                <button class="btn btn-sm btn-primary" onclick="document.getElementById('fileInput').click()">⬆️ Subir Archivo</button>
                <input type="file" id="fileInput" style="display: none" onchange="handleFileUpload(this.files)">
            </div>
        </div>
        
        <div class="drop-zone" id="dropZone">
            Arrastra archivos aquí o haz clic para subir
        </div>
        
        <div class="files-grid" id="filesGrid">
            <!-- Contenido dinámico -->
        </div>
    </div>

    <!-- Menú Contextual -->
    <div class="context-menu" id="contextMenu">
        <div class="context-menu-item" onclick="downloadFile()">⬇️ Descargar</div>
        <div class="context-menu-item" onclick="deleteItem()">🗑️ Eliminar</div>
    </div>

    <script src="../../assets/js/lexipro.js"></script>
    <script>
        const CASO_ID = <?= $casoId ?>;
        let currentFolderId = null;
        let selectedItem = null; // {type: 'file'|'folder', id: 1, name: ''}

        document.addEventListener('DOMContentLoaded', () => {
            loadContent(null);
            setupDragAndDrop();
        });

        // Cargar contenido
        async function loadContent(folderId) {
            currentFolderId = folderId;
            const url = `api.php?caso_id=${CASO_ID}${folderId ? '&parent_id=' + folderId : ''}`;
            
            try {
                const response = await fetch(url);
                const data = await response.json();
                
                if (data.success) {
                    renderGrid(data.data);
                    renderBreadcrumbs(data.data.breadcrumbs);
                }
            } catch (error) {
                console.error(error);
            }
        }

        function renderGrid(data) {
            const grid = document.getElementById('filesGrid');
            grid.innerHTML = '';
            
            // Carpetas
            data.carpetas.forEach(folder => {
                const div = document.createElement('div');
                div.className = 'file-item';
                div.innerHTML = `
                    <span class="file-icon">📁</span>
                    <div class="file-name">${folder.nombre}</div>
                `;
                div.onclick = () => loadContent(folder.id);
                div.oncontextmenu = (e) => showContextMenu(e, 'folder', folder);
                grid.appendChild(div);
            });
            
            // Archivos
            data.archivos.forEach(file => {
                const div = document.createElement('div');
                div.className = 'file-item';
                div.innerHTML = `
                    <span class="file-icon">${getFileIcon(file.tipo)}</span>
                    <div class="file-name">${file.nombre}</div>
                    <div class="file-meta">v${file.versiones} • ${formatSize(file.tamano)}</div>
                `;
                div.oncontextmenu = (e) => showContextMenu(e, 'file', file);
                grid.appendChild(div);
            });
            
            if (data.carpetas.length === 0 && data.archivos.length === 0) {
                grid.innerHTML = '<div style="grid-column: 1/-1; text-align: center; color: var(--text-muted); padding: 40px;">Carpeta vacía</div>';
            }
        }

        function renderBreadcrumbs(crumbs) {
            const container = document.getElementById('breadcrumbs');
            container.innerHTML = '<span class="breadcrumb-item" onclick="loadContent(null)">📁 Raíz</span>';
            
            crumbs.forEach(crumb => {
                container.innerHTML += `
                    <span class="breadcrumb-separator">›</span>
                    <span class="breadcrumb-item" onclick="loadContent(${crumb.id})">${crumb.nombre}</span>
                `;
            });
        }

        // Crear carpeta
        async function createFolder() {
            const name = prompt('Nombre de la carpeta:');
            if (!name) return;
            
            const formData = new FormData();
            formData.append('action', 'create_folder');
            formData.append('caso_id', CASO_ID);
            formData.append('nombre', name);
            if (currentFolderId) formData.append('parent_id', currentFolderId);
            
            const result = await LexiPro.fetch('api.php', {
                method: 'POST',
                body: formData
            });
            
            if (result.success) loadContent(currentFolderId);
        }

        // Subir archivo
        async function handleFileUpload(files) {
            if (!files.length) return;
            
            const file = files[0];
            const formData = new FormData();
            formData.append('action', 'upload_file');
            formData.append('caso_id', CASO_ID);
            formData.append('file', file);
            if (currentFolderId) formData.append('parent_id', currentFolderId);
            
            try {
                const result = await fetch('api.php', {
                    method: 'POST',
                    body: formData
                }).then(r => r.json());
                
                if (result.success) {
                    loadContent(currentFolderId);
                } else {
                    alert(result.error);
                }
            } catch (error) {
                console.error(error);
                alert('Error al subir archivo');
            }
        }

        // Drag and Drop
        function setupDragAndDrop() {
            const dropZone = document.getElementById('dropZone');
            
            dropZone.onclick = () => document.getElementById('fileInput').click();
            
            dropZone.ondragover = (e) => {
                e.preventDefault();
                dropZone.classList.add('dragover');
            };
            
            dropZone.ondragleave = () => {
                dropZone.classList.remove('dragover');
            };
            
            dropZone.ondrop = (e) => {
                e.preventDefault();
                dropZone.classList.remove('dragover');
                handleFileUpload(e.dataTransfer.files);
            };
        }

        // Menú Contextual
        function showContextMenu(e, type, item) {
            e.preventDefault();
            selectedItem = { type, ...item };
            
            const menu = document.getElementById('contextMenu');
            menu.style.display = 'block';
            menu.style.left = e.pageX + 'px';
            menu.style.top = e.pageY + 'px';
        }

        document.addEventListener('click', () => {
            document.getElementById('contextMenu').style.display = 'none';
        });

        async function deleteItem() {
            if (!confirm(`¿Eliminar ${selectedItem.nombre}?`)) return;
            
            const result = await LexiPro.fetch('api.php', {
                method: 'DELETE',
                body: JSON.stringify({
                    type: selectedItem.type,
                    id: selectedItem.id
                })
            });
            
            if (result.success) loadContent(currentFolderId);
        }

        // Utilidades
        function getFileIcon(mime) {
            if (mime.includes('pdf')) return '📕';
            if (mime.includes('image')) return '🖼️';
            if (mime.includes('word')) return '📝';
            if (mime.includes('excel') || mime.includes('sheet')) return '📊';
            return '📄';
        }

        function formatSize(bytes) {
            if (bytes === 0) return '0 B';
            const k = 1024;
            const sizes = ['B', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
    </script>
</body>
</html>
