<?php
/**
 * LexiPro - API de Gestor de Archivos
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Listar contenido
    if ($method === 'GET') {
        $auth->requirePermission('archivos', 'ver');
        
        $casoId = $_GET['caso_id'] ?? null;
        $parentId = $_GET['parent_id'] ?? null; // null = raíz
        
        if (!$casoId) {
            jsonError('ID de caso requerido');
        }
        
        // Obtener carpetas
        $sqlCarpetas = "SELECT * FROM carpetas WHERE caso_id = ?";
        $paramsCarpetas = [$casoId];
        
        if ($parentId) {
            $sqlCarpetas .= " AND parent_id = ?";
            $paramsCarpetas[] = $parentId;
        } else {
            $sqlCarpetas .= " AND parent_id IS NULL";
        }
        
        $carpetas = $db->fetchAll($sqlCarpetas, $paramsCarpetas);
        
        // Obtener archivos
        $sqlArchivos = "SELECT a.*, 
                        (SELECT COUNT(*) FROM archivo_versiones WHERE archivo_id = a.id) as versiones,
                        (SELECT created_at FROM archivo_versiones WHERE archivo_id = a.id ORDER BY version DESC LIMIT 1) as updated_at,
                        (SELECT tipo_mime FROM archivo_versiones WHERE archivo_id = a.id ORDER BY version DESC LIMIT 1) as tipo,
                        (SELECT tamano FROM archivo_versiones WHERE archivo_id = a.id ORDER BY version DESC LIMIT 1) as tamano
                        FROM archivos a 
                        WHERE a.caso_id = ?";
        $paramsArchivos = [$casoId];
        
        if ($parentId) {
            $sqlArchivos .= " AND a.carpeta_id = ?";
            $paramsArchivos[] = $parentId;
        } else {
            $sqlArchivos .= " AND a.carpeta_id IS NULL";
        }
        
        $archivos = $db->fetchAll($sqlArchivos, $paramsArchivos);
        
        // Obtener breadcrumbs
        $breadcrumbs = [];
        if ($parentId) {
            $current = $parentId;
            while ($current) {
                $folder = $db->fetchOne("SELECT id, nombre, parent_id FROM carpetas WHERE id = ?", [$current]);
                if ($folder) {
                    array_unshift($breadcrumbs, $folder);
                    $current = $folder['parent_id'];
                } else {
                    break;
                }
            }
        }
        
        jsonSuccess([
            'carpetas' => $carpetas,
            'archivos' => $archivos,
            'breadcrumbs' => $breadcrumbs
        ]);
    }
    
    // POST - Crear carpeta o subir archivo
    elseif ($method === 'POST') {
        $auth->requirePermission('archivos', 'crear');
        
        $action = $_POST['action'] ?? '';
        
        if ($action === 'create_folder') {
            $nombre = sanitize($_POST['nombre']);
            $casoId = (int)$_POST['caso_id'];
            $parentId = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : null;
            
            if (empty($nombre)) jsonError('Nombre requerido');
            
            $id = $db->insert('carpetas', [
                'caso_id' => $casoId,
                'nombre' => $nombre,
                'parent_id' => $parentId
            ]);
            
            jsonSuccess(['id' => $id], 'Carpeta creada');
        }
        
        elseif ($action === 'upload_file') {
            $casoId = (int)$_POST['caso_id'];
            $carpetaId = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : null;
            
            if (!isset($_FILES['file'])) jsonError('No se envió ningún archivo');
            
            $file = $_FILES['file'];
            $nombreOriginal = $file['name'];
            $tipoMime = $file['type'];
            $tamano = $file['size'];
            $tmpName = $file['tmp_name'];
            
            // Verificar si ya existe un archivo con ese nombre en la misma ubicación
            $existingFile = $db->fetchOne(
                "SELECT id FROM archivos WHERE caso_id = ? AND nombre = ? AND " . 
                ($carpetaId ? "carpeta_id = ?" : "carpeta_id IS NULL"),
                $carpetaId ? [$casoId, $nombreOriginal, $carpetaId] : [$casoId, $nombreOriginal]
            );
            
            $archivoId = null;
            $version = 1;
            
            $db->beginTransaction();
            
            try {
                if ($existingFile) {
                    // Nueva versión
                    $archivoId = $existingFile['id'];
                    $lastVersion = $db->fetchOne(
                        "SELECT MAX(version) as v FROM archivo_versiones WHERE archivo_id = ?", 
                        [$archivoId]
                    );
                    $version = ($lastVersion['v'] ?? 0) + 1;
                } else {
                    // Nuevo archivo
                    $archivoId = $db->insert('archivos', [
                        'caso_id' => $casoId,
                        'carpeta_id' => $carpetaId,
                        'nombre' => $nombreOriginal,
                        'descripcion' => ''
                    ]);
                }
                
                // Guardar archivo físico
                $uploadDir = '../../uploads/casos/' . $casoId . '/';
                if (!file_exists($uploadDir)) {
                    mkdir($uploadDir, 0777, true);
                }
                
                $fileName = $archivoId . '_v' . $version . '_' . time() . '.' . pathinfo($nombreOriginal, PATHINFO_EXTENSION);
                $targetPath = $uploadDir . $fileName;
                
                if (move_uploaded_file($tmpName, $targetPath)) {
                    $db->insert('archivo_versiones', [
                        'archivo_id' => $archivoId,
                        'ruta' => $fileName,
                        'nombre_original' => $nombreOriginal,
                        'tipo_mime' => $tipoMime,
                        'tamano' => $tamano,
                        'version' => $version,
                        'subido_por' => $userId
                    ]);
                    
                    $db->commit();
                    jsonSuccess(['id' => $archivoId], 'Archivo subido exitosamente');
                } else {
                    throw new Exception('Error al mover el archivo subido');
                }
                
            } catch (Exception $e) {
                $db->rollback();
                throw $e;
            }
        }
    }
    
    // DELETE - Eliminar
    elseif ($method === 'DELETE') {
        $auth->requirePermission('archivos', 'eliminar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        $type = $data['type'] ?? ''; // 'folder' o 'file'
        $id = $data['id'] ?? 0;
        
        if ($type === 'folder') {
            // Eliminar carpeta (y recursivamente su contenido por FK CASCADE)
            $db->delete('carpetas', 'id = ?', [$id]);
            jsonSuccess([], 'Carpeta eliminada');
        } elseif ($type === 'file') {
            // Eliminar archivo físico (opcional, por ahora solo DB)
            // Idealmente deberíamos borrar los archivos físicos también
            $versiones = $db->fetchAll("SELECT ruta, caso_id FROM archivo_versiones av JOIN archivos a ON av.archivo_id = a.id WHERE a.id = ?", [$id]);
            foreach ($versiones as $v) {
                $path = '../../uploads/casos/' . $v['caso_id'] . '/' . $v['ruta'];
                if (file_exists($path)) unlink($path);
            }
            
            $db->delete('archivos', 'id = ?', [$id]);
            jsonSuccess([], 'Archivo eliminado');
        }
    }

} catch (Exception $e) {
    error_log("Error en API archivos: " . $e->getMessage());
    jsonError($e->getMessage(), 500);
}
