<?php
/**
 * LexiPro - Sistema de Agenda
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$currentUser = $auth->getCurrentUser();

// Breadcrumbs
$breadcrumbs = [
    ['label' => 'Agenda']
];

// Obtener usuarios para filtros
$usuarios = $db->fetchAll("SELECT id, nombre, apellido FROM usuarios WHERE activo = 1 ORDER BY nombre");

// Obtener casos para el selector
$casos = $db->fetchAll("SELECT id, numero_caso, nombre FROM casos WHERE estado != 'archivado' ORDER BY created_at DESC LIMIT 100");

// Obtener clientes
$clientes = $db->fetchAll("SELECT id, nombre, apellido, razon_social, tipo_persona FROM clientes WHERE activo = 1 ORDER BY nombre LIMIT 100");
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Agenda - LexiPro</title>
    <link rel="stylesheet" href="../../assets/css/lexipro.css">
    <!-- FullCalendar CSS -->
    <link href="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.css" rel="stylesheet">
</head>
<body>
    <?php include '../../includes/header.php'; ?>
    <style>
        .calendar-container {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        .calendar-filters {
            display: flex;
            gap: 15px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        
        .filter-item {
            flex: 1;
            min-width: 200px;
        }
        
        .filter-item label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
            font-size: 13px;
            color: #374151;
        }
        
        .filter-item select {
            width: 100%;
            padding: 10px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
        }
        
        .legend {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
            margin-top: 20px;
            padding: 15px;
            background: #f9fafb;
            border-radius: 8px;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 13px;
        }
        
        .legend-color {
            width: 16px;
            height: 16px;
            border-radius: 4px;
        }
        
        /* FullCalendar customization */
        .fc {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .fc-event {
            cursor: pointer;
            border-radius: 4px;
            padding: 2px 4px;
        }
        
        .fc-daygrid-event {
            white-space: normal;
        }
        
        .fc-toolbar-title {
            font-size: 24px !important;
            font-weight: 600;
        }
        
        .fc-button {
            background: var(--primary) !important;
            border-color: var(--primary) !important;
            text-transform: capitalize !important;
        }
        
        .fc-button:hover {
            background: var(--primary-dark) !important;
        }
        
        .fc-button-active {
            background: var(--secondary) !important;
        }
        
        .event-details {
            padding: 15px;
        }
        
        .event-details-row {
            padding: 8px 0;
            border-bottom: 1px solid #f3f4f6;
        }
        
        .event-details-row:last-child {
            border-bottom: none;
        }
        
        .event-details-label {
            font-weight: 600;
            color: #6b7280;
            margin-bottom: 4px;
        }
        
        .event-details-value {
            color: #1f2937;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>📅 Agenda</h1>
                <p style="color: #6b7280; margin-top: 5px;">Calendario de eventos, audiencias y plazos</p>
            </div>
            <button class="btn btn-primary" onclick="openEventModal()">+ Nuevo Evento</button>
        </div>
        
        <div class="calendar-filters">
            <div class="filter-item">
                <label>Usuario</label>
                <select id="usuarioFilter" onchange="loadCalendar()">
                    <option value="<?= $currentUser['id'] ?>">Mis eventos</option>
                    <option value="todos">Todos los usuarios</option>
                    <optgroup label="Usuarios">
                        <?php foreach ($usuarios as $usuario): ?>
                            <option value="<?= $usuario['id'] ?>"><?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?></option>
                        <?php endforeach; ?>
                    </optgroup>
                </select>
            </div>
            
            <div class="filter-item">
                <label>Tipo de Evento</label>
                <select id="tipoFilter" onchange="loadCalendar()">
                    <option value="">Todos los tipos</option>
                    <option value="reunion">Reunión</option>
                    <option value="audiencia">Audiencia</option>
                    <option value="plazo">Plazo</option>
                    <option value="cita">Cita</option>
                    <option value="otro">Otro</option>
                </select>
            </div>
        </div>
        
        <div class="calendar-container">
            <div id="calendar"></div>
            
            <div class="legend">
                <div class="legend-item">
                    <div class="legend-color" style="background: #10b981;"></div>
                    <span>Reunión</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #ef4444;"></div>
                    <span>Audiencia</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #f59e0b;"></div>
                    <span>Plazo</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #8b5cf6;"></div>
                    <span>Cita</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #3b82f6;"></div>
                    <span>Otro</span>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Modal de Evento -->
    <div class="modal" id="eventModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="eventModalTitle">Nuevo Evento</h2>
                <button class="modal-close" onclick="closeEventModal()">×</button>
            </div>
            <div class="modal-body">
                <form id="eventForm">
                    <input type="hidden" name="id" id="eventId">
                    
                    <div class="form-group">
                        <label>Título *</label>
                        <input type="text" name="titulo" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Tipo *</label>
                        <select name="tipo" required>
                            <option value="reunion">Reunión</option>
                            <option value="audiencia">Audiencia</option>
                            <option value="plazo">Plazo</option>
                            <option value="cita">Cita</option>
                            <option value="otro">Otro</option>
                        </select>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Fecha y Hora Inicio *</label>
                            <input type="datetime-local" name="fecha_inicio" required>
                        </div>
                        <div class="form-group">
                            <label>Fecha y Hora Fin *</label>
                            <input type="datetime-local" name="fecha_fin" required>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Caso</label>
                            <select name="caso_id">
                                <option value="">Sin caso asociado</option>
                                <?php foreach ($casos as $caso): ?>
                                    <option value="<?= $caso['id'] ?>"><?= e($caso['numero_caso'] . ' - ' . $caso['nombre']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Cliente</label>
                            <select name="cliente_id">
                                <option value="">Sin cliente asociado</option>
                                <?php foreach ($clientes as $cliente): ?>
                                    <?php 
                                    $nombreCliente = $cliente['tipo_persona'] === 'juridica' 
                                        ? $cliente['razon_social'] 
                                        : $cliente['nombre'] . ' ' . ($cliente['apellido'] ?? '');
                                    ?>
                                    <option value="<?= $cliente['id'] ?>"><?= e($nombreCliente) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Asignado a *</label>
                            <select name="usuario_id" required>
                                <?php foreach ($usuarios as $usuario): ?>
                                    <option value="<?= $usuario['id'] ?>" <?= $usuario['id'] == $currentUser['id'] ? 'selected' : '' ?>>
                                        <?= e($usuario['nombre'] . ' ' . $usuario['apellido']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Recordatorio</label>
                            <select name="recordatorio">
                                <option value="0">Sin recordatorio</option>
                                <option value="15">15 minutos antes</option>
                                <option value="30" selected>30 minutos antes</option>
                                <option value="60">1 hora antes</option>
                                <option value="1440">1 día antes</option>
                                <option value="2880">2 días antes</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Ubicación</label>
                        <input type="text" name="ubicacion" placeholder="Ej: Juzgado de Primera Instancia">
                    </div>
                    
                    <div class="form-group">
                        <label>Descripción</label>
                        <textarea name="descripcion" rows="3" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>Notas</label>
                        <textarea name="notas" rows="2" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px;"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="completado" value="1">
                            Marcar como completado
                        </label>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                        Guardar Evento
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Modal de Detalles -->
    <div class="modal" id="detailsModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="detailsTitle">Detalles del Evento</h2>
                <button class="modal-close" onclick="closeDetailsModal()">×</button>
            </div>
            <div class="modal-body">
                <div class="event-details" id="eventDetails"></div>
                <div style="display: flex; gap: 10px; margin-top: 20px;">
                    <button class="btn btn-primary" onclick="editEventFromDetails()" style="flex: 1;">Editar</button>
                    <button class="btn btn-danger" onclick="deleteEventFromDetails()" style="flex: 1;">Eliminar</button>
                </div>
            </div>
        </div>
    </div>
    
    <script src="../../assets/js/lexipro.js"></script>
    <!-- FullCalendar JS -->
    <script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/locales/es.global.min.js"></script>
    
    <script>
        let calendar;
        let currentEvent = null;
        
        // Inicializar calendario
        document.addEventListener('DOMContentLoaded', function() {
            const calendarEl = document.getElementById('calendar');
            
            calendar = new FullCalendar.Calendar(calendarEl, {
                locale: 'es',
                initialView: 'dayGridMonth',
                headerToolbar: {
                    left: 'prev,next today',
                    center: 'title',
                    right: 'dayGridMonth,timeGridWeek,timeGridDay,listWeek'
                },
                buttonText: {
                    today: 'Hoy',
                    month: 'Mes',
                    week: 'Semana',
                    day: 'Día',
                    list: 'Lista'
                },
                height: 'auto',
                navLinks: true,
                editable: true,
                selectable: true,
                selectMirror: true,
                dayMaxEvents: true,
                weekends: true,
                
                // Cargar eventos
                events: function(info, successCallback, failureCallback) {
                    loadEvents(info.startStr, info.endStr, successCallback, failureCallback);
                },
                
                // Click en evento
                eventClick: function(info) {
                    showEventDetails(info.event);
                },
                
                // Seleccionar rango de fechas
                select: function(info) {
                    openEventModal(null, info.startStr, info.endStr);
                },
                
                // Arrastrar evento
                eventDrop: function(info) {
                    updateEventDates(info.event);
                },
                
                // Redimensionar evento
                eventResize: function(info) {
                    updateEventDates(info.event);
                }
            });
            
            calendar.render();
        });
        
        // Cargar eventos
        async function loadEvents(start, end, successCallback, failureCallback) {
            try {
                const usuario = document.getElementById('usuarioFilter').value;
                const tipo = document.getElementById('tipoFilter').value;
                
                const params = new URLSearchParams({
                    start: start,
                    end: end,
                    usuario: usuario
                });
                
                if (tipo) params.append('tipo', tipo);
                
                const response = await fetch(`api.php?${params}`);
                const data = await response.json();
                
                if (data.success) {
                    successCallback(data.eventos);
                } else {
                    failureCallback(data.error);
                }
            } catch (error) {
                console.error('Error al cargar eventos:', error);
                failureCallback(error);
            }
        }
        
        // Recargar calendario
        function loadCalendar() {
            calendar.refetchEvents();
        }
        
        // Mostrar detalles del evento
        function showEventDetails(event) {
            currentEvent = event;
            const props = event.extendedProps;
            
            document.getElementById('detailsTitle').textContent = event.title;
            
            let html = '';
            
            html += `<div class="event-details-row">
                <div class="event-details-label">Tipo</div>
                <div class="event-details-value"><span class="badge badge-info">${getTipoLabel(props.tipo)}</span></div>
            </div>`;
            
            html += `<div class="event-details-row">
                <div class="event-details-label">Fecha y Hora</div>
                <div class="event-details-value">${LexiPro.formatDateTime(event.start)} - ${LexiPro.formatDateTime(event.end)}</div>
            </div>`;
            
            if (props.caso_nombre) {
                html += `<div class="event-details-row">
                    <div class="event-details-label">Caso</div>
                    <div class="event-details-value">${props.numero_caso} - ${props.caso_nombre}</div>
                </div>`;
            }
            
            if (props.cliente_nombre) {
                html += `<div class="event-details-row">
                    <div class="event-details-label">Cliente</div>
                    <div class="event-details-value">${props.cliente_nombre} ${props.cliente_apellido || ''}</div>
                </div>`;
            }
            
            if (props.ubicacion) {
                html += `<div class="event-details-row">
                    <div class="event-details-label">Ubicación</div>
                    <div class="event-details-value">${props.ubicacion}</div>
                </div>`;
            }
            
            if (props.descripcion) {
                html += `<div class="event-details-row">
                    <div class="event-details-label">Descripción</div>
                    <div class="event-details-value">${props.descripcion}</div>
                </div>`;
            }
            
            html += `<div class="event-details-row">
                <div class="event-details-label">Estado</div>
                <div class="event-details-value">
                    <span class="badge ${props.completado ? 'badge-success' : 'badge-warning'}">
                        ${props.completado ? 'Completado' : 'Pendiente'}
                    </span>
                </div>
            </div>`;
            
            document.getElementById('eventDetails').innerHTML = html;
            document.getElementById('detailsModal').classList.add('active');
        }
        
        // Cerrar modal de detalles
        function closeDetailsModal() {
            document.getElementById('detailsModal').classList.remove('active');
            currentEvent = null;
        }
        
        // Editar desde detalles
        function editEventFromDetails() {
            if (currentEvent) {
                closeDetailsModal();
                openEventModal(currentEvent.id);
            }
        }
        
        // Eliminar desde detalles
        async function deleteEventFromDetails() {
            if (!currentEvent) return;
            
            if (!confirm('¿Estás seguro de eliminar este evento?')) return;
            
            try {
                const response = await fetch('api.php', {
                    method: 'DELETE',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({id: currentEvent.id})
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    closeDetailsModal();
                    calendar.refetchEvents();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al eliminar evento');
                console.error(error);
            }
        }
        
        // Abrir modal de evento
        function openEventModal(eventId = null, startDate = null, endDate = null) {
            const modal = document.getElementById('eventModal');
            const form = document.getElementById('eventForm');
            const title = document.getElementById('eventModalTitle');
            
            form.reset();
            
            if (eventId) {
                title.textContent = 'Editar Evento';
                // Cargar datos del evento
                loadEventData(eventId);
            } else {
                title.textContent = 'Nuevo Evento';
                
                // Si hay fechas seleccionadas
                if (startDate && endDate) {
                    form.elements['fecha_inicio'].value = startDate.slice(0, 16);
                    form.elements['fecha_fin'].value = endDate.slice(0, 16);
                }
            }
            
            modal.classList.add('active');
        }
        
        // Cargar datos del evento
        async function loadEventData(eventId) {
            // Aquí cargaríamos los datos completos del evento
            // Por ahora usamos los datos del currentEvent si está disponible
            if (currentEvent && currentEvent.id == eventId) {
                const form = document.getElementById('eventForm');
                form.elements['id'].value = currentEvent.id;
                form.elements['titulo'].value = currentEvent.title;
                form.elements['fecha_inicio'].value = currentEvent.start.toISOString().slice(0, 16);
                form.elements['fecha_fin'].value = currentEvent.end.toISOString().slice(0, 16);
                // Agregar más campos según sea necesario
            }
        }
        
        // Cerrar modal
        function closeEventModal() {
            document.getElementById('eventModal').classList.remove('active');
        }
        
        // Actualizar fechas de evento (drag & drop)
        async function updateEventDates(event) {
            try {
                const response = await fetch('api.php', {
                    method: 'PUT',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        id: event.id,
                        titulo: event.title,
                        tipo: event.extendedProps.tipo,
                        fecha_inicio: event.start.toISOString(),
                        fecha_fin: event.end.toISOString(),
                        usuario_id: <?= $currentUser['id'] ?>
                    })
                });
                
                const result = await response.json();
                
                if (!result.success) {
                    alert(result.error);
                    calendar.refetchEvents();
                }
            } catch (error) {
                console.error('Error al actualizar evento:', error);
                calendar.refetchEvents();
            }
        }
        
        // Guardar evento
        document.getElementById('eventForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {};
            
            for (let [key, value] of formData.entries()) {
                if (key === 'completado') {
                    data[key] = formData.get(key) ? 1 : 0;
                } else if (key === 'caso_id' || key === 'cliente_id') {
                    data[key] = value || null;
                } else {
                    data[key] = value;
                }
            }
            
            try {
                const method = data.id ? 'PUT' : 'POST';
                const response = await fetch('api.php', {
                    method,
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    closeEventModal();
                    calendar.refetchEvents();
                } else {
                    alert(result.error);
                }
            } catch (error) {
                alert('Error al guardar evento');
                console.error(error);
            }
        });
        
        // Helper
        function getTipoLabel(tipo) {
            const labels = {
                'reunion': 'Reunión',
                'audiencia': 'Audiencia',
                'plazo': 'Plazo',
                'cita': 'Cita',
                'otro': 'Otro'
            };
            return labels[tipo] || tipo;
        }
    </script>
</body>
</html>
