<?php
/**
 * LexiPro - API de Agenda
 */

session_start();
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';
require_once '../../config/database.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$userId = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];

try {
    // GET - Listar eventos
    if ($method === 'GET') {
        $auth->requirePermission('agenda', 'ver');
        
        $start = $_GET['start'] ?? null;
        $end = $_GET['end'] ?? null;
        $usuario = $_GET['usuario'] ?? $userId;
        $tipo = $_GET['tipo'] ?? '';
        
        $sql = "SELECT a.*, c.nombre as caso_nombre, c.numero_caso, 
                       cl.nombre as cliente_nombre, cl.apellido as cliente_apellido,
                       u.nombre as usuario_nombre, u.apellido as usuario_apellido
                FROM agenda a
                LEFT JOIN casos c ON a.caso_id = c.id
                LEFT JOIN clientes cl ON a.cliente_id = cl.id
                INNER JOIN usuarios u ON a.usuario_id = u.id
                WHERE 1=1";
        
        $params = [];
        
        // Filtrar por rango de fechas
        if ($start && $end) {
            $sql .= " AND a.fecha_inicio BETWEEN ? AND ?";
            $params[] = $start;
            $params[] = $end;
        }
        
        // Filtrar por usuario
        if ($usuario !== 'todos') {
            $sql .= " AND a.usuario_id = ?";
            $params[] = $usuario;
        }
        
        // Filtrar por tipo
        if ($tipo) {
            $sql .= " AND a.tipo = ?";
            $params[] = $tipo;
        }
        
        $sql .= " ORDER BY a.fecha_inicio ASC";
        
        $eventos = $db->fetchAll($sql, $params);
        
        // Formatear para FullCalendar
        $eventosCalendario = array_map(function($evento) {
            $color = '#3b82f6'; // Azul por defecto
            
            switch($evento['tipo']) {
                case 'audiencia':
                    $color = '#ef4444'; // Rojo
                    break;
                case 'plazo':
                    $color = '#f59e0b'; // Naranja
                    break;
                case 'reunion':
                    $color = '#10b981'; // Verde
                    break;
                case 'cita':
                    $color = '#8b5cf6'; // Púrpura
                    break;
            }
            
            return [
                'id' => $evento['id'],
                'title' => $evento['titulo'],
                'start' => $evento['fecha_inicio'],
                'end' => $evento['fecha_fin'],
                'backgroundColor' => $color,
                'borderColor' => $color,
                'extendedProps' => [
                    'tipo' => $evento['tipo'],
                    'descripcion' => $evento['descripcion'],
                    'caso_nombre' => $evento['caso_nombre'],
                    'numero_caso' => $evento['numero_caso'],
                    'cliente_nombre' => $evento['cliente_nombre'],
                    'cliente_apellido' => $evento['cliente_apellido'],
                    'ubicacion' => $evento['ubicacion'],
                    'completado' => $evento['completado']
                ]
            ];
        }, $eventos);
        
        jsonSuccess(['eventos' => $eventosCalendario]);
    }
    
    // POST - Crear evento
    elseif ($method === 'POST') {
        $auth->requirePermission('agenda', 'crear');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['titulo']) || empty($data['fecha_inicio']) || empty($data['fecha_fin'])) {
            jsonError('Título y fechas son obligatorios');
        }
        
        $eventoId = $db->insert('agenda', [
            'titulo' => sanitize($data['titulo']),
            'descripcion' => sanitize($data['descripcion'] ?? null),
            'tipo' => $data['tipo'] ?? 'otro',
            'caso_id' => $data['caso_id'] ?? null,
            'cliente_id' => $data['cliente_id'] ?? null,
            'usuario_id' => $data['usuario_id'] ?? $userId,
            'fecha_inicio' => $data['fecha_inicio'],
            'fecha_fin' => $data['fecha_fin'],
            'ubicacion' => sanitize($data['ubicacion'] ?? null),
            'recordatorio' => $data['recordatorio'] ?? 30,
            'notas' => sanitize($data['notas'] ?? null)
        ]);
        
        // Log de actividad
        logActivity($userId, 'crear', 'agenda', $eventoId, "Evento creado: {$data['titulo']}");
        
        // Crear notificación
        if ($data['usuario_id'] != $userId) {
            createNotification(
                $data['usuario_id'],
                'agenda',
                'Nuevo evento en agenda',
                "Se te ha asignado: {$data['titulo']}",
                "/modules/agenda/"
            );
        }
        
        jsonSuccess(['id' => $eventoId], 'Evento creado exitosamente');
    }
    
    // PUT - Actualizar evento
    elseif ($method === 'PUT') {
        $auth->requirePermission('agenda', 'editar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de evento requerido');
        }
        
        $updateData = [
            'titulo' => sanitize($data['titulo']),
            'descripcion' => sanitize($data['descripcion'] ?? null),
            'tipo' => $data['tipo'],
            'caso_id' => $data['caso_id'] ?? null,
            'cliente_id' => $data['cliente_id'] ?? null,
            'usuario_id' => $data['usuario_id'],
            'fecha_inicio' => $data['fecha_inicio'],
            'fecha_fin' => $data['fecha_fin'],
            'ubicacion' => sanitize($data['ubicacion'] ?? null),
            'recordatorio' => $data['recordatorio'] ?? 30,
            'completado' => (int)($data['completado'] ?? 0),
            'notas' => sanitize($data['notas'] ?? null)
        ];
        
        $db->update('agenda', $updateData, 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'actualizar', 'agenda', $data['id'], "Evento actualizado");
        
        jsonSuccess([], 'Evento actualizado exitosamente');
    }
    
    // DELETE - Eliminar evento
    elseif ($method === 'DELETE') {
        $auth->requirePermission('agenda', 'eliminar');
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['id'])) {
            jsonError('ID de evento requerido');
        }
        
        $db->delete('agenda', 'id = ?', [$data['id']]);
        
        // Log de actividad
        logActivity($userId, 'eliminar', 'agenda', $data['id'], "Evento eliminado");
        
        jsonSuccess([], 'Evento eliminado exitosamente');
    }
    
} catch (Exception $e) {
    error_log("Error en API de agenda: " . $e->getMessage());
    jsonError('Error al procesar la solicitud: ' . $e->getMessage(), 500);
}
