-- ============================================
-- LEXIPRO - Sistema de Gestión Jurídica
-- Esquema de Base de Datos MySQL
-- ============================================

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET time_zone = "+00:00";

-- ============================================
-- TABLA: configuracion
-- Datos de la empresa/estudio jurídico
-- ============================================
CREATE TABLE IF NOT EXISTS `configuracion` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nombre_empresa` varchar(255) NOT NULL,
  `razon_social` varchar(255) DEFAULT NULL,
  `ruc` varchar(50) DEFAULT NULL,
  `representante_legal` varchar(255) DEFAULT NULL,
  `direccion` text,
  `telefono` varchar(50) DEFAULT NULL,
  `email` varchar(100) DEFAULT NULL,
  `logo` varchar(255) DEFAULT NULL,
  `timbrado` varchar(50) DEFAULT NULL,
  `fecha_inicio_timbrado` date DEFAULT NULL,
  `fecha_fin_timbrado` date DEFAULT NULL,
  `punto_expedicion` varchar(10) DEFAULT '001',
  `establecimiento` varchar(10) DEFAULT '001',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: roles
-- Roles del sistema
-- ============================================
CREATE TABLE IF NOT EXISTS `roles` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nombre` varchar(100) NOT NULL,
  `descripcion` text,
  `nivel` int(11) NOT NULL DEFAULT '0',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `nombre` (`nombre`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insertar roles predefinidos
INSERT INTO `roles` (`nombre`, `descripcion`, `nivel`) VALUES
('Administrador General', 'Acceso total al sistema', 100),
('Abogado Senior', 'Abogado con experiencia y casos importantes', 80),
('Abogado Junior', 'Abogado en formación', 60),
('Notario', 'Notario del estudio', 70),
('Actuario', 'Actuario judicial', 50),
('Secretario', 'Secretario administrativo', 40),
('Oficial', 'Oficial del estudio', 30),
('Gestor', 'Gestor de casos', 30);

-- ============================================
-- TABLA: usuarios
-- Usuarios del sistema
-- ============================================
CREATE TABLE IF NOT EXISTS `usuarios` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nombre` varchar(100) NOT NULL,
  `apellido` varchar(100) NOT NULL,
  `usuario` varchar(50) NOT NULL,
  `password` varchar(255) NOT NULL,
  `email` varchar(100) NOT NULL,
  `telefono` varchar(50) DEFAULT NULL,
  `fecha_nacimiento` date DEFAULT NULL,
  `foto` varchar(255) DEFAULT NULL,
  `cargo` varchar(100) DEFAULT NULL,
  `rol_id` int(11) NOT NULL,
  `tipo_pago` enum('hora','mensual') NOT NULL DEFAULT 'mensual',
  `tarifa_hora` decimal(10,2) DEFAULT NULL,
  `salario_mensual` decimal(10,2) DEFAULT NULL,
  `es_proveedor_externo` tinyint(1) NOT NULL DEFAULT '0',
  `activo` tinyint(1) NOT NULL DEFAULT '1',
  `ultimo_acceso` timestamp NULL DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `usuario` (`usuario`),
  UNIQUE KEY `email` (`email`),
  KEY `rol_id` (`rol_id`),
  CONSTRAINT `fk_usuarios_rol` FOREIGN KEY (`rol_id`) REFERENCES `roles` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: permisos
-- Permisos granulares por módulo
-- ============================================
CREATE TABLE IF NOT EXISTS `permisos` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `rol_id` int(11) NOT NULL,
  `modulo` varchar(50) NOT NULL,
  `ver` tinyint(1) NOT NULL DEFAULT '0',
  `crear` tinyint(1) NOT NULL DEFAULT '0',
  `editar` tinyint(1) NOT NULL DEFAULT '0',
  `eliminar` tinyint(1) NOT NULL DEFAULT '0',
  PRIMARY KEY (`id`),
  UNIQUE KEY `rol_modulo` (`rol_id`, `modulo`),
  CONSTRAINT `fk_permisos_rol` FOREIGN KEY (`rol_id`) REFERENCES `roles` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: clientes
-- Clientes del estudio
-- ============================================
CREATE TABLE IF NOT EXISTS `clientes` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `tipo_persona` enum('fisica','juridica') NOT NULL DEFAULT 'fisica',
  `nombre` varchar(100) NOT NULL,
  `apellido` varchar(100) DEFAULT NULL,
  `razon_social` varchar(255) DEFAULT NULL,
  `ruc` varchar(50) DEFAULT NULL,
  `cedula` varchar(50) DEFAULT NULL,
  `email` varchar(100) DEFAULT NULL,
  `telefono` varchar(50) DEFAULT NULL,
  `celular` varchar(50) DEFAULT NULL,
  `direccion` text,
  `ciudad` varchar(100) DEFAULT NULL,
  `pais` varchar(100) DEFAULT 'Paraguay',
  `fecha_nacimiento` date DEFAULT NULL,
  `profesion` varchar(100) DEFAULT NULL,
  `notas` text,
  `activo` tinyint(1) NOT NULL DEFAULT '1',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `ruc` (`ruc`),
  KEY `cedula` (`cedula`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: casos
-- Casos jurídicos como proyectos
-- ============================================
CREATE TABLE IF NOT EXISTS `casos` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `numero_caso` varchar(50) NOT NULL,
  `nombre` varchar(255) NOT NULL,
  `caratula` text,
  `cliente_id` int(11) NOT NULL,
  `responsable_id` int(11) NOT NULL,
  `tipo_caso` varchar(100) DEFAULT NULL,
  `materia` varchar(100) DEFAULT NULL,
  `juzgado` varchar(255) DEFAULT NULL,
  `expediente` varchar(100) DEFAULT NULL,
  `fecha_inicio` date NOT NULL,
  `fecha_fin` date DEFAULT NULL,
  `estado` enum('abierto','en_proceso','suspendido','cerrado','archivado') NOT NULL DEFAULT 'abierto',
  `prioridad` enum('baja','media','alta','urgente') NOT NULL DEFAULT 'media',
  `monto_estimado` decimal(12,2) DEFAULT NULL,
  `descripcion` text,
  `notas` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `numero_caso` (`numero_caso`),
  KEY `cliente_id` (`cliente_id`),
  KEY `responsable_id` (`responsable_id`),
  KEY `estado` (`estado`),
  CONSTRAINT `fk_casos_cliente` FOREIGN KEY (`cliente_id`) REFERENCES `clientes` (`id`),
  CONSTRAINT `fk_casos_responsable` FOREIGN KEY (`responsable_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: casos_usuarios
-- Usuarios asignados a casos
-- ============================================
CREATE TABLE IF NOT EXISTS `casos_usuarios` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `caso_id` int(11) NOT NULL,
  `usuario_id` int(11) NOT NULL,
  `rol_en_caso` varchar(100) DEFAULT NULL,
  `fecha_asignacion` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `caso_usuario` (`caso_id`, `usuario_id`),
  KEY `usuario_id` (`usuario_id`),
  CONSTRAINT `fk_casos_usuarios_caso` FOREIGN KEY (`caso_id`) REFERENCES `casos` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_casos_usuarios_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: tareas
-- Tareas de los casos
-- ============================================
CREATE TABLE IF NOT EXISTS `tareas` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `caso_id` int(11) NOT NULL,
  `titulo` varchar(255) NOT NULL,
  `descripcion` text,
  `responsable_id` int(11) NOT NULL,
  `fecha_inicio` date DEFAULT NULL,
  `fecha_limite` date DEFAULT NULL,
  `fecha_completada` datetime DEFAULT NULL,
  `horas_estimadas` decimal(5,2) DEFAULT NULL,
  `horas_trabajadas` decimal(5,2) DEFAULT '0.00',
  `estado` enum('pendiente','en_proceso','completada','cancelada') NOT NULL DEFAULT 'pendiente',
  `prioridad` enum('baja','media','alta','urgente') NOT NULL DEFAULT 'media',
  `es_plazo_juridico` tinyint(1) NOT NULL DEFAULT '0',
  `notas` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `caso_id` (`caso_id`),
  KEY `responsable_id` (`responsable_id`),
  KEY `estado` (`estado`),
  KEY `es_plazo_juridico` (`es_plazo_juridico`),
  CONSTRAINT `fk_tareas_caso` FOREIGN KEY (`caso_id`) REFERENCES `casos` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_tareas_responsable` FOREIGN KEY (`responsable_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: agenda
-- Eventos y citas
-- ============================================
CREATE TABLE IF NOT EXISTS `agenda` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `titulo` varchar(255) NOT NULL,
  `descripcion` text,
  `tipo` enum('reunion','audiencia','plazo','cita','otro') NOT NULL DEFAULT 'reunion',
  `caso_id` int(11) DEFAULT NULL,
  `cliente_id` int(11) DEFAULT NULL,
  `usuario_id` int(11) NOT NULL,
  `fecha_inicio` datetime NOT NULL,
  `fecha_fin` datetime NOT NULL,
  `ubicacion` varchar(255) DEFAULT NULL,
  `recordatorio` int(11) DEFAULT '30' COMMENT 'Minutos antes',
  `completado` tinyint(1) NOT NULL DEFAULT '0',
  `notas` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `caso_id` (`caso_id`),
  KEY `cliente_id` (`cliente_id`),
  KEY `usuario_id` (`usuario_id`),
  KEY `fecha_inicio` (`fecha_inicio`),
  CONSTRAINT `fk_agenda_caso` FOREIGN KEY (`caso_id`) REFERENCES `casos` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_agenda_cliente` FOREIGN KEY (`cliente_id`) REFERENCES `clientes` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_agenda_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: archivos
-- Archivos adjuntos del sistema
-- ============================================
CREATE TABLE IF NOT EXISTS `archivos` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nombre_original` varchar(255) NOT NULL,
  `nombre_archivo` varchar(255) NOT NULL,
  `ruta` varchar(500) NOT NULL,
  `tipo_mime` varchar(100) DEFAULT NULL,
  `tamano` bigint(20) DEFAULT NULL,
  `modulo` varchar(50) NOT NULL COMMENT 'caso, tarea, cliente, etc',
  `modulo_id` int(11) NOT NULL,
  `usuario_id` int(11) NOT NULL,
  `descripcion` text,
  `version` int(11) NOT NULL DEFAULT '1',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `modulo_id` (`modulo`, `modulo_id`),
  KEY `usuario_id` (`usuario_id`),
  CONSTRAINT `fk_archivos_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: proveedores
-- Proveedores del estudio
-- ============================================
CREATE TABLE IF NOT EXISTS `proveedores` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `razon_social` varchar(255) NOT NULL,
  `ruc` varchar(50) NOT NULL,
  `email` varchar(100) DEFAULT NULL,
  `telefono` varchar(50) DEFAULT NULL,
  `direccion` text,
  `ciudad` varchar(100) DEFAULT NULL,
  `contacto_nombre` varchar(255) DEFAULT NULL,
  `contacto_telefono` varchar(50) DEFAULT NULL,
  `tipo_servicio` varchar(100) DEFAULT NULL,
  `notas` text,
  `activo` tinyint(1) NOT NULL DEFAULT '1',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `ruc` (`ruc`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: facturas_ventas
-- Facturas emitidas (RG90 Paraguay)
-- ============================================
CREATE TABLE IF NOT EXISTS `facturas_ventas` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `numero_factura` varchar(50) NOT NULL,
  `timbrado` varchar(50) NOT NULL,
  `fecha_emision` date NOT NULL,
  `cliente_id` int(11) NOT NULL,
  `caso_id` int(11) DEFAULT NULL,
  `condicion_venta` enum('contado','credito') NOT NULL DEFAULT 'contado',
  `subtotal_5` decimal(12,2) NOT NULL DEFAULT '0.00',
  `subtotal_10` decimal(12,2) NOT NULL DEFAULT '0.00',
  `subtotal_exento` decimal(12,2) NOT NULL DEFAULT '0.00',
  `iva_5` decimal(12,2) NOT NULL DEFAULT '0.00',
  `iva_10` decimal(12,2) NOT NULL DEFAULT '0.00',
  `total` decimal(12,2) NOT NULL,
  `saldo` decimal(12,2) NOT NULL,
  `estado` enum('pendiente','pagada','parcial','anulada') NOT NULL DEFAULT 'pendiente',
  `notas` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `numero_factura` (`numero_factura`),
  KEY `cliente_id` (`cliente_id`),
  KEY `caso_id` (`caso_id`),
  KEY `fecha_emision` (`fecha_emision`),
  CONSTRAINT `fk_facturas_ventas_cliente` FOREIGN KEY (`cliente_id`) REFERENCES `clientes` (`id`),
  CONSTRAINT `fk_facturas_ventas_caso` FOREIGN KEY (`caso_id`) REFERENCES `casos` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: facturas_ventas_detalle
-- Detalle de facturas de venta
-- ============================================
CREATE TABLE IF NOT EXISTS `facturas_ventas_detalle` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `factura_id` int(11) NOT NULL,
  `descripcion` varchar(500) NOT NULL,
  `cantidad` decimal(10,2) NOT NULL DEFAULT '1.00',
  `precio_unitario` decimal(12,2) NOT NULL,
  `tipo_iva` enum('exento','5','10') NOT NULL DEFAULT '10',
  `subtotal` decimal(12,2) NOT NULL,
  PRIMARY KEY (`id`),
  KEY `factura_id` (`factura_id`),
  CONSTRAINT `fk_facturas_ventas_detalle` FOREIGN KEY (`factura_id`) REFERENCES `facturas_ventas` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: cobros
-- Cobros de facturas
-- ============================================
CREATE TABLE IF NOT EXISTS `cobros` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `factura_id` int(11) NOT NULL,
  `fecha_cobro` date NOT NULL,
  `monto` decimal(12,2) NOT NULL,
  `forma_pago` enum('efectivo','transferencia','cheque','tarjeta','otro') NOT NULL,
  `referencia` varchar(100) DEFAULT NULL,
  `banco` varchar(100) DEFAULT NULL,
  `numero_cheque` varchar(50) DEFAULT NULL,
  `notas` text,
  `usuario_id` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `factura_id` (`factura_id`),
  KEY `usuario_id` (`usuario_id`),
  KEY `fecha_cobro` (`fecha_cobro`),
  CONSTRAINT `fk_cobros_factura` FOREIGN KEY (`factura_id`) REFERENCES `facturas_ventas` (`id`),
  CONSTRAINT `fk_cobros_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: facturas_compras
-- Facturas de proveedores (RG90 Paraguay)
-- ============================================
CREATE TABLE IF NOT EXISTS `facturas_compras` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `numero_factura` varchar(50) NOT NULL,
  `timbrado` varchar(50) NOT NULL,
  `fecha_emision` date NOT NULL,
  `proveedor_id` int(11) NOT NULL,
  `condicion_compra` enum('contado','credito') NOT NULL DEFAULT 'contado',
  `subtotal_5` decimal(12,2) NOT NULL DEFAULT '0.00',
  `subtotal_10` decimal(12,2) NOT NULL DEFAULT '0.00',
  `subtotal_exento` decimal(12,2) NOT NULL DEFAULT '0.00',
  `iva_5` decimal(12,2) NOT NULL DEFAULT '0.00',
  `iva_10` decimal(12,2) NOT NULL DEFAULT '0.00',
  `total` decimal(12,2) NOT NULL,
  `saldo` decimal(12,2) NOT NULL,
  `estado` enum('pendiente','pagada','parcial','anulada') NOT NULL DEFAULT 'pendiente',
  `notas` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `proveedor_id` (`proveedor_id`),
  KEY `fecha_emision` (`fecha_emision`),
  CONSTRAINT `fk_facturas_compras_proveedor` FOREIGN KEY (`proveedor_id`) REFERENCES `proveedores` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: facturas_compras_detalle
-- Detalle de facturas de compra
-- ============================================
CREATE TABLE IF NOT EXISTS `facturas_compras_detalle` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `factura_id` int(11) NOT NULL,
  `descripcion` varchar(500) NOT NULL,
  `cantidad` decimal(10,2) NOT NULL DEFAULT '1.00',
  `precio_unitario` decimal(12,2) NOT NULL,
  `tipo_iva` enum('exento','5','10') NOT NULL DEFAULT '10',
  `subtotal` decimal(12,2) NOT NULL,
  PRIMARY KEY (`id`),
  KEY `factura_id` (`factura_id`),
  CONSTRAINT `fk_facturas_compras_detalle` FOREIGN KEY (`factura_id`) REFERENCES `facturas_compras` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: pagos
-- Pagos a proveedores
-- ============================================
CREATE TABLE IF NOT EXISTS `pagos` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `factura_id` int(11) NOT NULL,
  `fecha_pago` date NOT NULL,
  `monto` decimal(12,2) NOT NULL,
  `forma_pago` enum('efectivo','transferencia','cheque','tarjeta','otro') NOT NULL,
  `referencia` varchar(100) DEFAULT NULL,
  `banco` varchar(100) DEFAULT NULL,
  `numero_cheque` varchar(50) DEFAULT NULL,
  `notas` text,
  `usuario_id` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `factura_id` (`factura_id`),
  KEY `usuario_id` (`usuario_id`),
  KEY `fecha_pago` (`fecha_pago`),
  CONSTRAINT `fk_pagos_factura` FOREIGN KEY (`factura_id`) REFERENCES `facturas_compras` (`id`),
  CONSTRAINT `fk_pagos_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: plan_cuentas
-- Plan de cuentas contable
-- ============================================
CREATE TABLE IF NOT EXISTS `plan_cuentas` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `codigo` varchar(20) NOT NULL,
  `nombre` varchar(255) NOT NULL,
  `tipo` enum('activo','pasivo','patrimonio','ingreso','egreso') NOT NULL,
  `padre_id` int(11) DEFAULT NULL,
  `nivel` int(11) NOT NULL DEFAULT '1',
  `acepta_movimiento` tinyint(1) NOT NULL DEFAULT '1',
  `saldo_inicial` decimal(12,2) NOT NULL DEFAULT '0.00',
  `activo` tinyint(1) NOT NULL DEFAULT '1',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `codigo` (`codigo`),
  KEY `padre_id` (`padre_id`),
  KEY `tipo` (`tipo`),
  CONSTRAINT `fk_plan_cuentas_padre` FOREIGN KEY (`padre_id`) REFERENCES `plan_cuentas` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: asientos_contables
-- Libro diario
-- ============================================
CREATE TABLE IF NOT EXISTS `asientos_contables` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `numero_asiento` int(11) NOT NULL,
  `fecha` date NOT NULL,
  `descripcion` text NOT NULL,
  `tipo` enum('manual','automatico') NOT NULL DEFAULT 'manual',
  `origen` varchar(50) DEFAULT NULL COMMENT 'factura_venta, factura_compra, etc',
  `origen_id` int(11) DEFAULT NULL,
  `usuario_id` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `numero_asiento` (`numero_asiento`),
  KEY `fecha` (`fecha`),
  KEY `usuario_id` (`usuario_id`),
  CONSTRAINT `fk_asientos_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: asientos_detalle
-- Detalle de asientos (debe/haber)
-- ============================================
CREATE TABLE IF NOT EXISTS `asientos_detalle` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `asiento_id` int(11) NOT NULL,
  `cuenta_id` int(11) NOT NULL,
  `debe` decimal(12,2) NOT NULL DEFAULT '0.00',
  `haber` decimal(12,2) NOT NULL DEFAULT '0.00',
  `descripcion` varchar(500) DEFAULT NULL,
  PRIMARY KEY (`id`),
  KEY `asiento_id` (`asiento_id`),
  KEY `cuenta_id` (`cuenta_id`),
  CONSTRAINT `fk_asientos_detalle_asiento` FOREIGN KEY (`asiento_id`) REFERENCES `asientos_contables` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_asientos_detalle_cuenta` FOREIGN KEY (`cuenta_id`) REFERENCES `plan_cuentas` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: chat_interno
-- Mensajes del chat interno
-- ============================================
CREATE TABLE IF NOT EXISTS `chat_interno` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `tipo` enum('individual','grupo') NOT NULL DEFAULT 'individual',
  `grupo_id` int(11) DEFAULT NULL,
  `remitente_id` int(11) NOT NULL,
  `destinatario_id` int(11) DEFAULT NULL,
  `mensaje` text NOT NULL,
  `archivo_id` int(11) DEFAULT NULL,
  `leido` tinyint(1) NOT NULL DEFAULT '0',
  `fecha_leido` datetime DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `remitente_id` (`remitente_id`),
  KEY `destinatario_id` (`destinatario_id`),
  KEY `grupo_id` (`grupo_id`),
  KEY `created_at` (`created_at`),
  CONSTRAINT `fk_chat_interno_remitente` FOREIGN KEY (`remitente_id`) REFERENCES `usuarios` (`id`),
  CONSTRAINT `fk_chat_interno_destinatario` FOREIGN KEY (`destinatario_id`) REFERENCES `usuarios` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: chat_grupos
-- Grupos de chat interno
-- ============================================
CREATE TABLE IF NOT EXISTS `chat_grupos` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nombre` varchar(255) NOT NULL,
  `descripcion` text,
  `creador_id` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `creador_id` (`creador_id`),
  CONSTRAINT `fk_chat_grupos_creador` FOREIGN KEY (`creador_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: chat_grupos_miembros
-- Miembros de grupos de chat
-- ============================================
CREATE TABLE IF NOT EXISTS `chat_grupos_miembros` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `grupo_id` int(11) NOT NULL,
  `usuario_id` int(11) NOT NULL,
  `fecha_ingreso` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `grupo_usuario` (`grupo_id`, `usuario_id`),
  KEY `usuario_id` (`usuario_id`),
  CONSTRAINT `fk_chat_grupos_miembros_grupo` FOREIGN KEY (`grupo_id`) REFERENCES `chat_grupos` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_chat_grupos_miembros_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: chat_web
-- Chat público del sitio web
-- ============================================
CREATE TABLE IF NOT EXISTS `chat_web` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `sesion_id` varchar(100) NOT NULL,
  `nombre` varchar(100) NOT NULL,
  `apellido` varchar(100) NOT NULL,
  `email` varchar(100) NOT NULL,
  `mensaje` text NOT NULL,
  `es_cliente` tinyint(1) NOT NULL DEFAULT '0' COMMENT '1=mensaje del visitante, 0=respuesta del estudio',
  `usuario_id` int(11) DEFAULT NULL COMMENT 'Usuario que responde',
  `leido` tinyint(1) NOT NULL DEFAULT '0',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `sesion_id` (`sesion_id`),
  KEY `usuario_id` (`usuario_id`),
  KEY `created_at` (`created_at`),
  CONSTRAINT `fk_chat_web_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: notificaciones
-- Sistema de notificaciones
-- ============================================
CREATE TABLE IF NOT EXISTS `notificaciones` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `usuario_id` int(11) NOT NULL,
  `tipo` enum('agenda','chat','tarea','plazo','sistema','otro') NOT NULL,
  `titulo` varchar(255) NOT NULL,
  `mensaje` text NOT NULL,
  `url` varchar(500) DEFAULT NULL,
  `icono` varchar(50) DEFAULT NULL,
  `leido` tinyint(1) NOT NULL DEFAULT '0',
  `fecha_leido` datetime DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `usuario_id` (`usuario_id`),
  KEY `leido` (`leido`),
  KEY `created_at` (`created_at`),
  CONSTRAINT `fk_notificaciones_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: biblioteca
-- Biblioteca jurídica
-- ============================================
CREATE TABLE IF NOT EXISTS `biblioteca` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `titulo` varchar(255) NOT NULL,
  `tipo` enum('libro','articulo','expediente','jurisprudencia','legislacion','otro') NOT NULL,
  `autor` varchar(255) DEFAULT NULL,
  `editorial` varchar(255) DEFAULT NULL,
  `año` int(11) DEFAULT NULL,
  `isbn` varchar(50) DEFAULT NULL,
  `descripcion` text,
  `es_archivo` tinyint(1) NOT NULL DEFAULT '1',
  `archivo_id` int(11) DEFAULT NULL,
  `url_externa` varchar(500) DEFAULT NULL,
  `categoria` varchar(100) DEFAULT NULL,
  `etiquetas` text,
  `usuario_id` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `tipo` (`tipo`),
  KEY `categoria` (`categoria`),
  KEY `usuario_id` (`usuario_id`),
  CONSTRAINT `fk_biblioteca_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: ia_consultas
-- Historial de consultas a IA
-- ============================================
CREATE TABLE IF NOT EXISTS `ia_consultas` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `usuario_id` int(11) NOT NULL,
  `tipo` enum('analisis','generacion','reporte') NOT NULL,
  `prompt` text NOT NULL,
  `respuesta` longtext,
  `archivo_id` int(11) DEFAULT NULL,
  `tokens_usados` int(11) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `usuario_id` (`usuario_id`),
  KEY `tipo` (`tipo`),
  CONSTRAINT `fk_ia_consultas_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABLA: configuracion_ia
-- Configuración de IA
-- ============================================
CREATE TABLE IF NOT EXISTS `configuracion_ia` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `api_key` varchar(255) DEFAULT NULL,
  `modelo` varchar(100) DEFAULT 'gemini-pro',
  `activo` tinyint(1) NOT NULL DEFAULT '0',
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insertar configuración IA por defecto
INSERT INTO `configuracion_ia` (`api_key`, `modelo`, `activo`) VALUES (NULL, 'gemini-pro', 0);

-- ============================================
-- TABLA: contactos_web
-- Formulario de contacto del sitio web
-- ============================================
CREATE TABLE IF NOT EXISTS `contactos_web` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nombre` varchar(100) NOT NULL,
  `email` varchar(100) NOT NULL,
  `telefono` varchar(50) DEFAULT NULL,
  `asunto` varchar(255) DEFAULT NULL,
  `mensaje` text NOT NULL,
  `ip` varchar(50) DEFAULT NULL,
  `atendido` tinyint(1) NOT NULL DEFAULT '0',
  `usuario_atencion_id` int(11) DEFAULT NULL,
  `fecha_atencion` datetime DEFAULT NULL,
  `notas_atencion` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `atendido` (`atendido`),
  KEY `usuario_atencion_id` (`usuario_atencion_id`),
  CONSTRAINT `fk_contactos_web_usuario` FOREIGN KEY (`usuario_atencion_id`) REFERENCES `usuarios` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- FIN DEL ESQUEMA
-- ============================================
