<?php
/**
 * LexiPro - Funciones Auxiliares
 */

/**
 * Escapar HTML
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Formatear fecha
 */
function formatDate($date, $format = 'd/m/Y') {
    if (!$date) return '';
    $timestamp = is_numeric($date) ? $date : strtotime($date);
    return date($format, $timestamp);
}

/**
 * Formatear fecha y hora
 */
function formatDateTime($datetime, $format = 'd/m/Y H:i') {
    if (!$datetime) return '';
    $timestamp = is_numeric($datetime) ? $datetime : strtotime($datetime);
    return date($format, $timestamp);
}

/**
 * Formatear moneda (Guaraníes)
 */
function formatMoney($amount, $currency = 'Gs.') {
    return $currency . ' ' . number_format($amount, 0, ',', '.');
}

/**
 * Formatear número
 */
function formatNumber($number, $decimals = 2) {
    return number_format($number, $decimals, ',', '.');
}

/**
 * Generar URL
 */
function url($path = '') {
    $base = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
    return $base . '/' . ltrim($path, '/');
}

/**
 * Redireccionar
 */
function redirect($url) {
    header("Location: $url");
    exit;
}

/**
 * Obtener configuración
 */
function getConfig($key = null) {
    static $config = null;
    
    if ($config === null) {
        $db = Database::getInstance();
        $configData = $db->fetchOne("SELECT * FROM configuracion LIMIT 1");
        $config = $configData ?: [];
    }
    
    if ($key === null) {
        return $config;
    }
    
    return $config[$key] ?? null;
}

/**
 * Subir archivo
 */
function uploadFile($file, $directory = 'uploads') {
    if (!isset($file['error']) || is_array($file['error'])) {
        throw new Exception('Parámetros inválidos');
    }
    
    switch ($file['error']) {
        case UPLOAD_ERR_OK:
            break;
        case UPLOAD_ERR_NO_FILE:
            throw new Exception('No se seleccionó ningún archivo');
        case UPLOAD_ERR_INI_SIZE:
        case UPLOAD_ERR_FORM_SIZE:
            throw new Exception('El archivo excede el tamaño máximo permitido');
        default:
            throw new Exception('Error desconocido al subir el archivo');
    }
    
    // Validar tamaño (50MB máximo)
    if ($file['size'] > 50 * 1024 * 1024) {
        throw new Exception('El archivo excede el tamaño máximo de 50MB');
    }
    
    // Generar nombre único
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid() . '_' . time() . '.' . $extension;
    
    // Crear directorio si no existe
    $uploadPath = __DIR__ . '/../' . $directory;
    if (!is_dir($uploadPath)) {
        mkdir($uploadPath, 0755, true);
    }
    
    $filepath = $uploadPath . '/' . $filename;
    
    if (!move_uploaded_file($file['tmp_name'], $filepath)) {
        throw new Exception('Error al mover el archivo');
    }
    
    return [
        'nombre_original' => $file['name'],
        'nombre_archivo' => $filename,
        'ruta' => $directory . '/' . $filename,
        'tipo_mime' => $file['type'],
        'tamano' => $file['size']
    ];
}

/**
 * Eliminar archivo
 */
function deleteFile($filepath) {
    $fullPath = __DIR__ . '/../' . $filepath;
    if (file_exists($fullPath)) {
        return unlink($fullPath);
    }
    return false;
}

/**
 * Generar número de caso
 */
function generateCaseNumber() {
    $year = date('Y');
    $db = Database::getInstance();
    
    $lastCase = $db->fetchOne(
        "SELECT numero_caso FROM casos WHERE numero_caso LIKE ? ORDER BY id DESC LIMIT 1",
        [$year . '%']
    );
    
    if ($lastCase) {
        $lastNumber = (int) substr($lastCase['numero_caso'], -4);
        $newNumber = $lastNumber + 1;
    } else {
        $newNumber = 1;
    }
    
    return $year . '-' . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
}

/**
 * Generar número de factura
 */
function generateInvoiceNumber($tipo = 'venta') {
    $config = getConfig();
    $establecimiento = $config['establecimiento'] ?? '001';
    $puntoExpedicion = $config['punto_expedicion'] ?? '001';
    
    $db = Database::getInstance();
    $table = $tipo === 'venta' ? 'facturas_ventas' : 'facturas_compras';
    
    $lastInvoice = $db->fetchOne(
        "SELECT numero_factura FROM {$table} ORDER BY id DESC LIMIT 1"
    );
    
    if ($lastInvoice) {
        $parts = explode('-', $lastInvoice['numero_factura']);
        $lastNumber = (int) end($parts);
        $newNumber = $lastNumber + 1;
    } else {
        $newNumber = 1;
    }
    
    return $establecimiento . '-' . $puntoExpedicion . '-' . str_pad($newNumber, 7, '0', STR_PAD_LEFT);
}

/**
 * Calcular IVA
 */
function calculateIVA($subtotal, $tipo = '10') {
    if ($tipo === 'exento') {
        return 0;
    }
    
    $rate = $tipo === '5' ? 0.05 : 0.10;
    return round($subtotal * $rate / (1 + $rate), 2);
}

/**
 * Validar RUC Paraguay
 */
function validateRUC($ruc) {
    // Formato: 12345678-9
    return preg_match('/^\d{6,8}-\d{1}$/', $ruc);
}

/**
 * Validar Cédula Paraguay
 */
function validateCedula($cedula) {
    // Formato: 1.234.567 o 1234567
    $cedula = str_replace('.', '', $cedula);
    return preg_match('/^\d{6,8}$/', $cedula);
}

/**
 * Crear notificación
 */
function createNotification($userId, $tipo, $titulo, $mensaje, $url = null) {
    $db = Database::getInstance();
    
    return $db->insert('notificaciones', [
        'usuario_id' => $userId,
        'tipo' => $tipo,
        'titulo' => $titulo,
        'mensaje' => $mensaje,
        'url' => $url,
        'leido' => 0
    ]);
}

/**
 * Obtener notificaciones no leídas
 */
function getUnreadNotifications($userId) {
    $db = Database::getInstance();
    
    return $db->fetchAll(
        "SELECT * FROM notificaciones WHERE usuario_id = ? AND leido = 0 ORDER BY created_at DESC LIMIT 10",
        [$userId]
    );
}

/**
 * Contar notificaciones no leídas
 */
function countUnreadNotifications($userId) {
    $db = Database::getInstance();
    
    $result = $db->fetchOne(
        "SELECT COUNT(*) as total FROM notificaciones WHERE usuario_id = ? AND leido = 0",
        [$userId]
    );
    
    return $result['total'] ?? 0;
}

/**
 * Registrar actividad
 */
function logActivity($userId, $accion, $modulo, $moduloId = null, $detalles = null) {
    $db = Database::getInstance();
    
    // Crear tabla de logs si no existe
    $db->query("CREATE TABLE IF NOT EXISTS logs_actividad (
        id INT AUTO_INCREMENT PRIMARY KEY,
        usuario_id INT NOT NULL,
        accion VARCHAR(100) NOT NULL,
        modulo VARCHAR(50) NOT NULL,
        modulo_id INT,
        detalles TEXT,
        ip VARCHAR(50),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (usuario_id) REFERENCES usuarios(id)
    )");
    
    return $db->insert('logs_actividad', [
        'usuario_id' => $userId,
        'accion' => $accion,
        'modulo' => $modulo,
        'modulo_id' => $moduloId,
        'detalles' => $detalles,
        'ip' => $_SERVER['REMOTE_ADDR'] ?? null
    ]);
}

/**
 * Sanitizar input
 */
function sanitize($input) {
    if (is_array($input)) {
        return array_map('sanitize', $input);
    }
    return trim(strip_tags($input));
}

/**
 * Generar token CSRF
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verificar token CSRF
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Respuesta JSON
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Error JSON
 */
function jsonError($message, $statusCode = 400) {
    jsonResponse(['error' => $message], $statusCode);
}

/**
 * Success JSON
 */
function jsonSuccess($data = [], $message = 'Operación exitosa') {
    jsonResponse(array_merge(['success' => true, 'message' => $message], $data));
}
