<?php
/**
 * LexiPro - Funciones Auxiliares de Contabilidad
 */

/**
 * Generar asiento contable desde factura de venta
 */
function generarAsientoVenta($facturaId) {
    $db = Database::getInstance();
    
    // Obtener factura
    $factura = $db->fetchOne("SELECT * FROM facturas_ventas WHERE id = ?", [$facturaId]);
    if (!$factura) return false;
    
    // Obtener configuración de cuentas
    $cuentaCaja = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.1'"); // Caja
    $cuentaClientes = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.3'"); // Clientes
    $cuentaIngresos = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '4.1'"); // Ingresos
    $cuentaIVA = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '2.1.2'"); // IVA a Pagar
    
    if (!$cuentaCaja || !$cuentaClientes || !$cuentaIngresos || !$cuentaIVA) {
        return false; // Cuentas no configuradas
    }
    
    $db->beginTransaction();
    
    try {
        // Crear asiento
        $asientoId = $db->insert('asientos_contables', [
            'fecha' => $factura['fecha'],
            'descripcion' => "Factura de venta {$factura['numero_factura']}",
            'tipo' => 'automatico',
            'referencia_tipo' => 'factura_venta',
            'referencia_id' => $facturaId,
            'usuario_id' => $factura['usuario_id']
        ]);
        
        // Determinar cuenta de débito según condición
        $cuentaDebito = $factura['condicion_venta'] === 'contado' ? $cuentaCaja['id'] : $cuentaClientes['id'];
        
        // Línea DEBE - Caja o Clientes
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaDebito,
            'debe' => $factura['total_general'],
            'haber' => 0,
            'descripcion' => $factura['condicion_venta'] === 'contado' ? 'Cobro al contado' : 'Venta a crédito'
        ]);
        
        // Línea HABER - Ingresos
        $subtotalTotal = $factura['subtotal_5'] + $factura['subtotal_10'] + $factura['subtotal_exento'];
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaIngresos['id'],
            'debe' => 0,
            'haber' => $subtotalTotal,
            'descripcion' => 'Ingresos por servicios'
        ]);
        
        // Línea HABER - IVA a Pagar
        if ($factura['total_iva'] > 0) {
            $db->insert('asientos_detalle', [
                'asiento_id' => $asientoId,
                'cuenta_id' => $cuentaIVA['id'],
                'debe' => 0,
                'haber' => $factura['total_iva'],
                'descripcion' => 'IVA generado'
            ]);
        }
        
        $db->commit();
        return $asientoId;
        
    } catch (Exception $e) {
        $db->rollback();
        error_log("Error generando asiento de venta: " . $e->getMessage());
        return false;
    }
}

/**
 * Generar asiento contable desde factura de compra
 */
function generarAsientoCompra($facturaId) {
    $db = Database::getInstance();
    
    // Obtener factura
    $factura = $db->fetchOne("SELECT * FROM facturas_compras WHERE id = ?", [$facturaId]);
    if (!$factura) return false;
    
    // Obtener configuración de cuentas
    $cuentaCaja = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.1'"); // Caja
    $cuentaProveedores = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '2.1.1'"); // Proveedores
    $cuentaGastos = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '5.1'"); // Gastos
    $cuentaIVACredito = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.4'"); // IVA Crédito
    
    if (!$cuentaCaja || !$cuentaProveedores || !$cuentaGastos) {
        return false;
    }
    
    $db->beginTransaction();
    
    try {
        // Crear asiento
        $asientoId = $db->insert('asientos_contables', [
            'fecha' => $factura['fecha'],
            'descripcion' => "Factura de compra {$factura['numero_factura']}",
            'tipo' => 'automatico',
            'referencia_tipo' => 'factura_compra',
            'referencia_id' => $facturaId,
            'usuario_id' => $factura['usuario_id']
        ]);
        
        // Línea DEBE - Gastos
        $subtotalTotal = $factura['subtotal_5'] + $factura['subtotal_10'] + $factura['subtotal_exento'];
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaGastos['id'],
            'debe' => $subtotalTotal,
            'haber' => 0,
            'descripcion' => 'Gastos operativos'
        ]);
        
        // Línea DEBE - IVA Crédito Fiscal
        if ($factura['total_iva'] > 0 && $cuentaIVACredito) {
            $db->insert('asientos_detalle', [
                'asiento_id' => $asientoId,
                'cuenta_id' => $cuentaIVACredito['id'],
                'debe' => $factura['total_iva'],
                'haber' => 0,
                'descripcion' => 'IVA crédito fiscal'
            ]);
        }
        
        // Línea HABER - Caja o Proveedores
        $cuentaCredito = $factura['condicion_compra'] === 'contado' ? $cuentaCaja['id'] : $cuentaProveedores['id'];
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaCredito,
            'debe' => 0,
            'haber' => $factura['total_general'],
            'descripcion' => $factura['condicion_compra'] === 'contado' ? 'Pago al contado' : 'Compra a crédito'
        ]);
        
        $db->commit();
        return $asientoId;
        
    } catch (Exception $e) {
        $db->rollback();
        error_log("Error generando asiento de compra: " . $e->getMessage());
        return false;
    }
}

/**
 * Generar asiento de cobro
 */
function generarAsientoCobro($cobroId) {
    $db = Database::getInstance();
    
    $cobro = $db->fetchOne("SELECT * FROM cobros WHERE id = ?", [$cobroId]);
    if (!$cobro) return false;
    
    $cuentaCaja = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.1'");
    $cuentaClientes = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.3'");
    
    if (!$cuentaCaja || !$cuentaClientes) return false;
    
    $db->beginTransaction();
    
    try {
        $asientoId = $db->insert('asientos_contables', [
            'fecha' => $cobro['fecha'],
            'descripcion' => "Cobro de factura",
            'tipo' => 'automatico',
            'referencia_tipo' => 'cobro',
            'referencia_id' => $cobroId,
            'usuario_id' => $cobro['usuario_id']
        ]);
        
        // DEBE Caja
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaCaja['id'],
            'debe' => $cobro['monto'],
            'haber' => 0,
            'descripcion' => 'Cobro recibido'
        ]);
        
        // HABER Clientes
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaClientes['id'],
            'debe' => 0,
            'haber' => $cobro['monto'],
            'descripcion' => 'Cancelación de cuenta por cobrar'
        ]);
        
        $db->commit();
        return $asientoId;
        
    } catch (Exception $e) {
        $db->rollback();
        return false;
    }
}

/**
 * Generar asiento de pago
 */
function generarAsientoPago($pagoId) {
    $db = Database::getInstance();
    
    $pago = $db->fetchOne("SELECT * FROM pagos WHERE id = ?", [$pagoId]);
    if (!$pago) return false;
    
    $cuentaCaja = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '1.1.1'");
    $cuentaProveedores = $db->fetchOne("SELECT id FROM plan_cuentas WHERE codigo = '2.1.1'");
    
    if (!$cuentaCaja || !$cuentaProveedores) return false;
    
    $db->beginTransaction();
    
    try {
        $asientoId = $db->insert('asientos_contables', [
            'fecha' => $pago['fecha'],
            'descripcion' => "Pago a proveedor",
            'tipo' => 'automatico',
            'referencia_tipo' => 'pago',
            'referencia_id' => $pagoId,
            'usuario_id' => $pago['usuario_id']
        ]);
        
        // DEBE Proveedores
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaProveedores['id'],
            'debe' => $pago['monto'],
            'haber' => 0,
            'descripcion' => 'Cancelación de cuenta por pagar'
        ]);
        
        // HABER Caja
        $db->insert('asientos_detalle', [
            'asiento_id' => $asientoId,
            'cuenta_id' => $cuentaCaja['id'],
            'debe' => 0,
            'haber' => $pago['monto'],
            'descripcion' => 'Pago realizado'
        ]);
        
        $db->commit();
        return $asientoId;
        
    } catch (Exception $e) {
        $db->rollback();
        return false;
    }
}

/**
 * Validar que un asiento cuadre (debe = haber)
 */
function validarAsiento($asientoId) {
    $db = Database::getInstance();
    
    $totales = $db->fetchOne(
        "SELECT SUM(debe) as total_debe, SUM(haber) as total_haber 
         FROM asientos_detalle 
         WHERE asiento_id = ?",
        [$asientoId]
    );
    
    return abs($totales['total_debe'] - $totales['total_haber']) < 0.01; // Tolerancia de 1 centavo
}

/**
 * Obtener saldo de una cuenta a una fecha
 */
function getSaldoCuenta($cuentaId, $fecha = null) {
    $db = Database::getInstance();
    
    $sql = "SELECT 
                COALESCE(SUM(debe), 0) as total_debe,
                COALESCE(SUM(haber), 0) as total_haber
            FROM asientos_detalle ad
            INNER JOIN asientos_contables ac ON ad.asiento_id = ac.id
            WHERE ad.cuenta_id = ?";
    
    $params = [$cuentaId];
    
    if ($fecha) {
        $sql .= " AND ac.fecha <= ?";
        $params[] = $fecha;
    }
    
    $totales = $db->fetchOne($sql, $params);
    
    // Obtener tipo de cuenta para determinar si es deudora o acreedora
    $cuenta = $db->fetchOne("SELECT tipo FROM plan_cuentas WHERE id = ?", [$cuentaId]);
    
    $saldo = $totales['total_debe'] - $totales['total_haber'];
    
    // Para cuentas de pasivo, patrimonio e ingresos, el saldo es al revés
    if (in_array($cuenta['tipo'], ['pasivo', 'patrimonio', 'ingreso'])) {
        $saldo = -$saldo;
    }
    
    return $saldo;
}

/**
 * Obtener balance general a una fecha
 */
function getBalance($fecha = null) {
    $db = Database::getInstance();
    
    if (!$fecha) $fecha = date('Y-m-d');
    
    $balance = [
        'activo' => [],
        'pasivo' => [],
        'patrimonio' => [],
        'total_activo' => 0,
        'total_pasivo' => 0,
        'total_patrimonio' => 0
    ];
    
    // Obtener cuentas de activo
    $cuentasActivo = $db->fetchAll("SELECT * FROM plan_cuentas WHERE tipo = 'activo' AND es_movimiento = 1 ORDER BY codigo");
    foreach ($cuentasActivo as $cuenta) {
        $saldo = getSaldoCuenta($cuenta['id'], $fecha);
        if (abs($saldo) > 0.01) {
            $balance['activo'][] = [
                'codigo' => $cuenta['codigo'],
                'nombre' => $cuenta['nombre'],
                'saldo' => $saldo
            ];
            $balance['total_activo'] += $saldo;
        }
    }
    
    // Obtener cuentas de pasivo
    $cuentasPasivo = $db->fetchAll("SELECT * FROM plan_cuentas WHERE tipo = 'pasivo' AND es_movimiento = 1 ORDER BY codigo");
    foreach ($cuentasPasivo as $cuenta) {
        $saldo = getSaldoCuenta($cuenta['id'], $fecha);
        if (abs($saldo) > 0.01) {
            $balance['pasivo'][] = [
                'codigo' => $cuenta['codigo'],
                'nombre' => $cuenta['nombre'],
                'saldo' => $saldo
            ];
            $balance['total_pasivo'] += $saldo;
        }
    }
    
    // Obtener cuentas de patrimonio
    $cuentasPatrimonio = $db->fetchAll("SELECT * FROM plan_cuentas WHERE tipo = 'patrimonio' AND es_movimiento = 1 ORDER BY codigo");
    foreach ($cuentasPatrimonio as $cuenta) {
        $saldo = getSaldoCuenta($cuenta['id'], $fecha);
        if (abs($saldo) > 0.01) {
            $balance['patrimonio'][] = [
                'codigo' => $cuenta['codigo'],
                'nombre' => $cuenta['nombre'],
                'saldo' => $saldo
            ];
            $balance['total_patrimonio'] += $saldo;
        }
    }
    
    return $balance;
}

/**
 * Obtener estado de resultados para un período
 */
function getEstadoResultados($fechaDesde, $fechaHasta) {
    $db = Database::getInstance();
    
    $resultado = [
        'ingresos' => [],
        'egresos' => [],
        'total_ingresos' => 0,
        'total_egresos' => 0,
        'utilidad' => 0
    ];
    
    // Obtener cuentas de ingresos
    $cuentasIngresos = $db->fetchAll("SELECT * FROM plan_cuentas WHERE tipo = 'ingreso' AND es_movimiento = 1 ORDER BY codigo");
    foreach ($cuentasIngresos as $cuenta) {
        $sql = "SELECT COALESCE(SUM(haber - debe), 0) as total
                FROM asientos_detalle ad
                INNER JOIN asientos_contables ac ON ad.asiento_id = ac.id
                WHERE ad.cuenta_id = ? AND ac.fecha BETWEEN ? AND ?";
        
        $total = $db->fetchOne($sql, [$cuenta['id'], $fechaDesde, $fechaHasta]);
        
        if (abs($total['total']) > 0.01) {
            $resultado['ingresos'][] = [
                'codigo' => $cuenta['codigo'],
                'nombre' => $cuenta['nombre'],
                'monto' => $total['total']
            ];
            $resultado['total_ingresos'] += $total['total'];
        }
    }
    
    // Obtener cuentas de egresos
    $cuentasEgresos = $db->fetchAll("SELECT * FROM plan_cuentas WHERE tipo = 'egreso' AND es_movimiento = 1 ORDER BY codigo");
    foreach ($cuentasEgresos as $cuenta) {
        $sql = "SELECT COALESCE(SUM(debe - haber), 0) as total
                FROM asientos_detalle ad
                INNER JOIN asientos_contables ac ON ad.asiento_id = ac.id
                WHERE ad.cuenta_id = ? AND ac.fecha BETWEEN ? AND ?";
        
        $total = $db->fetchOne($sql, [$cuenta['id'], $fechaDesde, $fechaHasta]);
        
        if (abs($total['total']) > 0.01) {
            $resultado['egresos'][] = [
                'codigo' => $cuenta['codigo'],
                'nombre' => $cuenta['nombre'],
                'monto' => $total['total']
            ];
            $resultado['total_egresos'] += $total['total'];
        }
    }
    
    $resultado['utilidad'] = $resultado['total_ingresos'] - $resultado['total_egresos'];
    
    return $resultado;
}
